# Deferred Callbacks

Allows modules to register callbacks that execute after the HTTP response is sent to the client. This enables non-blocking execution of operations that should not delay the response delivery.

## Purpose

This module provides a mechanism to defer execution of callbacks until after the HTTP response has been sent to the client. This is particularly useful for operations that are important but should not block the user's request, such as sending emails, logging, cache warming, or analytics tracking.

## Usage

Register a deferred callback using the service:

```php
\Drupal::service('deferred_callbacks.collection')
  ->push(function() {
    // Your code here
  });
```

You can specify a priority (higher values execute first) and an optional name for debugging:

```php
\Drupal::service('deferred_callbacks.collection')
  ->push(function() {
    // High priority callback
  }, 10, 'my_callback_name');
```

### Dependency Injection

```php
use Drupal\deferred_callbacks\DeferredCallbackCollectionInterface;

class MyService {
  public function __construct(
    private readonly DeferredCallbackCollectionInterface $deferredCallbacks,
  ) {}

  public function doSomething() {
    $this->deferredCallbacks->push(function() {
      // Deferred operation
    });
  }
}
```

## Use Cases

- **Email sending**: Send transactional emails without blocking the response
- **Logging**: Log user actions or system events asynchronously
- **Analytics**: Track page views or user behavior without impacting response time
- **Cache warming**: Pre-populate caches for related content
- **External API calls**: Make non-critical API requests after response delivery
- **Statistics collection**: Gather metrics without affecting user experience

## Important Limitations

### Server Requirements

**This module only provides true asynchronous execution with PHP-FPM or LiteSpeed servers.**

On Apache with mod_php, callbacks will execute synchronously before the response is sent, which defeats the purpose of deferring execution.

### Technical Details

The asynchronous behavior relies on PHP's `fastcgi_finish_request()` or `litespeed_finish_request()` functions, which are called by Symfony's `Response::send()` method. These functions allow the PHP process to finish sending the response to the client while continuing to execute PHP code. Without these functions (as in Apache/mod_php), the response cannot be sent until all PHP execution completes.

### When NOT to Use This Module

**Do not use deferred callbacks for:**
- Long-running operations (use the Queue API instead)
- Critical operations that must complete before the response
- Operations that require user feedback or confirmation
- Heavy computations or batch processing

For long-running or resource-intensive tasks, use Drupal's Queue API, which is designed for background job processing and can be processed by separate workers.

## How It Works

1. During request processing, callbacks are registered via the `DeferredCallbackCollection` service
2. Callbacks are stored with their priority and optional name
3. On the `kernel.terminate` event (after response is sent), all registered callbacks are executed
4. Callbacks execute in priority order (higher priority first)
5. Exceptions in callbacks are caught and logged, preventing one failure from affecting others

## Requirements

- Drupal 10 or 11
- PHP-FPM or LiteSpeed for true asynchronous execution (Apache/mod_php will execute synchronously)
