<?php

declare(strict_types=1);

namespace Drupal\deferred_callbacks;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Deferred callback collection.
 */
final class DeferredCallbackCollection implements DeferredCallbackCollectionInterface {

  use LoggerChannelTrait;

  /**
   * The deferred callbacks collection.
   *
   * @var array<int, array<int, array{callback: callable, name: string|null}>>
   *   [priority => [ [callback, name], ... ]].
   */
  private array $collection = [];

  public function __construct(
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->setLoggerFactory($loggerFactory);
  }

  /**
   * {@inheritdoc}
   */
  public function push(callable $callback, int $priority = 0, ?string $name = NULL): void {
    $this->collection[$priority][] = [
      'callback' => $callback,
      'name' => $name,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    if ($this->collection === []) {
      return;
    }

    // Higher priority callbacks run first.
    krsort($this->collection);

    foreach ($this->collection as $priority => $items) {
      foreach ($items as $item) {
        $this->tryExecute($item['callback'], $item['name']);
      }
    }

    $this->collection = [];
  }

  /**
   * Executes a callback, catching any exceptions.
   *
   * @param callable $callback
   *   The callback to execute.
   * @param string|null $name
   *   Optional name for logging.
   */
  private function tryExecute(callable $callback, ?string $name = NULL): void {
    try {
      $callback();
    }
    catch (\Throwable $e) {
      $context = [
        '@message' => $e->getMessage(),
        '@exception' => get_class($e),
      ];

      $logger = $this->getLogger('deferred_callbacks');

      if ($name !== NULL) {
        $context['@name'] = $name;
        $logger->error('Failed to execute deferred callback "@name": @message', $context);
      }
      else {
        $logger->error('Failed to execute deferred callback: @message', $context);
      }
    }
  }

}
