<?php

declare(strict_types=1);

namespace Drupal\Tests\deferred_callbacks\Kernel;

use Drupal\deferred_callbacks\DeferredCallbackCollectionInterface;
use Drupal\deferred_callbacks\EventSubscriber\TerminateSubscriber;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Integration tests for deferred callbacks module.
 */
#[Group('deferred_callbacks')]
final class DeferredCallbacksIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'deferred_callbacks',
    'system',
  ];

  /**
   * Tests that the service is available.
   */
  public function testServiceAvailable(): void {
    $collection = $this->container->get('deferred_callbacks.collection');
    $this->assertInstanceOf(DeferredCallbackCollectionInterface::class, $collection);
  }

  /**
   * Tests that callbacks are executed on terminate event.
   */
  public function testCallbacksExecuteOnTerminate(): void {
    $collection = $this->container->get('deferred_callbacks.collection');
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'callback1';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'callback2';
    };

    $collection->push($callback1);
    $collection->push($callback2);

    // Simulate terminate event.
    $kernel = $this->container->get('http_kernel');
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $event_dispatcher = $this->container->get('event_dispatcher');
    $event_dispatcher->dispatch($event, KernelEvents::TERMINATE);

    $this->assertCount(2, $executed);
    $this->assertContains('callback1', $executed);
    $this->assertContains('callback2', $executed);
  }

  /**
   * Tests that callbacks execute in priority order on terminate.
   */
  public function testCallbacksExecuteInPriorityOrderOnTerminate(): void {
    $collection = $this->container->get('deferred_callbacks.collection');
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'low';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'high';
    };
    $callback3 = function () use (&$executed): void {
      $executed[] = 'medium';
    };

    $collection->push($callback1, 0);
    $collection->push($callback2, 10);
    $collection->push($callback3, 5);

    $kernel = $this->container->get('http_kernel');
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $event_dispatcher = $this->container->get('event_dispatcher');
    $event_dispatcher->dispatch($event, KernelEvents::TERMINATE);

    $this->assertSame(['high', 'medium', 'low'], $executed);
  }

  /**
   * Tests that exceptions in callbacks don't break the terminate event.
   */
  public function testExceptionHandlingOnTerminate(): void {
    $collection = $this->container->get('deferred_callbacks.collection');
    $executed = [];

    $failingCallback = function (): void {
      throw new \RuntimeException('Test exception');
    };
    $successCallback = function () use (&$executed): void {
      $executed[] = 'success';
    };

    $collection->push($failingCallback);
    $collection->push($successCallback);

    $kernel = $this->container->get('http_kernel');
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $event_dispatcher = $this->container->get('event_dispatcher');
    // Should not throw exception.
    $event_dispatcher->dispatch($event, KernelEvents::TERMINATE);

    // Success callback should still execute.
    $this->assertContains('success', $executed);
  }

  /**
   * Tests that the event subscriber is registered.
   */
  public function testEventSubscriberRegistered(): void {
    /** @var \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher */
    $event_dispatcher = $this->container->get('event_dispatcher');
    $listeners = $event_dispatcher->getListeners(KernelEvents::TERMINATE);

    $this->assertNotEmpty($listeners, 'Terminate event should have listeners');

    // Check that our subscriber is registered.
    $subscriber_found = FALSE;
    foreach ($listeners as $listener) {
      if (is_array($listener) && isset($listener[0])) {
        $subscriber = $listener[0];
        if ($subscriber instanceof TerminateSubscriber) {
          $subscriber_found = TRUE;
          break;
        }
      }
    }

    $this->assertTrue($subscriber_found, 'TerminateSubscriber should be registered');
  }

  /**
   * Tests that callbacks are cleared after execution.
   */
  public function testCallbacksClearedAfterExecution(): void {
    $collection = $this->container->get('deferred_callbacks.collection');
    $executed = 0;

    $callback = function () use (&$executed): void {
      $executed++;
    };

    $collection->push($callback);

    $kernel = $this->container->get('http_kernel');
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $event_dispatcher = $this->container->get('event_dispatcher');
    $event_dispatcher->dispatch($event, KernelEvents::TERMINATE);

    $this->assertSame(1, $executed);

    // Dispatch terminate event again - callback should not execute.
    $event2 = new TerminateEvent($kernel, $request, $response);
    $event_dispatcher->dispatch($event2, KernelEvents::TERMINATE);

    $this->assertSame(1, $executed);
  }

}
