<?php

declare(strict_types=1);

namespace Drupal\Tests\deferred_callbacks\Unit;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\deferred_callbacks\DeferredCallbackCollection;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the DeferredCallbackCollection class.
 */
#[CoversClass(DeferredCallbackCollection::class)]
#[Group('deferred_callbacks')]
final class DeferredCallbackCollectionTest extends UnitTestCase {

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  private LoggerChannelFactoryInterface|MockObject $loggerFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  private LoggerChannelInterface|MockObject $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('deferred_callbacks')
      ->willReturn($this->logger);
  }

  /**
   * Tests that push adds callbacks to the collection.
   */
  public function testPushAddsCallbacks(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'callback1';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'callback2';
    };

    $collection->push($callback1);
    $collection->push($callback2);

    $collection->execute();

    $this->assertCount(2, $executed);
    $this->assertContains('callback1', $executed);
    $this->assertContains('callback2', $executed);
  }

  /**
   * Tests that execute runs callbacks in priority order.
   */
  public function testExecuteRunsCallbacksInPriorityOrder(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'low';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'high';
    };
    $callback3 = function () use (&$executed): void {
      $executed[] = 'medium';
    };

    // Push with different priorities: higher priority runs first.
    $collection->push($callback1, 0);
    $collection->push($callback2, 10);
    $collection->push($callback3, 5);

    $collection->execute();

    $this->assertSame(['high', 'medium', 'low'], $executed);
  }

  /**
   * Tests that execute clears the collection after running.
   */
  public function testExecuteClearsCollection(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = 0;

    $callback = function () use (&$executed): void {
      $executed++;
    };

    $collection->push($callback);
    $collection->execute();

    $this->assertSame(1, $executed);

    // Execute again - should not run callbacks since collection is empty.
    $collection->execute();

    $this->assertSame(1, $executed);
  }

  /**
   * Tests that callbacks with same priority run in order they were added.
   */
  public function testCallbacksWithSamePriorityRunInOrder(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'first';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'second';
    };
    $callback3 = function () use (&$executed): void {
      $executed[] = 'third';
    };

    $collection->push($callback1, 5);
    $collection->push($callback2, 5);
    $collection->push($callback3, 5);

    $collection->execute();

    $this->assertSame(['first', 'second', 'third'], $executed);
  }

  /**
   * Tests that exceptions in callbacks are caught and logged.
   */
  public function testExceptionHandling(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $exception = new \RuntimeException('Test exception');
    $failingCallback = function () use ($exception): void {
      throw $exception;
    };
    $successCallback = function () use (&$executed): void {
      $executed[] = 'success';
    };

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Failed to execute deferred callback: @message',
        $this->callback(function (array $context) {
          return isset($context['@message']) && isset($context['@exception']);
        })
      );

    $collection->push($failingCallback);
    $collection->push($successCallback);

    // Should not throw exception, but should log it.
    $collection->execute();

    // Success callback should still execute.
    $this->assertContains('success', $executed);
  }

  /**
   * Tests that exceptions in named callbacks are logged with name.
   */
  public function testExceptionHandlingWithName(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);

    $exception = new \RuntimeException('Test exception');
    $failingCallback = function () use ($exception): void {
      throw $exception;
    };

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Failed to execute deferred callback "@name": @message',
        $this->callback(function (array $context) {
          return isset($context['@name']) &&
            $context['@name'] === 'test_callback' &&
            isset($context['@message']) &&
            isset($context['@exception']);
        })
      );

    $collection->push($failingCallback, 0, 'test_callback');
    $collection->execute();
  }

  /**
   * Tests that multiple exceptions are all caught and logged.
   */
  public function testMultipleExceptionsHandling(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $exception1 = new \RuntimeException('Exception 1');
    $exception2 = new \InvalidArgumentException('Exception 2');

    $failingCallback1 = function () use ($exception1): void {
      throw $exception1;
    };
    $failingCallback2 = function () use ($exception2): void {
      throw $exception2;
    };
    $successCallback = function () use (&$executed): void {
      $executed[] = 'success';
    };

    $this->logger->expects($this->exactly(2))
      ->method('error');

    $collection->push($failingCallback1);
    $collection->push($failingCallback2);
    $collection->push($successCallback);

    $collection->execute();

    // Success callback should still execute.
    $this->assertContains('success', $executed);
  }

  /**
   * Tests that callbacks can receive and modify external variables.
   */
  public function testCallbacksCanModifyVariables(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $value = 0;

    $callback = function () use (&$value): void {
      $value = 42;
    };

    $collection->push($callback);
    $collection->execute();

    $this->assertSame(42, $value);
  }

  /**
   * Tests that callbacks with negative priority work correctly.
   */
  public function testNegativePriority(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'negative';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'zero';
    };
    $callback3 = function () use (&$executed): void {
      $executed[] = 'positive';
    };

    $collection->push($callback1, -10);
    $collection->push($callback2, 0);
    $collection->push($callback3, 10);

    $collection->execute();

    $this->assertSame(['positive', 'zero', 'negative'], $executed);
  }

  /**
   * Tests that callbacks can be closures with parameters.
   */
  public function testClosureCallbacks(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $result = '';

    $callback = function (string $text) use (&$result): void {
      $result = $text;
    };

    // Note: callbacks are called without arguments, so we need to bind them.
    $boundCallback = function () use ($callback, &$result): void {
      $callback('Hello World');
    };

    $collection->push($boundCallback);
    $collection->execute();

    $this->assertSame('Hello World', $result);
  }

  /**
   * Tests that named callbacks work correctly.
   */
  public function testNamedCallbacks(): void {
    $collection = new DeferredCallbackCollection($this->loggerFactory);
    $executed = [];

    $callback1 = function () use (&$executed): void {
      $executed[] = 'callback1';
    };
    $callback2 = function () use (&$executed): void {
      $executed[] = 'callback2';
    };

    $collection->push($callback1, 0, 'first_callback');
    $collection->push($callback2, 0, 'second_callback');

    $collection->execute();

    $this->assertCount(2, $executed);
  }

}
