<?php

declare(strict_types=1);

namespace Drupal\Tests\deferred_callbacks\Unit;

use Drupal\deferred_callbacks\DeferredCallbackCollectionInterface;
use Drupal\deferred_callbacks\EventSubscriber\TerminateSubscriber;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Tests the TerminateSubscriber class.
 */
#[CoversClass(TerminateSubscriber::class)]
#[Group('deferred_callbacks')]
final class TerminateSubscriberTest extends UnitTestCase {

  /**
   * Tests that getSubscribedEvents returns the correct events.
   */
  public function testGetSubscribedEvents(): void {
    $events = TerminateSubscriber::getSubscribedEvents();

    $this->assertArrayHasKey(KernelEvents::TERMINATE, $events);

    // The structure is ['onTerminate', 0] as defined in getSubscribedEvents().
    $terminate_events = $events[KernelEvents::TERMINATE];
    $this->assertIsArray($terminate_events);
    $this->assertSame('onTerminate', $terminate_events[0]);
    $this->assertSame(0, $terminate_events[1]);
  }

  /**
   * Tests that onTerminate executes the collection.
   */
  public function testOnTerminateExecutesCollection(): void {
    $collection = $this->createMock(DeferredCallbackCollectionInterface::class);
    $collection->expects($this->once())
      ->method('execute');

    $subscriber = new TerminateSubscriber($collection);
    $kernel = $this->createMock(HttpKernelInterface::class);
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $subscriber->onTerminate($event);
  }

  /**
   * Tests that multiple terminate events execute the collection multiple times.
   */
  public function testMultipleTerminateEvents(): void {
    $collection = $this->createMock(DeferredCallbackCollectionInterface::class);
    $collection->expects($this->exactly(2))
      ->method('execute');

    $subscriber = new TerminateSubscriber($collection);
    $kernel = $this->createMock(HttpKernelInterface::class);
    $request = Request::create('/');
    $response = new Response();
    $event = new TerminateEvent($kernel, $request, $response);

    $subscriber->onTerminate($event);
    $subscriber->onTerminate($event);
  }

}
