<?php

namespace Drupal\deindex_unpublished_files\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManager;

/**
 * The settings form class.
 */
class SettingsForm extends ConfigFormBase {
  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;
  /**
   * The stream wrapper manager service.
   *
   * @var \Drupal\Core\File\StreamWrapper\StreamWrapperManager
   */
  protected $streamWrapperManager;

  public function __construct(FileSystemInterface $fileSystem, StreamWrapperManager $streamWrapperManager) {
    $this->fileSystem = $fileSystem;
    $this->streamWrapperManager = $streamWrapperManager;
  }

  /**
   * Create function.
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('file_system'),
                $container->get('stream_wrapper_manager')
      );
  }

  /**
   * Get the form id.
   */
  public function getFormId() {
    return 'deindex_unpublished_files_settings';
  }

  /**
   * Get the config name.
   */
  protected function getEditableConfigNames() {
    return ['deindex_unpublished_files.settings'];
  }

  /**
   * The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('deindex_unpublished_files.settings');

    $form['unpublish_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Action on media unpublish'),
      '#default_value' => $config->get('unpublish_mode') ?: 'prefix',
      '#options' => [
        'prefix' => $this->t('Prefix filename with .ht_'),
        'move' => $this->t('Move file to private/unpublishedfiles'),
      ],
      '#description' => $this->t('If you choose "Move file" You have to deny the access to the folder unpublishedfiles , Every role that has access to administer media has access to private files'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit function.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('deindex_unpublished_files.settings')
      ->set('unpublish_mode', $form_state->getValue('unpublish_mode'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Validate function.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $values = $form_state->getValues();

    if (!empty($values['unpublish_mode']) && $values['unpublish_mode'] == "move") {
      try {
        $wrapper = $this->streamWrapperManager->getViaScheme('private');

        if ($wrapper) {
          $private_path = $wrapper->getDirectoryPath();
        }
        if (empty($private_path)) {
          $form_state->setErrorByName('unpublish_mode', $this->t('Private file system path is not configured. Please set it in settings.php.'));
          return;
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('unpublish_mode', $this->t('Private file system path is not configured. Please set it in settings.php.'));
        return;
      }

      if (empty($private_path)) {
        $form_state->setErrorByName('unpublish_mode', $this->t('Private file system path is not configured. Please set it in the file system settings.'));
        return;
      }

      $folder_path = 'private://unpublishedfiles';
      $real_path = $this->fileSystem->realpath($folder_path);

      if (!is_dir($real_path)) {
        $form_state->setErrorByName(
              'unpublish_mode', $this->t(
                  'The folder %folder does not exist. Please create it first.', [
                    '%folder' => $folder_path,
                  ]
              )
          );
      }
    }
  }

}
