<?php

namespace Drupal\deindex_unpublished_files\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

/**
 * Configure settings for Deindex Unpublished Files.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager service.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $streamWrapperManager
   *   The stream wrapper manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected FileSystemInterface $fileSystem,
    protected StreamWrapperManagerInterface $streamWrapperManager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('file_system'),
      $container->get('stream_wrapper_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'deindex_unpublished_files_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['deindex_unpublished_files.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['unpublish_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Action on media unpublish'),
      '#config_target' => 'deindex_unpublished_files.settings:unpublish_mode',
      '#options' => [
        'prefix' => $this->t('Prefix filename with .ht_'),
        'move' => $this->t('Move file to private/unpublishedfiles'),
      ],
      '#description' => $this->t('If you choose "Move file", ensure that the <code>unpublishedfiles</code> folder is protected and not directly accessible.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    if ($form_state->getValue('unpublish_mode') === 'move') {
      try {
        $wrapper = $this->streamWrapperManager->getViaScheme('private');
        $private_path = $wrapper ? $wrapper->getDirectoryPath() : NULL;
      }
      catch (\Exception $e) {
        $private_path = NULL;
      }

      if (empty($private_path)) {
        $form_state->setErrorByName(
          'unpublish_mode',
          $this->t('Private file system path is not configured. Please set it in settings.php.')
        );
        return;
      }

      $folder_path = 'private://unpublishedfiles';
      $real_path = $this->fileSystem->realpath($folder_path);

      if (!is_dir($real_path)) {
        $form_state->setErrorByName(
          'unpublish_mode',
          $this->t('The folder %folder does not exist. Please create it inside your private files directory.', [
            '%folder' => $folder_path,
          ])
        );
      }
    }
  }

}
