<?php

namespace Drupal\deindex_unpublished_files\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\media\MediaInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Object-oriented hook implementations for media events.
 */
final class MediaHooks {

  /**
   * MediaHooks constructor.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    protected FileSystemInterface $fileSystem,
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * Implements hook_media_presave() using PHP 8 attributes.
   */
  #[Hook('media_presave')]
  public function mediaPresave(MediaInterface $media): void {
    if ($media->isNew()) {
      return;
    }

    // Use injected EntityTypeManager, not \Drupal::entityTypeManager().
    $original = $this->entityTypeManager
      ->getStorage('media')
      ->loadUnchanged($media->id());

    if (!$original) {
      return;
    }

    $was_published = $original->isPublished();
    $is_published = $media->isPublished();

    if ($was_published === $is_published) {
      return;
    }

    $config = $this->configFactory->get('deindex_unpublished_files.settings');
    $mode = $config->get('unpublish_mode');

    foreach ($media->getFields() as $field) {
      $definition = $field->getFieldDefinition();
      if ($definition->getType() !== 'file') {
        continue;
      }

      foreach ($field->referencedEntities() as $file) {
        $original_uri = $file->getFileUri();
        $filename = basename($original_uri);
        $new_uri = NULL;

        if ($mode === 'move') {
          if (!$is_published && str_starts_with($original_uri, 'public://')) {
            $relative_path = str_replace('public://', '', dirname($original_uri));
            if (!empty($relative_path) && $relative_path !== 'public:') {
              $filename = $relative_path
                . '__'
                . $filename;
            }
            $new_uri = 'private://unpublishedfiles/' . $filename;
          }
          elseif ($is_published && str_starts_with($original_uri, 'private://unpublishedfiles/')) {
            if (strpos($filename, '__') !== FALSE) {
              [$folder, $real_filename] = explode('__', $filename, 2);
              $new_uri = 'public://' . $folder . '/' . $real_filename;
            }
            else {
              $new_uri = 'public://' . $filename;
            }
          }
        }
        elseif ($mode === 'prefix') {
          if (!$is_published && strpos($filename, '.ht_') === FALSE) {
            $dirname = dirname($original_uri);
            $new_uri = $dirname . '/.ht_' . $filename;
          }
          elseif ($is_published && strpos($filename, '.ht_') !== FALSE) {
            $new_uri = str_replace('.ht_', '', $original_uri);
          }
        }

        if (!empty($new_uri)) {
          $new_path = $this->fileSystem->move(
            $original_uri,
            $new_uri,
            FileExists::Rename,
          );
          if ($new_path) {
            $file->setFileUri($new_path);
            $file->save();
          }
        }
      }
    }
  }

}
