<?php

// phpcs:ignore
namespace Drupal\deprecation_status\Plugin\rest\resource;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Drupal\deprecation_status\DataSource;

/**
 * Provides a resource to get the deprecation status of an API item.
 */
#[RestResource(
  id: "deprecation_status",
  label: new TranslatableMarkup("Deprecation status"),
  uri_paths: [
    "canonical" => "/api/deprecation-status/{target_version}",
  ],
)]

class DeprecationStatusResource extends ResourceBase {

  /**
   * Responds to entity GET requests.
   *
   * @param string $target_version
   *   The target version to filter the files.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing the deprecation status data.
   */
  public function get($target_version) {
    // Validate target version
    $available_versions = $this->getAvailableTargetVersions();
    if (!in_array($target_version, $available_versions)) {
      $response_data = [
        'error' => 'Invalid target version',
        'message' => "Target version '{$target_version}' is not available.",
        'available_versions' => $available_versions,
      ];
      return new ResourceResponse($response_data, 400);
    }

    $response = $this->getFileData($target_version);
    return new ResourceResponse($response);
  }

  /**
   * Get available target versions from the data source.
   *
   * @return array
   *   Array of available target version strings.
   */
  protected function getAvailableTargetVersions(): array {
    return array_map('strval', DataSource::getAvailableTargetVersions());
  }

  /**
   * Get the deprecation status data from the CSV files.
   *
   * @param string $target_version
   *   The target version to filter the files.
   *
   * @return array<string>
   *   An array containing the deprecation status data.
   */
  public function getFileData($target_version): array {
    $fileData = [];
    foreach (DataSource::$datafiles as $file) {
      if (!str_starts_with($file, $target_version . '_')) {
        continue;
      }
      $base_filename = substr($file, strlen($target_version . '_'));
      $fullpath = DataSource::getFullPath($base_filename, (int) $target_version);
      $csv = [];
      $private_file = fopen($fullpath, 'r');
      if ($private_file) {
        while (!feof($private_file)) {
          $csv[] =
            fgetcsv($private_file, 0, ';');
        }

        $fileData[pathinfo($base_filename, PATHINFO_FILENAME)] = $csv;
      }
    }
    return $fileData;
  }

}
