<?php

namespace Drupal\deprecation_status\Plugin\rest\resource;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Drupal\deprecation_status\DataSource;

/**
 * Provides a resource to get available deprecation status target versions.
 */
#[RestResource(
  id: "deprecation_versions",
  label: new TranslatableMarkup("Available deprecation target versions"),
  uri_paths: [
    "canonical" => "/api/deprecation-versions",
  ],
)]

class DeprecationVersionsResource extends ResourceBase {

  /**
   * Responds to GET requests for available target versions.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing available target versions.
   */
  public function get() {
    $available_versions = array_values($this->getAvailableTargetVersions());
    $response_data = [
      'available_target_versions' => $available_versions,
      'endpoint_pattern' => '/api/deprecation-status/{target_version}',
      'endpoints' => array_combine($available_versions, array_map(function($version) {
        return "/api/deprecation-status/{$version}";
      }, $available_versions)),
      'data_files_by_version' => $this->getDataFilesByVersion(),
    ];
    
    return new ResourceResponse($response_data);
  }

  /**
   * Get available target versions from the data source.
   *
   * @return array
   *   Array of available target version strings.
   */
  protected function getAvailableTargetVersions(): array {
    return array_map('strval', DataSource::getAvailableTargetVersions());
  }

  /**
   * Get data files grouped by version.
   *
   * @return array
   *   Array of data files grouped by target version.
   */
  protected function getDataFilesByVersion(): array {
    $files_by_version = [];
    foreach (DataSource::$datafiles as $file) {
      if (preg_match('/^(\d+)_(.+)$/', $file, $matches)) {
        $version = $matches[1];
        $filename = $matches[2];
        $files_by_version[$version][] = pathinfo($filename, PATHINFO_FILENAME);
      }
    }
    return $files_by_version;
  }

}