<?php

declare(strict_types=1);

namespace Drupal\dev_entity_browser\Commands;

use Drupal\dev_entity_browser\Service\DevEntityBrowser;
use Drush\Commands\DrushCommands;

final class DevEntityBrowserCommands extends DrushCommands {

  public function __construct(
    private readonly DevEntityBrowser $devEntityBrowser,
  ) {
    parent::__construct();
  }

  /**
   * List content entity types.
   *
   * @command dev-entity-browser:list-entities
   * @aliases deb:entities
   * @option format Output format: table (default) or json.
   * @default format table
   */
  public function listEntities(array $options = ['format' => 'table']): mixed {
    $entities = $this->devEntityBrowser->listEntities();

    if (($options['format'] ?? 'table') === 'json') {
      return $entities;
    }

    $rows = [];
    foreach ($entities as $entity) {
      $rows[] = [
        'label' => $entity['label'],
        'entity_type_id' => $entity['entity_type_id'],
        'bundle_count' => $entity['bundle_count'],
        'class' => $entity['class'],
      ];
    }

    return $this->io()->table(
      ['Label', 'Entity type', 'Bundles', 'Class'],
      array_map(static fn (array $r) => [$r['label'], $r['entity_type_id'], (string) $r['bundle_count'], $r['class']], $rows)
    );
  }

  /**
   * List bundles for an entity type.
   *
   * @command dev-entity-browser:list-bundles
   * @aliases deb:bundles
   *
   * @param string $entity_type_id The entity type ID, e.g. node, media.
   * @option format Output format: table (default) or json.
   * @default format table
   */
  public function listBundles(string $entity_type_id, array $options = ['format' => 'table']): mixed {
    $bundles = $this->devEntityBrowser->listBundles($entity_type_id);

    if (($options['format'] ?? 'table') === 'json') {
      return $bundles;
    }

    $rows = [];
    foreach ($bundles as $bundle) {
      $rows[] = [$bundle['label'], $bundle['bundle_id']];
    }

    return $this->io()->table(['Label', 'Bundle'], $rows);
  }

  /**
   * Show bundle field information.
   *
   * @command dev-entity-browser:entity-info
   * @aliases deb:info
   *
   * @param string $entity_type_id The entity type ID, e.g. node, media.
   * @param string $bundle_id The bundle ID (or base bundle id for bundle-less entities).
   * @option mode Report mode: full (default) or list.
   * @default mode full
   * @option format Output format: table (default) or json.
   * @default format table
   */
  public function entityInfo(string $entity_type_id, string $bundle_id, array $options = ['mode' => 'full', 'format' => 'table']): mixed {
    $report = $this->devEntityBrowser->buildBundleReport($entity_type_id, $bundle_id);

    $mode = $options['mode'] ?? 'full';
    $format = $options['format'] ?? 'table';

    if ($format === 'json') {
      if ($mode === 'list') {
        return array_keys($report['fields']);
      }
      return $report;
    }

    $this->io()->title(sprintf('%s (%s)', $report['entity']['label'], $report['entity']['machine_name']));
    $this->io()->writeln(sprintf('Bundle: %s (%s)', $report['bundle']['label'], $report['bundle']['machine_name']));
    $this->io()->writeln(sprintf('Class: %s', $report['entity']['class_name']));

    if ($mode === 'list') {
      foreach (array_keys($report['fields']) as $field_name) {
        $this->io()->writeln($field_name);
      }
      return NULL;
    }

    $rows = [];
    foreach ($report['fields'] as $field_name => $field) {
      $rows[] = [
        $field['label_label'] ?? $field_name,
        $field_name,
        $field['info']['type'] ?? '',
        isset($field['cardinality']) ? (string) $field['cardinality'] : '',
        $field['description'] ?? '',
      ];
    }

    return $this->io()->table(['Label', 'Field', 'Type', 'Cardinality', 'Description'], $rows);
  }

}
