<?php

declare(strict_types=1);

namespace Drupal\dev_entity_browser\Service;

use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\field\Entity\FieldConfig;

/**
 * Service that builds the Developer Entity Browser report data/markup.
 */
final class DevEntityBrowser {

  public function __construct(
    private readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Build the full render array for the report.
   */
  public function buildReport(): array {
    // Collection Array
    $table_of_contents = [];
    // Content Entity Types
    $content_entity_types = [];
    // Collect Field Maps
    $field_map = $this->collectFieldMaps();
    // Collect Entity Definitions
    $entity_type_definitions = $this->collectEntityDefinitions();

    foreach ($entity_type_definitions as $key => $definition) {
      if ($definition instanceof ContentEntityType) {
        $content_entity_types[$key] = $definition;
      }
    }

    foreach ($content_entity_types as $entity_type_id => $entity_type) {
      $table_of_contents[$entity_type_id] = [
        'label' => self::convertLabelToString($entity_type->getLabel()),
        'machine_name' => $entity_type_id,
        'class_name' => $entity_type->getClass(),
        'bundles' => [],
      ];
      switch ($entity_type->id()) {
        // remove unwanted Entities
        case 'block_content':
        case 'webform_submission':
          break;
        default:
          $bundles = $this->entityTypeBundleInfo
            ->getBundleInfo($entity_type->id());
          foreach ($bundles as $bundle_id => $bundle) {
            $build_build = [
              'machine_name' => $bundle_id,
              'label' => self::convertLabelToString($bundle['label']),
              'fields' => [],
              'fieldMap' => [],
            ];

            if (isset($field_map[$entity_type->id()])) {
              foreach ($field_map[$entity_type->id()] as $field_name => $field_info) {
                foreach ($field_info['bundles'] as $field_bundle_key => $field_bundle_name) {
                  $build_build['fieldMap'][$field_bundle_key][$field_name] = [
                    'name' => $field_name,
                    'type' => $field_info['type'],
                  ];
                }
              }
            }
            // Get Bundle specific fields.
            $all_bundle_fields = $this->entityFieldManager
              ->getFieldDefinitions($entity_type_id, $bundle_id);
            foreach ($all_bundle_fields as $bundle_field_key => $bundle_field_info) {
              $field_build = [];
              $getSettings = ($bundle_field_info->getItemDefinition()
                ->getSettings());
              $field_build['description'] = self::convertLabelToString($bundle_field_info->getDescription());
              // Allowed fields.
              if (isset($getSettings['allowed_values'])) {
                $field_build['allowed_values'] = [];
                foreach ($getSettings['allowed_values'] as $allowed_value_key => $allowed_value) {
                  $field_build['allowed_values'][$allowed_value_key] = $allowed_value;
                }
              }
              $extraInfo = $this->exploreType($entity_type_id, $bundle_id, $bundle_field_info, $bundle_field_key);
              $field_build['info'] = $extraInfo;
              $field_build['label_label'] = self::convertLabelToString($bundle_field_info->getLabel());
              $field_build['label_key'] = self::convertLabelToString($bundle_field_key);
              $field_build['cardinality'] = 1;
              if (method_exists($bundle_field_info, 'getCardinality')) {
                $field_build['cardinality'] = ($bundle_field_info->getCardinality());
              }
              $build_build['fields'][$bundle_field_key] = $field_build;
              $table_of_contents[$entity_type_id] ['bundles'][$bundle_id] = $build_build;
            }
          }
          break;
      }
    }
    // Sort the table of contents by the label rather than machine id.
    $table_of_contentsOrder = [];
    foreach ($table_of_contents as $tocKey => $toc) {
      $table_of_contentsOrder[$toc['label'] . $tocKey] = $toc;
    }
    // Perform the sort.
    ksort($table_of_contentsOrder, SORT_STRING);
    // Add to render array.
    $build['dev_entity_browser_report'] = [
      '#type' => 'dev_entity_browser_report',
      '#theme' => 'dev_entity_browser_report',
      '#report' => $table_of_contentsOrder,
    ];

    return $build;
  }

  public function listEntities(): array {
    $entities = [];
    $entity_type_definitions = $this->collectEntityDefinitions();

    foreach ($entity_type_definitions as $entity_type_id => $definition) {
      if (!$definition instanceof ContentEntityType) {
        continue;
      }
      switch ($entity_type_id) {
        case 'block_content':
        case 'webform_submission':
          continue 2;
      }

      $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      if (!$bundles) {
        $bundles = [$entity_type_id => ['label' => $definition->getLabel()]];
      }

      $entities[$entity_type_id] = [
        'entity_type_id' => $entity_type_id,
        'label' => self::convertLabelToString($definition->getLabel()),
        'class' => $definition->getClass(),
        'bundle_count' => count($bundles),
      ];
    }

    uasort($entities, static fn (array $a, array $b) => strcasecmp($a['label'], $b['label']));
    return $entities;
  }

  public function listBundles(string $entity_type_id): array {
    $definition = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);
    if (!$definition) {
      throw new \InvalidArgumentException("Unknown entity type: {$entity_type_id}");
    }

    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
    if (!$bundles) {
      $bundles = [$entity_type_id => ['label' => $definition->getLabel()]];
    }

    $out = [];
    foreach ($bundles as $bundle_id => $bundle) {
      $out[$bundle_id] = [
        'bundle_id' => $bundle_id,
        'label' => self::convertLabelToString($bundle['label'] ?? $bundle_id),
      ];
    }

    uasort($out, static fn (array $a, array $b) => strcasecmp($a['label'], $b['label']));
    return $out;
  }

  public function buildBundleReport(string $entity_type_id, string $bundle_id): array {
    $definition = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);
    if (!$definition || !$definition instanceof ContentEntityType) {
      throw new \InvalidArgumentException("Unknown or non-content entity type: {$entity_type_id}");
    }

    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
    if (!$bundles) {
      $bundles = [$entity_type_id => ['label' => $definition->getLabel()]];
    }
    if (!isset($bundles[$bundle_id])) {
      throw new \InvalidArgumentException("Unknown bundle '{$bundle_id}' for entity type '{$entity_type_id}'");
    }

    $field_map = $this->collectFieldMaps();
    $build = [
      'entity' => [
        'label' => self::convertLabelToString($definition->getLabel()),
        'machine_name' => $entity_type_id,
        'class_name' => $definition->getClass(),
      ],
      'bundle' => [
        'label' => self::convertLabelToString($bundles[$bundle_id]['label'] ?? $bundle_id),
        'machine_name' => $bundle_id,
      ],
      'fields' => [],
      'fieldMap' => [],
    ];

    if (isset($field_map[$entity_type_id])) {
      foreach ($field_map[$entity_type_id] as $field_name => $field_info) {
        foreach ($field_info['bundles'] as $field_bundle_key => $field_bundle_name) {
          $build['fieldMap'][$field_bundle_key][$field_name] = [
            'name' => $field_name,
            'type' => $field_info['type'],
          ];
        }
      }
    }

    $all_bundle_fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
    foreach ($all_bundle_fields as $bundle_field_key => $bundle_field_info) {
      $field_build = [];
      $getSettings = ($bundle_field_info->getItemDefinition()->getSettings());
      $field_build['description'] = self::convertLabelToString($bundle_field_info->getDescription());

      if (isset($getSettings['allowed_values'])) {
        $field_build['allowed_values'] = [];
        foreach ($getSettings['allowed_values'] as $allowed_value_key => $allowed_value) {
          $field_build['allowed_values'][$allowed_value_key] = $allowed_value;
        }
      }

      $extraInfo = $this->exploreType($entity_type_id, $bundle_id, $bundle_field_info, $bundle_field_key);
      $field_build['info'] = $extraInfo;
      $field_build['label_label'] = self::convertLabelToString($bundle_field_info->getLabel());
      $field_build['label_key'] = self::convertLabelToString($bundle_field_key);
      $field_build['cardinality'] = 1;
      if (method_exists($bundle_field_info, 'getCardinality')) {
        $field_build['cardinality'] = ($bundle_field_info->getCardinality());
      }

      $build['fields'][$bundle_field_key] = $field_build;
    }

    return $build;
  }
  private function collectFieldMaps(): array {
    return $this->entityFieldManager->getFieldMap();
  }

  private function collectEntityDefinitions() {
    return $this->entityTypeManager->getDefinitions();
  }

  private static function convertLabelToString(mixed $label): string {
    $outputLabel = '';
    if ($label) {
      if (is_object($label) and method_exists($label, '__toString')) {
        $outputLabel = $label->__toString();
      }
      elseif (is_string($label)) {
        $outputLabel = $label;
      }
    }
    return $outputLabel;
  }

  /**
   * @param $entity_id
   * @param $bundle_id
   * @param $bundle_field_info
   * @param $bundle_field_key
   *
   * @return array
   */
  private function exploreType($entity_id, $bundle_id, $bundle_field_info, $bundle_field_key) : array {
    $report = [
      'type' => $bundle_field_info->getType(),
    ];
    switch ($bundle_field_info->getType()) {
      case 'file':
      case 'image':
      case 'entity_reference':
      case 'entity_reference_revisions':
      case 'list_string':
      case 'list_integer':
      case 'list_float':
        /**
         * @var \Drupal\field\Entity\FieldConfig $FieldConfig
         */
        $FieldConfig = FieldConfig::loadByName($entity_id, $bundle_id, $bundle_field_key);
        if ($FieldConfig != NULL) {
          $settings = ($FieldConfig->getSettings());
          $this->getFieldSettingsReport($settings, $report, $bundle_field_key);
        }
        else {
          if (is_object($bundle_field_info)) {
            $settings = $bundle_field_info->getSettings();
            $this->getFieldSettingsReport($settings, $report, $bundle_field_key);
          }
        }
    }
    return $report;
  }

  /**
   * @param array $settings
   * @param array $report
   * @param $field
   *
   * @return void
   */
  private function getFieldSettingsReport(array $settings, array &$report, $field): void {
    foreach ($settings as $key => $value) {
      $report['settings'][$key] = ['id' => $key, 'field' => $field];
      if (is_string($value)) {
        $report['settings'][$key]['value'] = [str_replace('_', ' ',   $value)];
      }
      else {
        if (is_array($value)) {
          if (isset($value['target_bundles']) && is_array($value['target_bundles']) && count($value['target_bundles'])) {
            $report['settings'][$key]['target_bundles'] = $value['target_bundles'];
          }
        }
      }
    }
  }

}
