<?php

namespace Drupal\dev_user_cleaner\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\user\Entity\User;

/**
 * Configuration form for sanitizing users.
 */
class UserSanitizeForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dev_user_cleaner_sanitize_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dev_user_cleaner.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dev_user_cleaner.settings');

    $form['warning'] = [
      '#type' => 'markup',
      '#markup' => Markup::create('
        <div style="padding:10px; border:2px solid #e00; background:#ffe5e5; color:#900; margin-bottom:15px;">
          <strong>⚠️ WARNING:</strong> This operation <strong>should NOT be run on production environments</strong>.
          It will overwrite user email addresses and passwords with dummy values.
        </div>
      '),
    ];

    $form['sanitize_passwords'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Sanitize passwords'),
      '#default_value' => $config->get('sanitize_passwords'),
      '#description' => $this->t('If checked, user passwords will be reset to a default value.'),
    ];

    $form['sanitize_emails'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Sanitize emails'),
      '#default_value' => $config->get('sanitize_emails'),
      '#description' => $this->t('If checked, user emails will be replaced with dummy values.'),
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Sanitize Users'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable('dev_user_cleaner.settings');
    $sanitize_passwords = $form_state->getValue('sanitize_passwords');
    $sanitize_emails = $form_state->getValue('sanitize_emails');

    $config->set('sanitize_passwords', $sanitize_passwords)
      ->set('sanitize_emails', $sanitize_emails)
      ->save();

    // Load all users except admin (uid=1)
    $uids = \Drupal::entityQuery('user')
      ->accessCheck(FALSE)
      ->condition('uid', 1, '!=')
      ->execute();

    $users = User::loadMultiple($uids);

    foreach ($users as $user) {
      if ($sanitize_passwords) {
        $user->setPassword('password');
      }
      if ($sanitize_emails) {
        $user->setEmail('user' . $user->id() . '@example.com');
      }
      $user->save();
    }

    $this->messenger()->addWarning($this->t('User data has been sanitized. This operation should never be run on production environments.'));
  }

}
