<?php

declare(strict_types=1);

namespace Drupal\developer_console;

use Drupal\Core\Site\Settings;
use Symfony\Component\DependencyInjection\Attribute\AutowireCallable;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * The time counter service.
 */
final class TimeCounter {

  private ?bool $enabled = NULL;
  private float $lastTime = 0;
  private array $lastBacktrace = [];
  private string $lastInformation = '';

  public function __construct(
    #[AutowireCallable(service: RequestStack::class, method: 'getCurrentRequest', lazy: TRUE)]
    private \Closure $getCurrentRequest,
    // Use warnings to ensure diaplay and indicate time tracking shouldn't
    // happen unintentionally.
    #[AutowireCallable(service: 'logger.channel.dev_time_counter', method: 'warning', lazy: TRUE)]
    private \Closure $log,
  ) {}

  /**
   * Check if time tracking is enabled from request query string.
   */
  public function isEnabled(?Request $request = NULL): bool {
    if ($this->enabled === NULL) {
      if ($request === NULL) {
        $request = ($this->getCurrentRequest)();
      }
      $query_string = $request->getQueryString() ?? '';
      $this->enabled =
        Settings::get('dev_time_counter_enabled') === TRUE &&
        \strpos($query_string, 'track_time') !== FALSE;
    }
    return $this->enabled;
  }

  /**
   * Initialize time tracking.
   */
  public function initialize(?Request $request = NULL): void {
    if (!$this->isEnabled($request)) {
      return;
    }

    $this->lastBacktrace = \debug_backtrace(\DEBUG_BACKTRACE_IGNORE_ARGS);
    ($this->log)('Time tracking initialized in %file, line %line', [
      '%file' => $this->lastBacktrace[0]['file'],
      '%line' => $this->lastBacktrace[0]['line'],
    ]);
    $this->lastTime = \microtime(TRUE);
  }

  /**
   * Track time from last call or initialization.
   *
   * @var string $additional_information
   *   Additional information to include in the log.
   * @var float $threshold
   *   If the elapsed time will be below this value in miliseconds,
   *   log entry will not be created.
   * @var bool $include_backtrace
   */
  public function track(string $additional_information = '', float $threshold = 0, bool $include_backtrace = FALSE): void {
    if (!$this->isEnabled()) {
      return;
    }

    // Not initialized - don't do anything.
    if ($this->lastTime === 0) {
      return;
    }

    $current_time = \microtime(TRUE);
    $elapsed = \round(($current_time - $this->lastTime) * 1000, 3);
    if ($elapsed < $threshold) {
      return;
    }

    $backtrace = \debug_backtrace(\DEBUG_BACKTRACE_IGNORE_ARGS);
    if ($additional_information === '') {
      $additional_information = \sprintf('Executed in %s, line %s', $backtrace[0]['file'], $backtrace[0]['line']);
    }

    // Generate backtrace information.
    $backtrace_info = '';
    if ($include_backtrace) {
      $backtrace_info .= \PHP_EOL;
      // Shorten backtrace - we don't need same items except one.
      $backtraces = [
        'last' => $this->lastBacktrace,
        'current' => $backtrace,
      ];
      foreach (['current', 'last'] as $backtrace_name) {
        $backtrace_info .= \PHP_EOL . \ucfirst($backtrace_name) . ' backtrace: ' . \PHP_EOL;
        for ($i = 0; $i < \count($backtraces[$backtrace_name]); $i++) {
          $backtrace_info .= $i . ': ';
          $info_array = [];
          foreach (['function', 'file', 'line'] as $info_key) {
            if (\array_key_exists($info_key, $backtraces[$backtrace_name][$i])) {
              $info_array[] = $backtraces[$backtrace_name][$i][$info_key];
            }
          }
          $backtrace_info .= \implode(':', $info_array) . \PHP_EOL;
        }
      }
    }
    else {
      $backtrace_info = NULL;
    }

    // Log entry.
    ($this->log)(
      '<pre>' .
      "Time elapsed since the last breakpoint: $elapsed ms." . \PHP_EOL .
      \PHP_EOL . 'Additional information:' . \PHP_EOL . $additional_information . \PHP_EOL .
      ($threshold === 0 && $this->lastInformation !== '' ? '' : \PHP_EOL . 'Last information:' . \PHP_EOL . $this->lastInformation . \PHP_EOL) .
      $backtrace_info .
      '</pre>');
    $this->lastInformation = $additional_information;
    $this->lastTime = $current_time;
    $this->lastBacktrace = $backtrace;
  }

}
