<?php

declare(strict_types=1);

namespace Drupal\developer_console;

use Drupal\Core\Site\Settings;
use Symfony\Component\DependencyInjection\Attribute\AutowireCallable;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * The time counter service.
 */
final class TimeCounter {

  private ?bool $enabled = NULL;
  private float $lastTime = 0;

  public function __construct(
    #[AutowireCallable(service: RequestStack::class, method: 'getCurrentRequest', lazy: TRUE)]
    private \Closure $getCurrentRequest,
    // Use warnings to ensure diaplay and indicate time tracking shouldn't
    // happen unintentionally.
    #[AutowireCallable(service: 'logger.channel.dev_time_counter', method: 'warning', lazy: TRUE)]
    private \Closure $log,
  ) {}

  /**
   * Check if time tracking is enabled from request query string.
   */
  public function isEnabled(?Request $request = NULL): bool {
    if ($this->enabled === NULL) {
      if ($request === NULL) {
        $request = ($this->getCurrentRequest)();
      }
      $query_string = $request->getQueryString() ?? '';
      $this->enabled =
        Settings::get('dev_time_counter_enabled') === TRUE &&
        \strpos($query_string, 'track_time') !== FALSE;
    }
    return $this->enabled;
  }

  /**
   * Initialize time tracking.
   */
  public function initialize(?Request $request = NULL): void {
    if (!$this->isEnabled($request)) {
      return;
    }

    $backtrace_info = \debug_backtrace(\DEBUG_BACKTRACE_IGNORE_ARGS);
    ($this->log)('Time tracking initialized in %file, line %line', [
      '%file' => $backtrace_info[0]['file'],
      '%line' => $backtrace_info[0]['line'],
    ]);
    $this->lastTime = \microtime(TRUE);
  }

  /**
   * Track time from last call or initialization.
   */
  public function track(string $additional_information = ''): void {
    if (!$this->isEnabled()) {
      return;
    }

    // Not initialized - don't do anything.
    if ($this->lastTime === 0) {
      return;
    }

    if ($additional_information === '') {
      $backtrace_info = \debug_backtrace(\DEBUG_BACKTRACE_IGNORE_ARGS);
      $additional_information = \sprintf('Executed in %s, line %s', $backtrace_info[0]['file'], $backtrace_info[0]['line']);
    }

    $current_time = \microtime(TRUE);
    ($this->log)('Time elapsed since the last breakpoint: %time ms. Additional information: %info', [
      '%time' => \round(($current_time - $this->lastTime) * 1000, 3),
      '%info' => $additional_information,
    ]);
    $this->lastTime = $current_time;
  }

}
