(function (Drupal, CKEditor5) {
  /**
   * @file
   * Defines Dfm plugins for CKEditor5.
   */

  /**
   * Defines dfm.DfmSelector plugin.
   *
   * Integrates DFM button into image/link url fields.
   */
  class DfmSelector extends CKEditor5.core.Plugin {
    init() {
      this.editor.ui.on('ready', function () {
        const plugins = this.editor.plugins;
        // Image.
        if (plugins.has('ImageInsertUI')) {
          const view = plugins.get('ImageInsertUI').dropdownView;
          if (view) {
            view.once('change:isOpen', function () {
              const el =
                view.element.getElementsByClassName('ck-input-text')[0];
              dfmEditor.processCKEditor5Input(el, 'image');
            });
          }
        }
        // Image v44.
        if (plugins.has('ImageInsertViaUrlUI')) {
          plugins.get('Dialog')?.once('show:insertImageViaUrl', (evt, data) => {
            const el =
              data.content.element.getElementsByClassName('ck-input-text')[0];
              dfmEditor.processCKEditor5Input(el, 'image');
          });
        }
        // Link.
        if (plugins.has('LinkUI')) {
          const ui = plugins.get('LinkUI');
          const process = () => {
            const el = ui.formView?.urlInputView?.fieldView?.element;
            if (el) {
              ui._balloon?.view?.off('change:isVisible', process);
              dfmEditor.processCKEditor5Input(el, 'link');
              return true;
            }
          };
          process() || ui._balloon?.view?.on('change:isVisible', process);
        }
      });
    }
  }

  /**
   * Defines dfm.DfmImage plugin.
   *
   * Provides a button that inserts multiple images from DFM.
   */
  class DfmImage extends CKEditor5.core.Plugin {
    init() {
      const label = Drupal.t('Insert images');
      dfmEditor.ckeditor5PluginInit(this.editor, 'image', label);
    }
  }

  /**
   * Defines dfm.DfmLink plugin.
   *
   * Provides a button that inserts multiple file links from DFM.
   */
  class DfmLink extends CKEditor5.core.Plugin {
    init() {
      const label = Drupal.t('Insert files');
      dfmEditor.ckeditor5PluginInit(this.editor, 'link', label);
    }
  }

  /**
   * Add dfm namespace.
   */
  CKEditor5.dfm = CKEditor5.dfm || {
    DfmSelector,
    DfmImage,
    DfmLink,
  };

  /**
   * Extend window.dfmEditor.
   */
  const dfmEditor = window.dfmEditor || {};

  /**
   * Init ckeditor5 image/link plugin.
   */
  dfmEditor.ckeditor5PluginInit = function (editor, type, label) {
    editor.ui.componentFactory.add('dfm_' + type, function () {
      const button = new CKEditor5.ui.ButtonView();
      button.set({
        label,
        class: 'ck-dfm-button ck-dfm-' + type + '-button',
        tooltip: true,
      });
      button.on('execute', function () {
        dfmEditor.ckeditor5Popup(editor, type);
      });
      return button;
    });
  };

  /**
   * Integrates DFM into a CKEditor5 url input.
   */
  dfmEditor.processCKEditor5Input = function (el, type) {
    if (!el) {
      return;
    }
    const name = 'ckeditor5Handler' + (Math.random() + '').substring(2);
    dfmEditor[name] = function (File, win) {
      el.value = File.getUrl();
      win.close();
      el.focus();
      el.dispatchEvent(new CustomEvent('input'));
      // Auto submit.
      if (el.form) {
        const button = el.form.getElementsByClassName('ck-button-save')[0];
        if (button) {
          button.click();
        }
      }
    };
    const button = dfmEditor.processUrlInput(el, type, 'after');
    button.className += ' dfm-selector-button';
    button.onclick = function () {
      dfmEditor.open('dfmEditor.' + name, type);
      return false;
    };
    el.parentNode.className += ' ck-dfm-wrp ck-dfm-' + type + '-wrp';
    return button;
  };

  /**
   * Opens dfm for inserting files/images into CKEditor5.
   */
  dfmEditor.ckeditor5Popup = function (editor, type) {
    const id = editor.sourceElement.getAttribute('data-ckeditor5-id');
    return dfmEditor.open('dfmEditor.sendtoCKEditor5', type, 'cke_id=' + id);
  };

  /**
   * Dfm sendto handler for inserting files/images into CKEditor5.
   */
  dfmEditor.sendtoCKEditor5 = function (File, win) {
    const dfm = win.dfm;
    const editor = Drupal.CKEditor5Instances.get(dfm.urlParam('cke_id'));
    if (editor) {
      const isImg = dfm.urlParam('type') === 'image';
      const selected = dfm.getSelectedItems();
      const inner = !isImg ? dfmEditor.ckeditor5GetSelection(editor) : '';
      const html = dfmEditor.filesHtml(selected, isImg, inner, '<br />');
      dfmEditor.ckeditor5SetSelection(editor, html, true);
    }
    win.close();
  };

  /**
   * Returns selected html from CKEditor5.
   */
  dfmEditor.ckeditor5GetSelection = function (editor) {
    let html = '';
    try {
      const model = editor.model;
      const content = model.getSelectedContent(model.document.selection);
      html = editor.data.stringify(content);
    } catch (err) {
      console.error(err);
    }
    return html;
  };

  /**
   * Inserts html into CKEditor5 selection.
   */
  dfmEditor.ckeditor5SetSelection = function (editor, html, focus) {
    try {
      const viewFragment = editor.data.processor.toView(html);
      const modelFragment = editor.data.toModel(viewFragment);
      editor.model.insertContent(modelFragment);
      if (focus) {
        editor.editing.view.focus();
      }
    } catch (err) {
      console.error(err);
    }
  };
})(Drupal, CKEditor5);
