<?php

namespace Drupal\dga_feedback\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dga_feedback\Service\DgaFeedbackService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for deleting a feedback submission.
 */
class DgaFeedbackDeleteForm extends ConfirmFormBase {

  /**
   * The feedback service.
   *
   * @var \Drupal\dga_feedback\Service\DgaFeedbackService
   */
  protected $feedbackService;

  /**
   * The submission ID.
   *
   * @var int
   */
  protected $submissionId;

  /**
   * The submission data.
   *
   * @var array|false
   */
  protected $submission;

  /**
   * Constructs a DgaFeedbackDeleteForm object.
   *
   * @param \Drupal\dga_feedback\Service\DgaFeedbackService $feedback_service
   *   The feedback service.
   */
  public function __construct(DgaFeedbackService $feedback_service) {
    $this->feedbackService = $feedback_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_feedback.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_feedback_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete this feedback submission?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('dga_feedback.admin');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    if ($this->submission) {
      return $this->t('This action cannot be undone. Feedback ID: @id, Useful: @useful', [
        '@id' => $this->submissionId,
        '@useful' => $this->submission['is_useful'] ?? 'unknown',
      ]);
    }
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->submissionId = $id;
    $this->submission = $this->feedbackService->getSubmissionById($id);
    
    if (!$this->submission) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $success = $this->feedbackService->deleteSubmission($this->submissionId);

    if ($success) {
      $this->messenger()->addMessage($this->t('Feedback submission deleted successfully.'));
      $form_state->setRedirect('dga_feedback.admin');
    } else {
      $this->messenger()->addError($this->t('Failed to delete feedback submission.'));
    }
  }

}

