<?php

namespace Drupal\dga_feedback\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DGA Feedback widget text settings.
 */
class DgaFeedbackSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dga_feedback.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_feedback_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    $form['closed_state'] = [
      '#type' => 'details',
      '#title' => $this->t('Closed State (Default View)'),
      '#description' => $this->t('Text displayed when the feedback widget is closed.'),
      '#open' => TRUE,
    ];

    $form['closed_state']['question_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Question text'),
      '#description' => $this->t('The main question displayed (e.g., "Was this page useful?").'),
      '#default_value' => $config->get('question_text') ?? 'Was this page useful?',
      '#required' => TRUE,
    ];

    $form['closed_state']['yes_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Yes button text'),
      '#description' => $this->t('Text on the Yes button.'),
      '#default_value' => $config->get('yes_button_text') ?? 'Yes',
      '#required' => TRUE,
    ];

    $form['closed_state']['no_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('No button text'),
      '#description' => $this->t('Text on the No button.'),
      '#default_value' => $config->get('no_button_text') ?? 'No',
      '#required' => TRUE,
    ];

    $form['closed_state']['stats_text_template'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Statistics text template'),
      '#description' => $this->t('Template for statistics text. Use @percentage for percentage and @count for total count.'),
      '#default_value' => $config->get('stats_text_template') ?? '@percentage% of users said Yes from @count Feedbacks',
      '#required' => TRUE,
    ];

    $form['form_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Feedback Form'),
      '#description' => $this->t('Text displayed in the feedback form.'),
      '#open' => TRUE,
    ];

    $form['form_section']['close_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close button text'),
      '#default_value' => $config->get('close_button_text') ?? 'Close',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reasons title'),
      '#description' => $this->t('Title for the reasons section (e.g., "Please tell us why").'),
      '#default_value' => $config->get('reasons_title') ?? 'Please tell us why',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_instruction'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reasons instruction'),
      '#description' => $this->t('Instruction text for reasons (e.g., "(you can select multiple options)").'),
      '#default_value' => $config->get('reasons_instruction') ?? '(you can select multiple options)',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_yes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reasons for Yes'),
      '#description' => $this->t('One reason per line. These will be shown when user clicks Yes.'),
      '#default_value' => $config->get('reasons_yes') ?? "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else",
      '#required' => TRUE,
      '#rows' => 5,
    ];

    $form['form_section']['reasons_no'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reasons for No'),
      '#description' => $this->t('One reason per line. These will be shown when user clicks No.'),
      '#default_value' => $config->get('reasons_no') ?? "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else",
      '#required' => TRUE,
      '#rows' => 5,
    ];

    $form['form_section']['feedback_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Feedback label'),
      '#default_value' => $config->get('feedback_label') ?? 'Feedback',
      '#required' => TRUE,
    ];

    $form['form_section']['feedback_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Feedback placeholder'),
      '#description' => $this->t('Placeholder text for the feedback textarea.'),
      '#default_value' => $config->get('feedback_placeholder') ?? 'text placeholder',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Gender label'),
      '#description' => $this->t('Label for gender section (e.g., "I\'m").'),
      '#default_value' => $config->get('gender_label') ?? 'I\'m',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_male'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Male option text'),
      '#default_value' => $config->get('gender_male') ?? 'Male',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_female'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Female option text'),
      '#default_value' => $config->get('gender_female') ?? 'Female',
      '#required' => TRUE,
    ];

    $form['form_section']['submit_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit button text'),
      '#default_value' => $config->get('submit_button_text') ?? 'Submit',
      '#required' => TRUE,
    ];

    $form['submitted_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Submitted State'),
      '#description' => $this->t('Text displayed after feedback is submitted.'),
      '#open' => FALSE,
    ];

    $form['submitted_section']['submitted_success_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Success message'),
      '#description' => $this->t('Message shown when feedback is successfully submitted.'),
      '#default_value' => $config->get('submitted_success_text') ?? 'Your feedback is submitted!',
      '#required' => TRUE,
    ];

    $form['security_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Security & Limits'),
      '#description' => $this->t('Configure security settings and input limits.'),
      '#open' => FALSE,
    ];

    $form['security_section']['rate_limit_max_submissions'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Maximum submissions per IP'),
      '#description' => $this->t('Maximum number of submissions allowed per IP address within the time window. Set to 0 to disable rate limiting.'),
      '#default_value' => $config->get('rate_limit_max_submissions') ?? 20,
      '#min' => 0,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    $form['security_section']['rate_limit_time_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Time window (seconds)'),
      '#description' => $this->t('Time window in seconds for rate limiting (e.g., 3600 = 1 hour).'),
      '#default_value' => $config->get('rate_limit_time_window') ?? 3600,
      '#min' => 60,
      '#max' => 86400,
      '#required' => TRUE,
      '#suffix' => '<div class="description">' . $this->t('Common values: 3600 (1 hour), 7200 (2 hours), 86400 (24 hours)') . '</div>',
    ];

    $form['security_section']['feedback_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum feedback text length'),
      '#description' => $this->t('Maximum number of characters allowed in feedback text.'),
      '#default_value' => $config->get('feedback_max_length') ?? 5000,
      '#min' => 100,
      '#max' => 50000,
      '#required' => TRUE,
    ];

    $form['security_section']['reason_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum length per reason'),
      '#description' => $this->t('Maximum number of characters allowed per reason text.'),
      '#default_value' => $config->get('reason_max_length') ?? 200,
      '#min' => 50,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    $form['security_section']['reason_max_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of reasons'),
      '#description' => $this->t('Maximum number of reasons a user can select.'),
      '#default_value' => $config->get('reason_max_count') ?? 10,
      '#min' => 1,
      '#max' => 50,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    // Closed state
    $config->set('question_text', $form_state->getValue('question_text'));
    $config->set('yes_button_text', $form_state->getValue('yes_button_text'));
    $config->set('no_button_text', $form_state->getValue('no_button_text'));
    $config->set('stats_text_template', $form_state->getValue('stats_text_template'));

    // Form section
    $config->set('close_button_text', $form_state->getValue('close_button_text'));
    $config->set('reasons_title', $form_state->getValue('reasons_title'));
    $config->set('reasons_instruction', $form_state->getValue('reasons_instruction'));
    $config->set('reasons_yes', $form_state->getValue('reasons_yes'));
    $config->set('reasons_no', $form_state->getValue('reasons_no'));
    $config->set('feedback_label', $form_state->getValue('feedback_label'));
    $config->set('feedback_placeholder', $form_state->getValue('feedback_placeholder'));
    $config->set('gender_label', $form_state->getValue('gender_label'));
    $config->set('gender_male', $form_state->getValue('gender_male'));
    $config->set('gender_female', $form_state->getValue('gender_female'));
    $config->set('submit_button_text', $form_state->getValue('submit_button_text'));

    // Submitted section
    $config->set('submitted_success_text', $form_state->getValue('submitted_success_text'));

    // Security section
    $config->set('rate_limit_max_submissions', (int) $form_state->getValue('rate_limit_max_submissions'));
    $config->set('rate_limit_time_window', (int) $form_state->getValue('rate_limit_time_window'));
    $config->set('feedback_max_length', (int) $form_state->getValue('feedback_max_length'));
    $config->set('reason_max_length', (int) $form_state->getValue('reason_max_length'));
    $config->set('reason_max_count', (int) $form_state->getValue('reason_max_count'));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}

