<?php

namespace Drupal\dga_feedback\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\dga_feedback\Service\DgaFeedbackService;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a DGA Feedback widget block.
 *
 * @Block(
 *   id = "dga_feedback_block",
 *   admin_label = @Translation("DGA Feedback Widget"),
 *   category = @Translation("Custom")
 * )
 */
class DgaFeedbackBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The feedback service.
   *
   * @var \Drupal\dga_feedback\Service\DgaFeedbackService
   */
  protected $feedbackService;

  /**
   * Constructs a DgaFeedbackBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dga_feedback\Service\DgaFeedbackService $feedback_service
   *   The feedback service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DgaFeedbackService $feedback_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->feedbackService = $feedback_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dga_feedback.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get current URL path - use canonical path, not alias, for consistency.
    $current_path = \Drupal::service('path.current')->getPath();

    // Get the current request to extract the actual URL path
    $request = \Drupal::request();
    $request_path = $request->getPathInfo();

    // Get the alias
    $url_alias = \Drupal::service('path_alias.manager')->getAliasByPath($current_path);

    // IMPORTANT: Use alias (without language prefix) for consistency
    if ($url_alias !== $current_path && $url_alias !== '/') {
      $url = rtrim($url_alias, '/');
    } else {
      // If no alias, strip language prefix from request path
      $url = $request_path;
      if (preg_match('#^/[a-z]{2}/(.+)$#', $url, $matches)) {
        $url = '/' . $matches[1];
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Ensure we don't have language prefix in the final URL
    if (preg_match('#^/[a-z]{2}/#', $url)) {
      $url = preg_replace('#^/[a-z]{2}#', '', $url);
      $url = rtrim($url, '/') ?: '/';
    }

    // Try to get entity context if available.
    $entity_type = NULL;
    $entity_id = NULL;
    $route_match = \Drupal::routeMatch();

    // Check if we're on a node page.
    $node = $route_match->getParameter('node');
    if ($node instanceof Node) {
      $entity_type = 'node';
      $entity_id = $node->id();
      if ($entity_id > 0) {
        $stats = $this->feedbackService->getStatistics($entity_type, $entity_id, $url);
      } else {
        $stats = $this->feedbackService->getStatisticsByUrl($url);
      }
    }
    else {
      // Fallback to URL-based statistics.
      $stats = $this->feedbackService->getStatisticsByUrl($url);
    }

    // Format the yes percentage
    $yes_percentage = $stats['yes_percentage'] > 0 ? number_format($stats['yes_percentage'], 0, '.', '') : '0';
    $total_count = (int) $stats['total_count'];

    // Ensure count is always a number (not string).
    $total_count = max(0, $total_count);

    // Load configuration for text strings.
    $config = \Drupal::config('dga_feedback.settings');

    // Translate reasons - split by newline and translate each reason individually
    $reasons_yes_raw = $config->get('reasons_yes') ?? "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else";
    $reasons_no_raw = $config->get('reasons_no') ?? "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else";
    
    $reasons_yes_lines = explode("\n", $reasons_yes_raw);
    $reasons_no_lines = explode("\n", $reasons_no_raw);
    
    $reasons_yes_translated = [];
    $reasons_no_translated = [];
    
    foreach ($reasons_yes_lines as $reason) {
      $reason = trim($reason);
      if (!empty($reason)) {
        $reasons_yes_translated[] = $this->t($reason)->render();
      }
    }
    
    foreach ($reasons_no_lines as $reason) {
      $reason = trim($reason);
      if (!empty($reason)) {
        $reasons_no_translated[] = $this->t($reason)->render();
      }
    }

    // Build render array.
    $build = [
      '#theme' => 'dga_feedback_widget',
      '#yes_percentage' => $yes_percentage,
      '#total_count' => $total_count,
      '#url' => $url,
      '#entity_type' => $entity_type,
      '#entity_id' => $entity_id,
      '#texts' => [
        'question_text' => $this->t($config->get('question_text') ?? 'Was this page useful?')->render(),
        'yes_button_text' => $this->t($config->get('yes_button_text') ?? 'Yes')->render(),
        'no_button_text' => $this->t($config->get('no_button_text') ?? 'No')->render(),
        'stats_text_template' => $this->t($config->get('stats_text_template') ?? '@percentage% of users said Yes from @count Feedbacks')->render(),
        'close_button_text' => $this->t($config->get('close_button_text') ?? 'Close')->render(),
        'reasons_title' => $this->t($config->get('reasons_title') ?? 'Please tell us why')->render(),
        'reasons_instruction' => $this->t($config->get('reasons_instruction') ?? '(you can select multiple options)')->render(),
        'feedback_label' => $this->t($config->get('feedback_label') ?? 'Feedback')->render(),
        'feedback_placeholder' => $this->t($config->get('feedback_placeholder') ?? 'text placeholder')->render(),
        'gender_label' => $this->t($config->get('gender_label') ?? 'I\'m')->render(),
        'gender_male' => $this->t($config->get('gender_male') ?? 'Male')->render(),
        'gender_female' => $this->t($config->get('gender_female') ?? 'Female')->render(),
        'submit_button_text' => $this->t($config->get('submit_button_text') ?? 'Submit')->render(),
        'submitted_success_text' => $this->t($config->get('submitted_success_text') ?? 'Your feedback is submitted!')->render(),
        'reasons_yes' => implode("\n", $reasons_yes_translated),
        'reasons_no' => implode("\n", $reasons_no_translated),
      ],
      '#attached' => [
        'library' => [
          'dga_feedback/feedback_widget',
        ],
        'drupalSettings' => [
          'dgaFeedback' => [
            'submitUrl' => \Drupal\Core\Url::fromRoute('dga_feedback.submit')->toString(),
            'statsUrl' => \Drupal\Core\Url::fromRoute('dga_feedback.stats')->toString(),
            'questionText' => $this->t($config->get('question_text') ?? 'Was this page useful?')->render(),
            'submittedSuccessText' => $this->t($config->get('submitted_success_text') ?? 'Your feedback is submitted!')->render(),
            'errorMessage' => $this->t('Failed to submit feedback. Please try again.'),
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0, // Disable cache for dynamic content.
        'contexts' => [
          'url.path',
          'route',
          'user.roles:anonymous',
        ],
        'tags' => [
          'dga_feedback:submissions',
          'config:dga_feedback.settings',
        ],
      ],
    ];

    return $build;
  }

}

