<?php

namespace Drupal\dga_feedback\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DGA Feedback widget text settings.
 */
class DgaFeedbackSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dga_feedback.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_feedback_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    $form['intro'] = [
      '#type' => 'markup',
      '#markup' => '<div class="description">' . $this->t('Use this form to configure the default English text, widget behaviour, and security limits. To manage Arabic translations (or override English text), go to <strong>DGA Feedback → Translations</strong>.') . '</div>',
    ];

    $form['closed_state'] = [
      '#type' => 'details',
      '#title' => $this->t('Closed State (Default View)'),
      '#description' => $this->t('Text displayed when the feedback widget is closed. The labels below represent the default English text.'),
      '#open' => TRUE,
    ];

    $form['closed_state']['question_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Question text'),
      '#description' => $this->t('The main question displayed (e.g., "Was this page useful?").'),
      '#default_value' => $config->get('question_text_en') ?? 'Was this page useful?',
      '#required' => TRUE,
    ];

    $form['closed_state']['yes_button_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Yes button text'),
      '#description' => $this->t('Text on the Yes button.'),
      '#default_value' => $config->get('yes_button_text_en') ?? 'Yes',
      '#required' => TRUE,
    ];

    $form['closed_state']['no_button_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('No button text'),
      '#description' => $this->t('Text on the No button.'),
      '#default_value' => $config->get('no_button_text_en') ?? 'No',
      '#required' => TRUE,
    ];

    $form['closed_state']['stats_text_template_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Statistics text template'),
      '#description' => $this->t('Template for statistics text. Use @percentage for percentage and @count for total count.'),
      '#default_value' => $config->get('stats_text_template_en') ?? '@percentage% of users said Yes from @count Feedbacks',
      '#required' => TRUE,
    ];

    $form['form_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Feedback Form'),
      '#description' => $this->t('Text displayed when the widget is opened. These values represent the default English copy.'),
      '#open' => TRUE,
    ];

    $form['form_section']['close_button_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close button text'),
      '#default_value' => $config->get('close_button_text_en') ?? 'Close',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_title_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reasons title'),
      '#description' => $this->t('Title for the reasons section (e.g., "Please tell us why").'),
      '#default_value' => $config->get('reasons_title_en') ?? 'Please tell us why',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_instruction_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reasons instruction'),
      '#description' => $this->t('Instruction text for reasons (e.g., "you can select multiple options"). Note: Parentheses will be added automatically in the widget.'),
      '#default_value' => $config->get('reasons_instruction_en') ?? 'you can select multiple options',
      '#required' => TRUE,
    ];

    $form['form_section']['reasons_yes_en'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reasons for Yes'),
      '#description' => $this->t('One reason per line. These will be shown when a user selects "Yes".'),
      '#default_value' => $config->get('reasons_yes_en') ?? "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else",
      '#required' => TRUE,
      '#rows' => 5,
    ];

    $form['form_section']['reasons_no_en'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reasons for No'),
      '#description' => $this->t('One reason per line. These will be shown when a user selects "No".'),
      '#default_value' => $config->get('reasons_no_en') ?? "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else",
      '#required' => TRUE,
      '#rows' => 5,
    ];

    $form['form_section']['feedback_label_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Feedback label'),
      '#default_value' => $config->get('feedback_label_en') ?? 'Feedback',
      '#required' => TRUE,
    ];

    $form['form_section']['feedback_placeholder_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Feedback placeholder'),
      '#description' => $this->t('Placeholder text for the feedback textarea.'),
      '#default_value' => $config->get('feedback_placeholder_en') ?? 'text placeholder',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_label_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Gender label'),
      '#description' => $this->t('Label for gender section (e.g., "I\'m").'),
      '#default_value' => $config->get('gender_label_en') ?? 'I\'m',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_male_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Male option text'),
      '#default_value' => $config->get('gender_male_en') ?? 'Male',
      '#required' => TRUE,
    ];

    $form['form_section']['gender_female_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Female option text'),
      '#default_value' => $config->get('gender_female_en') ?? 'Female',
      '#required' => TRUE,
    ];

    $form['form_section']['submit_button_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit button text'),
      '#default_value' => $config->get('submit_button_text_en') ?? 'Submit',
      '#required' => TRUE,
    ];

    $form['submitted_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Submitted State'),
      '#description' => $this->t('Text displayed after feedback is submitted.'),
      '#open' => FALSE,
    ];

    $form['submitted_section']['submitted_success_text_en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Success message'),
      '#description' => $this->t('Message shown when feedback is successfully submitted.'),
      '#default_value' => $config->get('submitted_success_text_en') ?? 'Your feedback is submitted!',
      '#required' => TRUE,
    ];

    $form['behavior_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Widget Behavior'),
      '#description' => $this->t('Configure widget behavior after submission.'),
      '#open' => FALSE,
    ];

    $form['behavior_section']['refresh_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Refresh delay (seconds)'),
      '#description' => $this->t('Time in seconds to wait before refreshing the widget after a successful submission. Default: 3 seconds.'),
      '#default_value' => ($config->get('refresh_delay') ?? 3000) / 1000,
      '#min' => 1,
      '#max' => 30,
      '#step' => 0.5,
      '#required' => TRUE,
      '#suffix' => '<div class="description">' . $this->t('Common values: 2 seconds, 3 seconds, 5 seconds') . '</div>',
    ];

    $form['security_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Security & Limits'),
      '#description' => $this->t('Configure security settings and input limits.'),
      '#open' => FALSE,
    ];

    $form['security_section']['rate_limit_max_submissions'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Maximum submissions per IP'),
      '#description' => $this->t('Maximum number of submissions allowed per IP address within the time window. Set to 0 to disable rate limiting.'),
      '#default_value' => $config->get('rate_limit_max_submissions') ?? 20,
      '#min' => 0,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    $form['security_section']['rate_limit_time_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Time window (seconds)'),
      '#description' => $this->t('Time window in seconds for rate limiting (e.g., 3600 = 1 hour).'),
      '#default_value' => $config->get('rate_limit_time_window') ?? 3600,
      '#min' => 60,
      '#max' => 86400,
      '#required' => TRUE,
      '#suffix' => '<div class="description">' . $this->t('Common values: 3600 (1 hour), 7200 (2 hours), 86400 (24 hours)') . '</div>',
    ];

    $form['security_section']['feedback_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum feedback text length'),
      '#description' => $this->t('Maximum number of characters allowed in feedback text.'),
      '#default_value' => $config->get('feedback_max_length') ?? 5000,
      '#min' => 100,
      '#max' => 50000,
      '#required' => TRUE,
    ];

    $form['security_section']['reason_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum length per reason'),
      '#description' => $this->t('Maximum number of characters allowed per reason text.'),
      '#default_value' => $config->get('reason_max_length') ?? 200,
      '#min' => 50,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    $form['security_section']['reason_max_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of reasons'),
      '#description' => $this->t('Maximum number of reasons a user can select.'),
      '#default_value' => $config->get('reason_max_count') ?? 10,
      '#min' => 1,
      '#max' => 50,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    // Security section
    $config->set('rate_limit_max_submissions', (int) $form_state->getValue('rate_limit_max_submissions'));
    $config->set('rate_limit_time_window', (int) $form_state->getValue('rate_limit_time_window'));
    $config->set('feedback_max_length', (int) $form_state->getValue('feedback_max_length'));
    $config->set('reason_max_length', (int) $form_state->getValue('reason_max_length'));
    $config->set('reason_max_count', (int) $form_state->getValue('reason_max_count'));

    // Behavior section.
    $refresh_delay_seconds = (float) $form_state->getValue('refresh_delay');
    $config->set('refresh_delay', (int) ($refresh_delay_seconds * 1000));

    // Closed state (English defaults).
    $config->set('question_text_en', $form_state->getValue('question_text_en'));
    $config->set('yes_button_text_en', $form_state->getValue('yes_button_text_en'));
    $config->set('no_button_text_en', $form_state->getValue('no_button_text_en'));
    $config->set('stats_text_template_en', $form_state->getValue('stats_text_template_en'));

    // Form section (English defaults).
    $config->set('close_button_text_en', $form_state->getValue('close_button_text_en'));
    $config->set('reasons_title_en', $form_state->getValue('reasons_title_en'));
    $config->set('reasons_instruction_en', $form_state->getValue('reasons_instruction_en'));
    $config->set('reasons_yes_en', $form_state->getValue('reasons_yes_en'));
    $config->set('reasons_no_en', $form_state->getValue('reasons_no_en'));
    $config->set('feedback_label_en', $form_state->getValue('feedback_label_en'));
    $config->set('feedback_placeholder_en', $form_state->getValue('feedback_placeholder_en'));
    $config->set('gender_label_en', $form_state->getValue('gender_label_en'));
    $config->set('gender_male_en', $form_state->getValue('gender_male_en'));
    $config->set('gender_female_en', $form_state->getValue('gender_female_en'));
    $config->set('submit_button_text_en', $form_state->getValue('submit_button_text_en'));

    // Submitted state (English defaults).
    $config->set('submitted_success_text_en', $form_state->getValue('submitted_success_text_en'));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}

