<?php

namespace Drupal\dga_feedback\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\dga_feedback\Service\DgaFeedbackService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for the admin submissions page.
 */
class DgaFeedbackAdminController extends ControllerBase {

  /**
   * The feedback service.
   *
   * @var \Drupal\dga_feedback\Service\DgaFeedbackService
   */
  protected $feedbackService;

  /**
   * Constructs a DgaFeedbackAdminController object.
   *
   * @param \Drupal\dga_feedback\Service\DgaFeedbackService $feedback_service
   *   The feedback service.
   */
  public function __construct(DgaFeedbackService $feedback_service) {
    $this->feedbackService = $feedback_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_feedback.service')
    );
  }

  /**
   * Displays all feedback submissions and statistics.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   Render array for the admin page.
   */
  public function listSubmissions(Request $request) {
    // Get pagination parameters.
    $page = $request->query->get('page', 0);
    $limit = 50;
    $offset = $page * $limit;

    // Get sort parameters.
    $sort_by = $request->query->get('sort', 'created');
    $sort_direction = $request->query->get('order', 'DESC');

    // Get filter parameters.
    $filters = [];
    if ($id = $request->query->get('id')) {
      $filters['id'] = (int) $id;
    }
    if ($entity_type = $request->query->get('entity_type')) {
      $filters['entity_type'] = $entity_type;
    }
    if ($entity_id = $request->query->get('entity_id')) {
      $filters['entity_id'] = (int) $entity_id;
    }
    if ($url = $request->query->get('url')) {
      $filters['url'] = $url;
    }
    if ($is_useful = $request->query->get('is_useful')) {
      $filters['is_useful'] = $is_useful;
    }
    if ($feedback = $request->query->get('feedback')) {
      $filters['feedback'] = $feedback;
    }
    if ($user_id = $request->query->get('user_id')) {
      $filters['user_id'] = (int) $user_id;
    }
    if ($ip_address = $request->query->get('ip_address')) {
      $filters['ip_address'] = $ip_address;
    }
    if ($date_from = $request->query->get('date_from')) {
      $filters['date_from'] = strtotime($date_from);
    }
    if ($date_to = $request->query->get('date_to')) {
      $filters['date_to'] = strtotime($date_to . ' 23:59:59');
    }

    // Get URL statistics grouping parameters.
    $url_stats_limit = (int) $request->query->get('url_stats_limit', 20);
    $url_stats_order_by = $request->query->get('url_stats_order', 'count');
    $url_stats_order_direction = $request->query->get('url_stats_order_dir', 'DESC');

    // Get all submissions.
    $submissions = $this->feedbackService->getAllSubmissions($limit, $offset, $filters, $sort_by, $sort_direction);
    $total_count = $this->feedbackService->getSubmissionsCount($filters);
    $total_count = (int) ($total_count ?? 0);

    // Get overall statistics.
    $overall_stats_raw = $this->feedbackService->getOverallStatistics();
    $overall_stats = [
      'yes_percentage' => $overall_stats_raw['yes_percentage'] ?? 0.0,
      'total_count' => $overall_stats_raw['total_count'] ?? 0,
    ];

    $usefulness_distribution = $this->feedbackService->getUsefulnessDistribution();
    $usefulness_distribution = [
      'yes' => (int) ($usefulness_distribution['yes'] ?? 0),
      'no' => (int) ($usefulness_distribution['no'] ?? 0),
    ];

    // Get statistics grouped by URL.
    $url_statistics = $this->feedbackService->getStatisticsGroupedByUrl($url_stats_limit, $url_stats_order_by, $url_stats_order_direction);
    
    // Get total count of unique URLs for average calculation.
    $unique_url_count = $this->feedbackService->getUniqueUrlCount();

    // Get additional statistics for advanced blocks.
    $top_feedback_page = $this->feedbackService->getMostUsefulPage();
    $most_feedback_page = $this->feedbackService->getMostFeedbackPage();
    $recent_activity = $this->feedbackService->getRecentActivity();
    $useful_percentage = $this->feedbackService->getUsefulPercentage();
    $user_type_stats = $this->feedbackService->getSubmissionsByUserType();

    // Process submissions for display.
    $processed_submissions = [];
    foreach ($submissions as $submission) {
      $processed_submissions[] = [
        'id' => $submission['id'] ?? 0,
        'is_useful' => $submission['is_useful'] ?? '',
        'reasons' => $submission['reasons'] ?? [],
        'feedback' => $submission['feedback'] ?? '',
        'gender' => $submission['gender'] ?? '',
        'url' => $submission['url'] ?? '/',
        'entity_type' => $submission['entity_type'] ?? '',
        'entity_id' => $submission['entity_id'] ?? 0,
        'user_id' => $submission['user_id'] ?? NULL,
        'ip_address' => $submission['ip_address'] ?? '',
        'created' => $submission['created'] ?? time(),
        'created_formatted' => isset($submission['created']) ? date('Y-m-d H:i:s', $submission['created']) : date('Y-m-d H:i:s'),
      ];
    }

    // Build pagination.
    $pager_manager = \Drupal::service('pager.manager');
    $pager_manager->createPager($total_count, $limit);

    $build = [
      '#theme' => 'dga_feedback_admin',
      '#overall_stats' => $overall_stats,
      '#usefulness_distribution' => $usefulness_distribution,
      '#url_statistics' => $url_statistics,
      '#unique_url_count' => $unique_url_count,
      '#submissions' => $processed_submissions,
      '#total_count' => $total_count,
      '#current_page' => $page,
      '#limit' => $limit,
      '#filters' => $filters,
      '#sort_by' => $sort_by,
      '#sort_direction' => $sort_direction,
      '#url_stats_limit' => $url_stats_limit,
      '#url_stats_order_by' => $url_stats_order_by,
      '#url_stats_order_direction' => $url_stats_order_direction,
      '#top_feedback_page' => $top_feedback_page,
      '#most_feedback_page' => $most_feedback_page,
      '#recent_activity' => $recent_activity,
      '#useful_percentage' => $useful_percentage,
      '#user_type_stats' => $user_type_stats,
      '#attached' => [
        'library' => [
          'dga_feedback/admin',
        ],
        'drupalSettings' => [
          'dgaFeedback' => [
            'bulkDeleteUrl' => Url::fromRoute('dga_feedback.admin.bulk_delete')->toString(),
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
    
    // Add pager to render array
    $build['pager'] = [
      '#type' => 'pager',
    ];
    
    return $build;
  }

  /**
   * Displays edit form for a feedback submission.
   *
   * @param int $id
   *   The feedback submission ID.
   *
   * @return array
   *   Render array for the edit form.
   */
  public function editSubmission($id) {
    // Build form using Drupal's form API.
    $form = \Drupal::formBuilder()->getForm('\Drupal\dga_feedback\Form\DgaFeedbackEditForm', $id);
    
    return $form;
  }

  /**
   * Handles bulk delete operation.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to the admin page.
   */
  public function bulkDelete(Request $request) {
    // Get submissions array from request
    $ids = [];
    $request_data = $request->request->all();
    
    if (isset($request_data['submissions'])) {
      $submissions = $request_data['submissions'];
      if (is_array($submissions)) {
        $ids = $submissions;
      } elseif (is_scalar($submissions)) {
        $ids = [$submissions];
      }
    }
    
    if (empty($ids)) {
      $this->messenger()->addError($this->t('No submissions selected for deletion.'));
      return new RedirectResponse(Url::fromRoute('dga_feedback.admin')->toString());
    }

    $deleted_count = $this->feedbackService->bulkDeleteSubmissions($ids);

    if ($deleted_count > 0) {
      $this->messenger()->addMessage($this->formatPlural($deleted_count, 'Deleted @count submission.', 'Deleted @count submissions.', ['@count' => $deleted_count]));
    } else {
      $this->messenger()->addError($this->t('Failed to delete submissions.'));
    }

    return new RedirectResponse(Url::fromRoute('dga_feedback.admin')->toString());
  }

}

