<?php

namespace Drupal\dga_feedback\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dga_feedback\Service\DgaFeedbackService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for editing a feedback submission.
 */
class DgaFeedbackEditForm extends FormBase {

  /**
   * The feedback service.
   *
   * @var \Drupal\dga_feedback\Service\DgaFeedbackService
   */
  protected $feedbackService;

  /**
   * The submission ID.
   *
   * @var int
   */
  protected $submissionId;

  /**
   * Constructs a DgaFeedbackEditForm object.
   *
   * @param \Drupal\dga_feedback\Service\DgaFeedbackService $feedback_service
   *   The feedback service.
   */
  public function __construct(DgaFeedbackService $feedback_service) {
    $this->feedbackService = $feedback_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_feedback.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_feedback_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->submissionId = $id;
    
    $submission = $this->feedbackService->getSubmissionById($id);
    
    if (!$submission) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    $form['is_useful'] = [
      '#type' => 'select',
      '#title' => $this->t('Was this useful?'),
      '#options' => [
        'yes' => $this->t('Yes'),
        'no' => $this->t('No'),
      ],
      '#default_value' => $submission['is_useful'] ?? 'yes',
      '#required' => TRUE,
    ];

    // Load reasons from config to show checkboxes
    $config = \Drupal::config('dga_feedback.settings');
    $language_manager = \Drupal::languageManager();
    $is_arabic = $language_manager->getCurrentLanguage()->getId() === 'ar';

    $getReasons = function ($base_key, $default_en, $default_ar) use ($config, $is_arabic) {
      $value_en = $config->get($base_key . '_en');
      $value_ar = $config->get($base_key . '_ar');

      $list = $is_arabic
        ? ($value_ar ?? $value_en ?? $default_ar)
        : ($value_en ?? $default_en);

      $lines = explode("\n", $list);
      return array_values(array_filter(array_map('trim', $lines)));
    };

    $reasons_yes = $getReasons(
      'reasons_yes',
      "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else",
      "المحتوى ذو صلة\nكان مكتوبًا بشكل جيد\nجعل التخطيط القراءة سهلة\nشيء آخر"
    );
    $reasons_no = $getReasons(
      'reasons_no',
      "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else",
      "المحتوى غير ذي صلة\nالمحتوى غير دقيق\nالمحتوى طويل جدًا\nشيء آخر"
    );
    
    $is_useful = $submission['is_useful'] ?? 'yes';
    $reasons_list = ($is_useful === 'yes') ? $reasons_yes : $reasons_no;
    $selected_reasons = $submission['reasons'] ?? [];

    $form['reasons'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Reasons'),
      '#options' => array_combine($reasons_list, $reasons_list),
      '#default_value' => array_intersect($selected_reasons, $reasons_list),
    ];

    $form['feedback'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Feedback'),
      '#default_value' => $submission['feedback'] ?? '',
      '#rows' => 5,
    ];

    $form['gender'] = [
      '#type' => 'select',
      '#title' => $this->t('Gender'),
      '#options' => [
        '' => $this->t('- None -'),
        'male' => $this->t('Male'),
        'female' => $this->t('Female'),
      ],
      '#default_value' => $submission['gender'] ?? '',
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dga_feedback.admin'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    
    // Process reasons checkboxes
    $reasons = [];
    if (isset($values['reasons']) && is_array($values['reasons'])) {
      $reasons = array_filter($values['reasons'], function($value) {
        return $value !== 0 && $value !== FALSE && $value !== '';
      });
      $reasons = array_values($reasons);
    }
    
    $success = $this->feedbackService->updateSubmission($this->submissionId, [
      'is_useful' => $values['is_useful'],
      'reasons' => $reasons,
      'feedback' => $values['feedback'],
      'gender' => $values['gender'] ?: NULL,
    ]);

    if ($success) {
      $this->messenger()->addMessage($this->t('Feedback submission updated successfully.'));
      $form_state->setRedirect('dga_feedback.admin');
    } else {
      $this->messenger()->addError($this->t('Failed to update feedback submission.'));
    }
  }

}

