(function (Drupal, once) {
  "use strict";

  Drupal.behaviors.feedbackWidget = {
    attach(context, settings) {
      once("feedbackWidget", "[data-feedback-widget]", context).forEach((root) => {
        // Check if already initialized
        if (root.hasAttribute('data-fw-initialized')) {
          return;
        }
        root.setAttribute('data-fw-initialized', 'true');

        // Elements
        const actions = root.querySelector("[data-fw-actions]");
        const btnYes = root.querySelector("[data-fw-yes]");
        const btnNo = root.querySelector("[data-fw-no]");
        const stats = root.querySelector("[data-fw-stats]");
        const btnClose = root.querySelector("[data-fw-close]");
        const questions = root.querySelector("[data-fw-questions]");
        const reasonsWrap = root.querySelector("[data-fw-reasons]");
        const feedbackEl = root.querySelector("[data-fw-feedback]");
        const genderEls = root.querySelectorAll("[data-fw-gender]");
        const submitBtn = root.querySelector("[data-fw-submit]");
        const statusIcon = root.querySelector(".fw-status-icon");
        const statusText = root.querySelector(".fw-status-text");

        // Unified question retrieval (prefer admin-provided values)
        const getQuestionText = () => {
          const ds = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback) ? drupalSettings.dgaFeedback : {};
          const fromSettings = (ds && typeof ds.questionText === 'string') ? ds.questionText.trim() : '';
          const fromAttr = statusText ? (statusText.getAttribute('data-original-question') || '').trim() : '';
          const fromDom = statusText ? (statusText.textContent || '').trim() : '';
          return fromAttr || fromSettings || fromDom || '';
        };

        // CRITICAL: Store the original translated question text
        // But first, check if current text is an error message and fix it
        if (statusText) {
          const currentText = statusText.textContent || '';
          // Check if current text is an error message (common patterns)
          const isErrorMessage = currentText.includes('يرجى') ||
                                currentText.includes('Please') ||
                                currentText.includes('error') ||
                                currentText.includes('خطأ') ||
                                currentText.includes('أولاً') ||
                                currentText.includes('select') ||
                                currentText.includes('اختيار');

          if (isErrorMessage) {
            // It's an error message - get question from data attribute or use default
            const questionFromData = statusText.getAttribute('data-original-question');
            if (questionFromData && questionFromData.trim() !== '' && !questionFromData.includes('يرجى') && !questionFromData.includes('Please select')) {
              statusText.textContent = questionFromData;
            } else {
              // Use default question
              const defaultQuestion = getQuestionText();
              statusText.textContent = defaultQuestion;
              statusText.setAttribute('data-original-question', defaultQuestion);
            }
          } else {
            // It's a valid question - store it
            if (!statusText.getAttribute('data-original-question')) {
          statusText.setAttribute('data-original-question', statusText.textContent || getQuestionText());
            }
          }
        }

        // Get initial data from attributes
        const statsTuple = (root.getAttribute("data-initial-stats") || "").split("|");
        const initialYesPct = statsTuple[0] || "0";
        const initialCount = statsTuple[1] || "0";
        const url = root.getAttribute("data-url") || window.location.pathname;
        const entityType = root.getAttribute("data-entity-type") || "";
        const entityId = root.getAttribute("data-entity-id") || "";

        const settingsObj = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback) ? drupalSettings.dgaFeedback : {};
        const validationMessages = {
          yesNo: root.getAttribute('data-validation-yes-no') || settingsObj.validationYesNo || '',
          reasonRequired: root.getAttribute('data-validation-reason-required') || '',
          reasonInvalid: root.getAttribute('data-validation-reason-invalid') || root.getAttribute('data-validation-reason-required') || '',
          feedbackRequired: root.getAttribute('data-validation-feedback-required') || '',
          genderRequired: root.getAttribute('data-validation-gender-required') || '',
          submissionFailed: root.getAttribute('data-validation-submission-failed') || settingsObj.errorMessage || '',
          unknownError: root.getAttribute('data-validation-unknown-error') || settingsObj.unknownMessage || '',
        };
        const buttonSubmittingText = root.getAttribute('data-button-submitting') || settingsObj.buttonSubmittingText || (submitBtn ? (submitBtn.textContent || '').trim() : '');
        const refreshDelay = typeof settingsObj.refreshDelay === 'number' ? settingsObj.refreshDelay : 3000;
        const refreshBlockUrl = settingsObj.refreshBlockUrl || '/dga-feedback/refresh-block';

        // Get reasons from data attributes (pipe-separated)
        const reasonsYesRaw = root.getAttribute("data-reasons-yes") || "";
        const reasonsNoRaw = root.getAttribute("data-reasons-no") || "";
        const OPTIONS = {
          yes: reasonsYesRaw ? reasonsYesRaw.split("|").map(r => r.trim()).filter(Boolean) : [],
          no: reasonsNoRaw ? reasonsNoRaw.split("|").map(r => r.trim()).filter(Boolean) : [],
        };

        // Update stats display
        const statsTemplate = stats ? (stats.getAttribute('data-template') || "@percentage% of users said Yes from @count Feedbacks") : "@percentage% of users said Yes from @count Feedbacks";

        function updateStatsText(percentage, count) {
          if (!stats) {
            return;
          }
          const template = stats.getAttribute('data-template') || statsTemplate;
          stats.textContent = template
            .replace("@percentage", percentage)
            .replace("@count", count);
        }

        // Initial stats display
        updateStatsText(initialYesPct, initialCount);

        // Render reasons checkboxes
        function renderReasons(container, list, selected) {
          if (!container) return;
          container.innerHTML = "";
          list.forEach((label, i) => {
            const id = `fw-reason-${i}-${label.replace(/\s+/g, "-").toLowerCase()}`;
            const wrapper = document.createElement("label");
            wrapper.className = "fw-checkbox";
            wrapper.setAttribute("for", id);

            const input = document.createElement("input");
            input.type = "checkbox";
            input.id = id;
            input.value = label;
            input.checked = selected.includes(label);
            input.addEventListener("change", () => {
              const idx = selected.indexOf(label);
              if (idx > -1) {
                selected.splice(idx, 1);
              } else {
                selected.push(label);
              }
            });

            const span = document.createElement("span");
            span.textContent = label;

            wrapper.appendChild(input);
            wrapper.appendChild(span);
            container.appendChild(wrapper);
          });
        }

        // State
        const state = {
          isUseful: null, // "yes" | "no" | null
          reasons: [],
          feedback: "",
          gender: "",
          open: false,
          submitted: false,
        };

        function setOpen(open) {
          state.open = open;
          // CRITICAL: Don't hide questions if there's a validation error - user needs to see and fix errors
          const widgetState = root.getAttribute('data-fw-state');
          const hasTempError = statusText && statusText.hasAttribute('data-temp-error');

          if (questions) {
            // Keep form open if:
            // 1. Explicitly opening (open === true)
            // 2. There's a validation error (hasTempError or widgetState === 'error')
            // Only hide if closing AND no errors AND not in error state
            const shouldHide = !open && !hasTempError && widgetState !== 'error' && !state.submitted;
            questions.hidden = shouldHide;
          }
          if (btnClose) {
            // Hide close button if form is closed, submitted, or has errors
            btnClose.hidden = !open || state.submitted || hasTempError || widgetState === 'error';
          }
          // Show stats only when closed or after submission
          if (stats) stats.hidden = open && !state.submitted ? true : false;
          // IMPORTANT: Don't clear data-fw-useful attribute when closing
          // It needs to persist for submission to work
        }

        function setSubmitted(submitted) {
          // CRITICAL: Check the new state machine first - don't allow old state to override
          const widgetState = widget.getAttribute('data-fw-state');
          if (widgetState === 'success' && !submitted) {
            // State machine says success, but old function wants to reset - respect state machine
            return;
          }
          if (widgetState === 'submitting' || widgetState === 'validating') {
            // Don't interfere with active submission
            return;
          }

          state.submitted = submitted;
          if (actions) actions.hidden = submitted;
          if (questions) questions.hidden = true;
          if (btnClose) btnClose.hidden = true;
          if (stats) stats.hidden = false;
          if (statusText) {
            if (submitted) {
              // Only show success if state machine confirms
              const widgetStateCheck = widget.getAttribute('data-fw-state');
              if (widgetStateCheck !== 'success') {
                // State machine doesn't confirm success, don't show success message
                const originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || getQuestionText();
                statusText.textContent = originalQuestion;
                return;
              }
              const successText = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback && drupalSettings.dgaFeedback.submittedSuccessText)
                ? drupalSettings.dgaFeedback.submittedSuccessText
                : (typeof Drupal !== 'undefined' ? Drupal.t('Your feedback is submitted!') : 'Your feedback is submitted!');
              statusText.textContent = successText;
            } else {
              // Get the original question text from the element (already translated from server)
              const originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || getQuestionText();
              statusText.textContent = originalQuestion;
            }
          }
          if (statusIcon) {
            statusIcon.hidden = !submitted;
          }
        }

        function mountReasons(kind) {
          state.reasons = []; // reset when toggling yes/no
          renderReasons(reasonsWrap, OPTIONS[kind], state.reasons);
        }

        // Events - ALL AJAX, no form submission
        if (btnYes) {
          btnYes.addEventListener("click", (e) => {
            e.preventDefault();
            e.stopPropagation();
            // CRITICAL: Reset state to idle when opening form (fixes anonymous user caching issue)
            root.setAttribute('data-fw-state', 'idle');
            state.isUseful = "yes";
            state.submitted = false; // Ensure submitted state is false
            state.open = true; // CRITICAL: Set state.open to true
            root.setAttribute('data-fw-useful', 'yes'); // Store in widget for global handler
            mountReasons("yes");
            setOpen(true);
            // Ensure status text shows question, not success/error message
            if (statusText) {
              // Remove temp error flag
              statusText.removeAttribute('data-temp-error');
              let originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || getQuestionText();
              // Check if it's an error message and fix it (inline check since isErrorMessage is defined later)
              const isError = originalQuestion.includes('يرجى') ||
                             originalQuestion.includes('Please') ||
                             originalQuestion.includes('error') ||
                             originalQuestion.includes('خطأ') ||
                             originalQuestion.includes('أولاً') ||
                             originalQuestion.includes('select') ||
                             originalQuestion.includes('اختيار');
              if (isError) {
                // Get default question
                const defaultQuestion = getQuestionText();
                statusText.textContent = defaultQuestion;
                statusText.setAttribute('data-original-question', defaultQuestion);
              } else {
                statusText.textContent = originalQuestion;
              }
              statusText.style.color = '';
              statusText.style.fontWeight = '';
            }
            if (statusIcon) {
              statusIcon.hidden = true;
            }
            // Clear any error styling when form opens
            if (feedbackEl) {
              feedbackEl.classList.remove('error');
              feedbackEl.removeAttribute('required');
            }
            const reasonsWrap = root.querySelector('[data-fw-reasons]');
            if (reasonsWrap) {
              reasonsWrap.classList.remove('error');
            }
            const genderWrap = root.querySelector('.fw-gender');
            if (genderWrap) {
              genderWrap.classList.remove('error');
            }
            const actions = root.querySelector('[data-fw-actions]');
            if (actions) {
              actions.classList.remove('error');
              // CRITICAL: Ensure actions are visible when form opens
              actions.hidden = false;
            }
          });
        }

        if (btnNo) {
          btnNo.addEventListener("click", (e) => {
            e.preventDefault();
            e.stopPropagation();
            // CRITICAL: Reset state to idle when opening form (fixes anonymous user caching issue)
            root.setAttribute('data-fw-state', 'idle');
            state.isUseful = "no";
            state.submitted = false; // Ensure submitted state is false
            state.open = true; // CRITICAL: Set state.open to true
            root.setAttribute('data-fw-useful', 'no'); // Store in widget for global handler
            mountReasons("no");
            setOpen(true);
            // Ensure status text shows question, not success/error message
            if (statusText) {
              // Remove temp error flag
              statusText.removeAttribute('data-temp-error');
              let originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || getQuestionText();
              // Check if it's an error message and fix it (inline check since isErrorMessage is defined later)
              const isError = originalQuestion.includes('يرجى') ||
                             originalQuestion.includes('Please') ||
                             originalQuestion.includes('error') ||
                             originalQuestion.includes('خطأ') ||
                             originalQuestion.includes('أولاً') ||
                             originalQuestion.includes('select') ||
                             originalQuestion.includes('اختيار');
              if (isError) {
                // Get default question
                const defaultQuestion = getQuestionText();
                statusText.textContent = defaultQuestion;
                statusText.setAttribute('data-original-question', defaultQuestion);
              } else {
                statusText.textContent = originalQuestion;
              }
              statusText.style.color = '';
              statusText.style.fontWeight = '';
            }
            if (statusIcon) {
              statusIcon.hidden = true;
            }
            // Clear any error styling when form opens
            if (feedbackEl) {
              feedbackEl.classList.remove('error');
              feedbackEl.removeAttribute('required');
            }
            const reasonsWrap = root.querySelector('[data-fw-reasons]');
            if (reasonsWrap) {
              reasonsWrap.classList.remove('error');
            }
            const genderWrap = root.querySelector('.fw-gender');
            if (genderWrap) {
              genderWrap.classList.remove('error');
            }
            const actions = root.querySelector('[data-fw-actions]');
            if (actions) {
              actions.classList.remove('error');
              // CRITICAL: Ensure actions are visible when form opens
              actions.hidden = false;
            }
          });
        }

        if (btnClose) {
          btnClose.addEventListener("click", (e) => {
            e.preventDefault();
            e.stopPropagation();
            // Don't clear the is_useful attribute when closing - keep it for submission
            // The attribute should remain set so submission can work
            setOpen(false);
          });
        }

        if (feedbackEl) {
          feedbackEl.addEventListener("input", (e) => {
            state.feedback = e.target.value;
            // Clear error styling when user starts typing
            if (e.target.value.trim().length > 0) {
              feedbackEl.classList.remove('error');
              const feedbackLabel = root.querySelector('label[for="fw-feedback"]');
              if (feedbackLabel) {
                const errorMsg = feedbackLabel.querySelector('.fw-error-message');
                if (errorMsg) errorMsg.remove();
              }
            }
          });
        }

        genderEls.forEach((el) => {
          el.addEventListener("change", (e) => {
            if (e.target.checked) {
              state.gender = e.target.value;
              // Clear error styling when user selects gender
              const genderWrap = root.querySelector('.fw-gender');
              if (genderWrap) {
                genderWrap.classList.remove('error');
                const genderTitle = genderWrap.querySelector('.fw-title-inline');
                if (genderTitle) {
                  const errorMsg = genderTitle.querySelector('.fw-error-message');
                  if (errorMsg) errorMsg.remove();
                }
              }
            }
          });
        });

        // Clear error styling when user checks a reason
        // Also intelligently infer and set Yes/No if not already chosen
        if (reasonsWrap) {
          reasonsWrap.addEventListener("change", (e) => {
            if (e.target.type === 'checkbox') {
              const reasonsWrapEl = e.target.closest('[data-fw-reasons]');
              if (reasonsWrapEl) {
                // 1) Clear error styling
                if (e.target.checked) {
                  reasonsWrapEl.classList.remove('error');
                  const reasonsTitle = reasonsWrapEl.parentElement.querySelector('.fw-title');
                  if (reasonsTitle) {
                    const errorMsg = reasonsTitle.querySelector('.fw-error-message');
                    if (errorMsg) errorMsg.remove();
                  }
                }

                // 2) Auto-set Yes/No when a reason is selected and is_useful is missing
                const widgetEl = reasonsWrapEl.closest('[data-feedback-widget]');
                if (widgetEl) {
                  const currentUseful = widgetEl.getAttribute('data-fw-useful');
                  if (!currentUseful || (currentUseful !== 'yes' && currentUseful !== 'no')) {
                    // Determine the label text for the changed checkbox
                    const labelEl = e.target.closest('label');
                    const labelText = labelEl ? (labelEl.querySelector('span') ? labelEl.querySelector('span').textContent.trim() : (labelEl.textContent || '').trim()) : (e.target.value || '').trim();
                    const yesReasonsRaw = widgetEl.getAttribute('data-reasons-yes') || '';
                    const noReasonsRaw = widgetEl.getAttribute('data-reasons-no') || '';

                    if (labelText) {
                      if (yesReasonsRaw.includes(labelText)) {
                        widgetEl.setAttribute('data-fw-useful', 'yes');
                      } else if (noReasonsRaw.includes(labelText)) {
                        widgetEl.setAttribute('data-fw-useful', 'no');
                      }
                    }
                  }
                }
              }
            }
          });
        }

        // Initial paint - CRITICAL: Always start in idle state
        // Reset any stale state from previous page loads (especially for anonymous users with caching)
        widget.setAttribute('data-fw-state', 'idle');
        widget.removeAttribute('data-fw-useful');

        // CRITICAL: Reset state object to ensure clean state
        state.isUseful = null;
        state.reasons = [];
        state.feedback = '';
        state.gender = '';
        state.open = false;
        state.submitted = false;

        // CRITICAL: Always ensure status text shows question, NEVER error/success message
        if (statusText) {
          // Get original question from data attribute (set by server)
          const originalQuestion = statusText.getAttribute('data-original-question');
          if (originalQuestion && originalQuestion.trim() !== '') {
            statusText.textContent = originalQuestion;
          } else {
            // Fallback: use current text if it looks like a question, otherwise use default
            const currentText = statusText.textContent || '';
            // Check if current text is an error message (contains common error keywords)
            const isErrorMessage = currentText.includes('يرجى') ||
                                  currentText.includes('Please') ||
                                  currentText.includes('error') ||
                                  currentText.includes('خطأ');
            if (isErrorMessage) {
              // It's an error message, replace with default question
              statusText.textContent = getQuestionText();
            } else if (currentText.trim() === '') {
              // Empty, use default
              statusText.textContent = getQuestionText();
            }
            // Store as original for future reference
            statusText.setAttribute('data-original-question', statusText.textContent);
          }
          statusText.style.color = '';
          statusText.style.fontWeight = '';
        }

        // CRITICAL: Always hide status icon on initialization
        if (statusIcon) {
          statusIcon.hidden = true;
        }

        // CRITICAL: Always show actions (Yes/No buttons) on initialization
        if (actions) {
          actions.hidden = false;
        }

        // CRITICAL: Always hide questions form on initialization
        if (questions) {
          questions.hidden = true;
        }

        // CRITICAL: Always hide close button on initialization
        if (btnClose) {
          btnClose.hidden = true;
        }

        setOpen(false);
        setSubmitted(false);
      });
    },
  };

  // GLOBAL HANDLER - Direct button handler that works independently (like dga_rating)
  function attachGlobalSubmitHandler() {
    const submitButtons = document.querySelectorAll('[data-fw-submit]');

    submitButtons.forEach(function(btn) {
      // Skip if already attached or if button doesn't exist
      if (!btn || btn.hasAttribute('data-global-handler-attached')) {
        return;
      }

      // Ensure button is type="button" to prevent form submission
      if (btn.type !== 'button') {
        btn.type = 'button';
      }

      btn.setAttribute('data-global-handler-attached', 'true');

      btn.onclick = function(e) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();

        // Find widget root
        const widget = btn.closest('[data-feedback-widget]');
        if (!widget) {
          console.error('DGA Feedback: Widget not found');
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          return false;
        }

        // STATE MACHINE: Define explicit states
        // idle -> validating -> submitting -> (success | error) -> idle
        const STATE = {
          IDLE: 'idle',
          VALIDATING: 'validating',
          SUBMITTING: 'submitting',
          SUCCESS: 'success',
          ERROR: 'error'
        };

        // Get current state from widget (default to IDLE)
        let currentState = widget.getAttribute('data-fw-state') || STATE.IDLE;

        // CRITICAL: Prevent submission if already in submitting/success state
        if (currentState === STATE.SUBMITTING || currentState === STATE.SUCCESS) {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          return false;
        }

        // Get validation messages and settings from widget data attributes
        const settingsObj = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback) ? drupalSettings.dgaFeedback : {};
        const validationMessages = {
          yesNo: widget.getAttribute('data-validation-yes-no') || settingsObj.validationYesNo || 'Please select Yes or No first.',
          reasonRequired: widget.getAttribute('data-validation-reason-required') || 'Please select at least one reason',
          reasonInvalid: widget.getAttribute('data-validation-reason-invalid') || widget.getAttribute('data-validation-reason-required') || 'At least one valid reason must be selected.',
          feedbackRequired: widget.getAttribute('data-validation-feedback-required') || 'Please provide feedback text',
          genderRequired: widget.getAttribute('data-validation-gender-required') || 'Please select your gender',
          submissionFailed: widget.getAttribute('data-validation-submission-failed') || settingsObj.errorMessage || 'Submission failed. Please try again.',
          unknownError: widget.getAttribute('data-validation-unknown-error') || settingsObj.unknownMessage || 'Unknown error',
        };
        const buttonSubmittingText = widget.getAttribute('data-button-submitting') || settingsObj.buttonSubmittingText || 'Submitting...';
        const refreshDelay = typeof settingsObj.refreshDelay === 'number' ? settingsObj.refreshDelay : 3000;
        const refreshBlockUrl = settingsObj.refreshBlockUrl || '/dga-feedback/refresh-block';

        // Store original button text for potential restoration
        const originalText = btn.textContent;

        // CRITICAL: Clear any previous success/error state when starting new submission
        if (currentState === STATE.SUCCESS || currentState === STATE.ERROR) {
          // Reset widget to idle state
          widget.setAttribute('data-fw-state', STATE.IDLE);
          currentState = STATE.IDLE;

          // Clear success message and restore question
          const statusText = widget.querySelector('.fw-status-text');
          const statusIcon = widget.querySelector('.fw-status-icon');
          const questions = widget.querySelector('[data-fw-questions]');
          const actions = widget.querySelector('[data-fw-actions]');

          if (statusText) {
            const originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || '';
            statusText.textContent = originalQuestion;
            statusText.style.color = '';
            statusText.style.fontWeight = '';
          }
          if (statusIcon) {
            statusIcon.hidden = true;
          }
          if (questions) {
            questions.hidden = false;
          }
          if (actions) {
            actions.hidden = false;
          }
        }

        // CRITICAL: Ensure button is type="button" to prevent form submission
        if (btn.type !== 'button') {
          btn.type = 'button';
        }

        // Transition to VALIDATING state
        widget.setAttribute('data-fw-state', STATE.VALIDATING);
        currentState = STATE.VALIDATING;

        // STEP 1: Validate is_useful (Yes/No selection)
        // First, try to get from data attribute (set when Yes/No button is clicked)
        // This attribute should persist even after closing the form
        // CRITICAL: Also check state object in case attribute was lost
        let isUseful = widget.getAttribute('data-fw-useful');

        // If attribute is missing, try to infer from form state
        if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
          // Check if questions form is open - if so, we can infer from visible reasons
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && !questions.hidden) {
            // Form is open, check which reasons are visible
            const reasonsWrap = widget.querySelector('[data-fw-reasons]');
            if (reasonsWrap) {
              const firstCheckbox = reasonsWrap.querySelector('input[type="checkbox"]');
              if (firstCheckbox) {
                const firstLabel = firstCheckbox.closest('label');
                if (firstLabel) {
                  const labelText = firstLabel.textContent.trim();
                  const yesReasons = widget.getAttribute('data-reasons-yes') || '';
                  const noReasons = widget.getAttribute('data-reasons-no') || '';

                  if (yesReasons.includes(labelText)) {
                    isUseful = 'yes';
                    widget.setAttribute('data-fw-useful', 'yes');
                  } else if (noReasons.includes(labelText)) {
                    isUseful = 'no';
                    widget.setAttribute('data-fw-useful', 'no');
                  }
                }
              }
            }
          }
        }

        // If attribute not found, try to infer from the current state
        // This can happen if the global handler runs before behaviors, or if form was closed
        if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
          // Method 1: Check if questions are visible (form is open) and infer from reasons
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && !questions.hidden) {
            const firstReason = widget.querySelector('[data-fw-reasons] label:first-child span');
            if (firstReason) {
              const firstReasonText = firstReason.textContent.trim();
              const yesReasons = widget.getAttribute('data-reasons-yes') || '';
              const noReasons = widget.getAttribute('data-reasons-no') || '';

              if (yesReasons.includes(firstReasonText)) {
                isUseful = 'yes';
                widget.setAttribute('data-fw-useful', 'yes');
              } else if (noReasons.includes(firstReasonText)) {
                isUseful = 'no';
                widget.setAttribute('data-fw-useful', 'no');
              }
            }
          }

          // Method 2: Check which checkboxes exist even if hidden (Yes reasons vs No reasons)
          if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
            const allReasonLabels = widget.querySelectorAll('[data-fw-reasons] label span');
            if (allReasonLabels.length > 0) {
              const firstLabelText = allReasonLabels[0].textContent.trim();
              const yesReasons = widget.getAttribute('data-reasons-yes') || '';
              const noReasons = widget.getAttribute('data-reasons-no') || '';

              if (yesReasons.includes(firstLabelText)) {
                isUseful = 'yes';
                widget.setAttribute('data-fw-useful', 'yes');
              } else if (noReasons.includes(firstLabelText)) {
                isUseful = 'no';
                widget.setAttribute('data-fw-useful', 'no');
              }
            }
          }

          // Method 3: Check if actions are hidden (means Yes/No was clicked, form might be closed now)
          if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
            const actions = widget.querySelector('[data-fw-actions]');
            if (actions && actions.hidden) {
              // Actions are hidden, which means Yes/No was clicked at some point
              // Check reasons to determine which (even if questions are hidden now)
              const firstReason = widget.querySelector('[data-fw-reasons] label:first-child span');
              if (firstReason) {
                const firstReasonText = firstReason.textContent.trim();
                const yesReasons = widget.getAttribute('data-reasons-yes') || '';
                const noReasons = widget.getAttribute('data-reasons-no') || '';

                if (yesReasons.includes(firstReasonText)) {
                  isUseful = 'yes';
                  widget.setAttribute('data-fw-useful', 'yes');
                } else if (noReasons.includes(firstReasonText)) {
                  isUseful = 'no';
                  widget.setAttribute('data-fw-useful', 'no');
                }
              }
            }
          }
        }

        // STEP 1: Final validation - if still not found, don't submit
        if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
          // CRITICAL: Stop ALL event propagation immediately - MUST BE FIRST
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();

          // Transition to ERROR state - validation failed
          widget.setAttribute('data-fw-state', STATE.ERROR);

          // CRITICAL: Ensure button cannot submit
          btn.type = 'button';
          btn.disabled = false;

          // Show Yes/No buttons and focus on them
          const actions = widget.querySelector('[data-fw-actions]');
          if (actions) {
            actions.hidden = false;
            actions.classList.add('error');
            // Scroll to actions
            actions.scrollIntoView({ behavior: 'smooth', block: 'center' });
          }

          // Show error message in status text temporarily, but mark it as error
          const statusText = widget.querySelector('.fw-status-text');
          if (statusText) {
            // Store original question before showing error
            if (!statusText.getAttribute('data-original-question') || isErrorMessage(statusText.getAttribute('data-original-question'))) {
              // Try to get from current text if it's a valid question
              const currentText = statusText.textContent || '';
              if (!isErrorMessage(currentText) && currentText.trim() !== '') {
                statusText.setAttribute('data-original-question', currentText);
              }
            }
            statusText.textContent = validationMessages.yesNo;
            statusText.style.color = '#dc3545';
            statusText.style.fontWeight = 'bold';
            // Mark as temporary error (will be reset when form opens)
            statusText.setAttribute('data-temp-error', 'true');
          }

          // CRITICAL: Keep questions form visible and ensure Yes/No buttons are visible
          // Don't close the form on validation error - user needs to see and fix errors
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions) {
            questions.hidden = false; // Keep form open
          }

          // CRITICAL: Ensure actions (Yes/No buttons) are visible so user can try again
          if (actions) {
            actions.hidden = false; // Show Yes/No buttons
          }

          // CRITICAL: Don't remove data-fw-useful attribute on error - keep it so form can reopen
          // The attribute should persist so user can fix errors and resubmit

          return false;
        }

        // STEP 2: Validate reasons (at least one checkbox must be selected)
        // IMPORTANT: Check ALL checkboxes, even if form is closed (hidden)
        const reasonCheckboxes = widget.querySelectorAll('[data-fw-reasons] input[type="checkbox"]:checked');
        const reasons = Array.from(reasonCheckboxes).map(cb => cb.value);

        if (reasons.length === 0) {
          // CRITICAL: Stop ALL event propagation immediately - MUST BE FIRST
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();

          // Transition to ERROR state - validation failed
          widget.setAttribute('data-fw-state', STATE.ERROR);

          // CRITICAL: Ensure button cannot submit
          btn.type = 'button';
          btn.disabled = false; // Re-enable in case it was disabled

          // CRITICAL: Show questions form if hidden (user might have closed it)
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && questions.hidden) {
            questions.hidden = false;
          }

          const reasonsWrap = widget.querySelector('[data-fw-reasons]');
          if (reasonsWrap) {
            reasonsWrap.classList.add('error');
            reasonsWrap.setAttribute('aria-invalid', 'true');
            // Scroll to reasons
            reasonsWrap.scrollIntoView({ behavior: 'smooth', block: 'center' });
            // Focus on first checkbox
            const firstCheckbox = reasonsWrap.querySelector('input[type="checkbox"]');
            if (firstCheckbox) {
              firstCheckbox.focus();
              firstCheckbox.setAttribute('aria-invalid', 'true');
            }
            // Show error message in reasons title (accessibility)
            const reasonsTitle = reasonsWrap.parentElement ? reasonsWrap.parentElement.querySelector('.fw-title') : null;
            if (reasonsTitle) {
              const errorSpan = document.createElement('span');
              errorSpan.textContent = ' - ' + validationMessages.reasonRequired;
              errorSpan.className = 'fw-error-message';
              errorSpan.id = 'fw-reasons-error';
              errorSpan.setAttribute('role', 'alert');
              errorSpan.setAttribute('aria-live', 'polite');
              // Remove existing error if any
              const existingError = reasonsTitle.querySelector('.fw-error-message');
              if (existingError) existingError.remove();
              reasonsTitle.appendChild(errorSpan);
              // Update aria-describedby
              reasonsWrap.setAttribute('aria-describedby', 'fw-reasons-error');
            }
          }

          // CRITICAL: Return false to prevent any further execution
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          return false;
        }

        // STEP 3: Validate feedback text (must not be empty)
        const feedbackEl = widget.querySelector('[data-fw-feedback]');
        const feedback = feedbackEl ? feedbackEl.value.trim() : '';

        if (!feedback || feedback.length === 0) {
          // CRITICAL: Stop ALL event propagation immediately - MUST BE FIRST
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();

          // Transition to ERROR state - validation failed
          widget.setAttribute('data-fw-state', STATE.ERROR);

          // CRITICAL: Ensure button cannot submit
          btn.type = 'button';
          btn.disabled = false;

          // CRITICAL: Show questions form if hidden
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && questions.hidden) {
            questions.hidden = false;
          }

          if (feedbackEl) {
            feedbackEl.classList.add('error');
            feedbackEl.setAttribute('aria-invalid', 'true');
            feedbackEl.focus();
            feedbackEl.scrollIntoView({ behavior: 'smooth', block: 'center' });
            // Show error message (accessibility)
            const feedbackError = widget.querySelector('#fw-feedback-error');
            if (feedbackError) {
              feedbackError.textContent = validationMessages.feedbackRequired;
              feedbackError.classList.remove('sr-only');
              feedbackError.setAttribute('role', 'alert');
            } else {
              // Fallback: show in label
            const feedbackLabel = widget.querySelector('label[for="fw-feedback"]');
            if (feedbackLabel) {
              const errorSpan = document.createElement('span');
              errorSpan.textContent = ' - ' + validationMessages.feedbackRequired;
              errorSpan.className = 'fw-error-message';
                errorSpan.id = 'fw-feedback-error';
                errorSpan.setAttribute('role', 'alert');
                errorSpan.setAttribute('aria-live', 'polite');
              // Remove existing error if any
              const existingError = feedbackLabel.querySelector('.fw-error-message');
              if (existingError) existingError.remove();
              feedbackLabel.appendChild(errorSpan);
            }
            }
            // Update aria-describedby
            feedbackEl.setAttribute('aria-describedby', 'fw-feedback-error');
          }

          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          return false;
        }

        // STEP 4: Validate gender (optional field - accepts male, female, prefer_not_to_say, or empty)
        const genderRadio = widget.querySelector('[data-fw-gender]:checked');
        const gender = genderRadio ? genderRadio.value : '';

        // Gender is optional, but if provided, must be a valid value
        if (gender && gender !== 'male' && gender !== 'female' && gender !== 'prefer_not_to_say') {
          // CRITICAL: Stop ALL event propagation immediately - MUST BE FIRST
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();

          // Transition to ERROR state - validation failed
          widget.setAttribute('data-fw-state', STATE.ERROR);

          // CRITICAL: Ensure button cannot submit
          btn.type = 'button';
          btn.disabled = false;

          // CRITICAL: Show questions form if hidden
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && questions.hidden) {
            questions.hidden = false;
          }

          const genderWrap = widget.querySelector('.fw-gender');
          if (genderWrap) {
            genderWrap.classList.add('error');
            genderWrap.scrollIntoView({ behavior: 'smooth', block: 'center' });
            // Focus on first gender radio
            const genderEls = widget.querySelectorAll('[data-fw-gender]');
            if (genderEls.length > 0) {
              genderEls[0].focus();
              genderEls.forEach(function(el) {
                el.setAttribute('aria-invalid', 'true');
              });
            }
            // Show error message (accessibility)
            const genderError = widget.querySelector('#fw-gender-error');
            if (genderError) {
              genderError.textContent = validationMessages.unknownError;
              genderError.classList.remove('sr-only');
              genderError.setAttribute('role', 'alert');
            } else {
              // Fallback: show in title
            const genderTitle = genderWrap.querySelector('.fw-title-inline');
            if (genderTitle) {
              const errorSpan = document.createElement('span');
                errorSpan.textContent = ' - ' + validationMessages.unknownError;
              errorSpan.className = 'fw-error-message';
                errorSpan.id = 'fw-gender-error';
                errorSpan.setAttribute('role', 'alert');
                errorSpan.setAttribute('aria-live', 'polite');
              // Remove existing error if any
              const existingError = genderTitle.querySelector('.fw-error-message');
              if (existingError) existingError.remove();
              genderTitle.appendChild(errorSpan);
            }
            }
            // Update aria-describedby on all gender radios
            genderEls.forEach(function(el) {
              el.setAttribute('aria-describedby', 'fw-gender-error');
            });
          }

          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          return false;
        }

        // ALL VALIDATIONS PASSED - Clear error styles

        // Clear any error styles and messages before submission
        const reasonsWrap = widget.querySelector('[data-fw-reasons]');
        if (reasonsWrap) {
          reasonsWrap.classList.remove('error');
          reasonsWrap.removeAttribute('aria-invalid');
          reasonsWrap.removeAttribute('aria-describedby');
          const reasonsTitle = reasonsWrap.parentElement ? reasonsWrap.parentElement.querySelector('.fw-title') : null;
          if (reasonsTitle) {
            const errorMsg = reasonsTitle.querySelector('.fw-error-message');
            if (errorMsg) errorMsg.remove();
          }
          // Clear aria-invalid from checkboxes
          const checkboxes = reasonsWrap.querySelectorAll('input[type="checkbox"]');
          checkboxes.forEach(function(cb) {
            cb.removeAttribute('aria-invalid');
          });
        }
        if (feedbackEl) {
          feedbackEl.classList.remove('error');
          feedbackEl.setAttribute('aria-invalid', 'false');
          const feedbackError = widget.querySelector('#fw-feedback-error');
          if (feedbackError) {
            feedbackError.textContent = '';
            feedbackError.classList.add('sr-only');
          }
          const feedbackLabel = widget.querySelector('label[for="fw-feedback"]');
          if (feedbackLabel) {
            const errorMsg = feedbackLabel.querySelector('.fw-error-message');
            if (errorMsg) errorMsg.remove();
          }
        }
        const genderWrap = widget.querySelector('.fw-gender');
        if (genderWrap) {
          genderWrap.classList.remove('error');
          const genderError = widget.querySelector('#fw-gender-error');
          if (genderError) {
            genderError.textContent = '';
            genderError.classList.add('sr-only');
          }
          const genderTitle = genderWrap.querySelector('.fw-title-inline');
          if (genderTitle) {
            const errorMsg = genderTitle.querySelector('.fw-error-message');
            if (errorMsg) errorMsg.remove();
          }
          // Clear aria-invalid from radios
          const genderEls = widget.querySelectorAll('[data-fw-gender]');
          genderEls.forEach(function(el) {
            el.removeAttribute('aria-invalid');
            el.removeAttribute('aria-describedby');
          });
        }
        const actions = widget.querySelector('[data-fw-actions]');
        if (actions) {
          actions.classList.remove('error');
        }

        // Get URL
        let url = widget.getAttribute('data-url') || window.location.pathname;
        url = url.trim().replace(/\/+$/, '') || '/';
        if (url.match(/^\/[a-z]{2}\//)) {
          url = url.replace(/^\/[a-z]{2}/, '');
          url = url || '/';
        }

        // Get entity info
        const entityType = widget.getAttribute('data-entity-type') || 'node';
        const entityId = widget.getAttribute('data-entity-id') || '0';

        // Validate is_useful one more time before proceeding
        if (!isUseful || (isUseful !== 'yes' && isUseful !== 'no')) {
          // Transition to ERROR state
          widget.setAttribute('data-fw-state', STATE.ERROR);
          btn.removeAttribute('data-submitting');
          btn.disabled = false;
          btn.textContent = originalText;
          return false;
        }

        // ALL VALIDATIONS PASSED - Transition to SUBMITTING state
        widget.setAttribute('data-fw-state', STATE.SUBMITTING);

        // Disable button - only after all validations pass
        btn.setAttribute('data-submitting', 'true');
        btn.disabled = true;
        btn.type = 'button'; // Ensure button type is set (prevents form submission)
        btn.textContent = buttonSubmittingText;

        // Prepare payload - ALL AJAX, NO FORM SUBMISSION
        const payload = {
          is_useful: isUseful, // This should be 'yes' or 'no' from data-fw-useful attribute
          reasons: reasons,
          feedback: feedback || '',
          gender: gender || '',
          url: url || '/',
          entity_type: entityType,
          entity_id: parseInt(entityId) || 0
        };

        // CRITICAL: Send request using synchronous XMLHttpRequest (AJAX only, like dga_rating)
        // This is AJAX - no page reload, no form submission
        const xhr = new XMLHttpRequest();
        // Use language-aware URL from drupalSettings when available
        const submitUrl = (window.drupalSettings && window.drupalSettings.dgaFeedback && window.drupalSettings.dgaFeedback.submitUrl)
          ? window.drupalSettings.dgaFeedback.submitUrl
          : '/dga-feedback/submit';
        xhr.open('POST', submitUrl, false); // false = synchronous, AJAX
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
        xhr.setRequestHeader('Content-Type', 'application/json');

        try {
          xhr.send(JSON.stringify(payload));

          let data;
          try {
            data = JSON.parse(xhr.responseText);
          } catch (e) {
            btn.removeAttribute('data-submitting');
            btn.disabled = false;
            btn.textContent = originalText;
            const statusText = widget.querySelector('.fw-status-text');
            if (statusText) {
            statusText.textContent = validationMessages.submissionFailed;
              statusText.style.color = '#dc3545';
            }
            return false;
          }

          // Validate response - ONLY show success if backend confirms success
          // Check for error responses first
          if (xhr.status !== 200 || !data || data.success !== true || !data.feedback_id) {
            // Transition to ERROR state - backend rejected submission
            widget.setAttribute('data-fw-state', STATE.ERROR);

            // Handle error response
            btn.removeAttribute('data-submitting');
            btn.disabled = false;
            btn.textContent = originalText;

            // Show error message
            const errorMsg = (data && data.message) ? data.message : validationMessages.submissionFailed;
            const statusText = widget.querySelector('.fw-status-text');
            if (statusText) {
              statusText.textContent = errorMsg;
              statusText.style.color = '#dc3545';
              statusText.style.fontWeight = 'bold';
            }

            // Ensure questions form is visible to show validation errors
            const questions = widget.querySelector('[data-fw-questions]');
            if (questions && questions.hidden) {
              questions.hidden = false;
            }

            // If backend returned specific validation errors, show them
            if (data && data.message) {
              // Check if it's a reason validation error
              if (data.message.includes('reason') || data.message.includes('سبب')) {
                const reasonsWrap = widget.querySelector('[data-fw-reasons]');
                if (reasonsWrap) {
                  reasonsWrap.classList.add('error');
                  const reasonsTitle = reasonsWrap.parentElement ? reasonsWrap.parentElement.querySelector('.fw-title') : null;
                  if (reasonsTitle) {
                    const errorSpan = document.createElement('span');
                    errorSpan.textContent = ' - ' + data.message;
                    errorSpan.className = 'fw-error-message';
                    const existingError = reasonsTitle.querySelector('.fw-error-message');
                    if (existingError) existingError.remove();
                    reasonsTitle.appendChild(errorSpan);
                  }
                }
              }
              // Check if it's a feedback validation error
              if (data.message.includes('feedback') || data.message.includes('تعليق')) {
                const feedbackEl = widget.querySelector('[data-fw-feedback]');
                if (feedbackEl) {
                  feedbackEl.classList.add('error');
                  const feedbackLabel = widget.querySelector('label[for="fw-feedback"]');
                  if (feedbackLabel) {
                    const errorSpan = document.createElement('span');
                    errorSpan.textContent = ' - ' + data.message;
                    errorSpan.className = 'fw-error-message';
                    const existingError = feedbackLabel.querySelector('.fw-error-message');
                    if (existingError) existingError.remove();
                    feedbackLabel.appendChild(errorSpan);
                  }
                }
              }
            }

            return false;
          }

          // ONLY proceed if we have confirmed success from backend
          // CRITICAL: Double-check we're still in SUBMITTING state (not ERROR)
          const currentStateCheck = widget.getAttribute('data-fw-state');
          if (currentStateCheck === STATE.ERROR) {
            // State was set to ERROR, don't show success
            btn.removeAttribute('data-submitting');
            btn.disabled = false;
            btn.textContent = originalText;
            return false;
          }

          // Transition to SUCCESS state - backend confirmed successful submission
          // This is the ONLY place where SUCCESS state can be set
          widget.setAttribute('data-fw-state', STATE.SUCCESS);

          if (xhr.status === 200 && data && data.success === true && data.feedback_id) {
            // CRITICAL: Clear ALL error styles and messages FIRST (before hiding form)
            // This ensures no error messages persist when success is shown
            const reasonsWrap = widget.querySelector('[data-fw-reasons]');
            if (reasonsWrap) {
              reasonsWrap.classList.remove('error');
              const reasonsTitle = reasonsWrap.parentElement ? reasonsWrap.parentElement.querySelector('.fw-title') : null;
              if (reasonsTitle) {
                const errorMsg = reasonsTitle.querySelector('.fw-error-message');
                if (errorMsg) errorMsg.remove();
              }
            }
            const feedbackEl = widget.querySelector('[data-fw-feedback]');
            if (feedbackEl) {
              feedbackEl.classList.remove('error');
              const feedbackLabel = widget.querySelector('label[for="fw-feedback"]');
              if (feedbackLabel) {
                const errorMsg = feedbackLabel.querySelector('.fw-error-message');
                if (errorMsg) errorMsg.remove();
              }
            }
            const genderWrap = widget.querySelector('.fw-gender');
            if (genderWrap) {
              genderWrap.classList.remove('error');
              const genderTitle = genderWrap.querySelector('.fw-title-inline');
              if (genderTitle) {
                const errorMsg = genderTitle.querySelector('.fw-error-message');
                if (errorMsg) errorMsg.remove();
              }
            }
            const actions = widget.querySelector('[data-fw-actions]');
            if (actions) {
              actions.classList.remove('error');
            }

            // CRITICAL: Now hide questions form AFTER clearing all errors
            const questions = widget.querySelector('[data-fw-questions]');
            if (questions) {
              questions.hidden = true;
            }

            // CRITICAL: Final verification - ensure no error messages exist in DOM
            const remainingErrors = widget.querySelectorAll('.fw-error-message');
            remainingErrors.forEach(function(errorEl) {
              errorEl.remove();
            });

            // Update stats if provided
            if (data.statistics) {
              const newPercentage = Math.round(data.statistics.yes_percentage || 0);
              const newCount = data.statistics.total_count || 0;
              // Update stats display directly
              const stats = widget.querySelector('[data-fw-stats]');
              if (stats) {
                const statsTemplate = stats.getAttribute('data-template') || "@percentage% of users said Yes from @count Feedbacks";
                stats.textContent = statsTemplate
                  .replace("@percentage", newPercentage)
                  .replace("@count", newCount);
              }
            }

            // Update widget UI - show success state
            const btnClose = widget.querySelector('[data-fw-close]');
            const statusIcon = widget.querySelector('.fw-status-icon');
            const statusText = widget.querySelector('.fw-status-text');

            if (actions) actions.hidden = true;
            if (btnClose) btnClose.hidden = true;
            if (statusIcon) statusIcon.hidden = false;
            if (statusText) {
              const successText = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback && drupalSettings.dgaFeedback.submittedSuccessText)
                ? drupalSettings.dgaFeedback.submittedSuccessText
                : (typeof Drupal !== 'undefined' ? Drupal.t('Your feedback is submitted!') : 'Your feedback is submitted!');
              statusText.textContent = successText;
              // Clear any error styling from status text
              statusText.style.color = '';
              statusText.style.fontWeight = '';
            }

            setTimeout(() => {
              let widgetUrl = widget.getAttribute('data-url') || window.location.pathname;
              widgetUrl = widgetUrl.trim().replace(/\/+$/, '') || '/';
              if (widgetUrl.match(/^\/[a-z]{2}\//)) {
                widgetUrl = widgetUrl.replace(/^\/[a-z]{2}/, '');
                widgetUrl = widgetUrl || '/';
              }

              let refreshUrl = refreshBlockUrl + '?url=' + encodeURIComponent(widgetUrl);
              if (entityType) {
                refreshUrl += '&entity_type=' + encodeURIComponent(entityType);
              }
              if (entityId) {
                refreshUrl += '&entity_id=' + encodeURIComponent(entityId);
              }

              fetch(refreshUrl, {
                method: 'GET',
                headers: {
                  'Accept': 'application/json',
                },
              })
                .then(response => response.json())
                .then(refreshData => {
                  if (refreshData.success && refreshData.statistics) {
                    const refreshedPercentage = Math.round(refreshData.statistics.yes_percentage || 0);
                    const refreshedCount = refreshData.statistics.total_count || 0;
                    // Update stats display directly
                    const stats = widget.querySelector('[data-fw-stats]');
                    if (stats) {
                      const statsTemplate = stats.getAttribute('data-template') || "@percentage% of users said Yes from @count Feedbacks";
                      stats.textContent = statsTemplate
                        .replace("@percentage", refreshedPercentage)
                        .replace("@count", refreshedCount);
                    }
                  }

                  // Reset widget state to IDLE after successful submission and refresh
                  widget.setAttribute('data-fw-state', STATE.IDLE);
                  widget.removeAttribute('data-fw-useful');

                  const reasonsWrapEl = widget.querySelector('[data-fw-reasons]');
                  const feedbackEl = widget.querySelector('[data-fw-feedback]');
                  const genderEls = widget.querySelectorAll('[data-fw-gender]');

                  if (reasonsWrapEl) {
                    reasonsWrapEl.innerHTML = '';
                  }
                  if (feedbackEl) {
                    feedbackEl.value = '';
                  }
                  genderEls.forEach((el) => {
                    el.checked = false;
                  });

                  // Get elements from widget
                  const actions = widget.querySelector('[data-fw-actions]');
                  const questions = widget.querySelector('[data-fw-questions]');
                  const btnClose = widget.querySelector('[data-fw-close]');
                  const statusIcon = widget.querySelector('.fw-status-icon');
                  const statusText = widget.querySelector('.fw-status-text');

                  if (actions) actions.hidden = false;
                  if (questions) questions.hidden = true;
                  if (btnClose) btnClose.hidden = true;
                  if (statusIcon) statusIcon.hidden = true;
                  if (statusText) {
                    const originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || '';
                    statusText.textContent = originalQuestion;
                    statusText.style.color = '';
                    statusText.style.fontWeight = '';
                  }

                  // Reset widget to closed state
                  if (questions) questions.hidden = true;
                  if (actions) actions.hidden = false;
                  if (statusIcon) statusIcon.hidden = true;
                })
                .catch(() => {
                  // Reset widget state to IDLE even if refresh fails
                  widget.setAttribute('data-fw-state', STATE.IDLE);
                  widget.removeAttribute('data-fw-useful');
                  const actions = widget.querySelector('[data-fw-actions]');
                  const questions = widget.querySelector('[data-fw-questions]');
                  const btnClose = widget.querySelector('[data-fw-close]');
                  const statusIcon = widget.querySelector('.fw-status-icon');
                  const statusText = widget.querySelector('.fw-status-text');

                  if (actions) actions.hidden = false;
                  if (questions) questions.hidden = true;
                  if (btnClose) btnClose.hidden = true;
                  if (statusIcon) statusIcon.hidden = true;
                  if (statusText) {
                    const originalQuestion = statusText.getAttribute('data-original-question') || statusText.textContent || '';
                    statusText.textContent = originalQuestion;
                    statusText.style.color = '';
                    statusText.style.fontWeight = '';
                  }

                  const feedbackEl = widget.querySelector('[data-fw-feedback]');
                  const genderEls = widget.querySelectorAll('[data-fw-gender]');
                  const reasonsWrap = widget.querySelector('[data-fw-reasons]');

                  if (feedbackEl) {
                    feedbackEl.value = '';
                  }
                  genderEls.forEach((el) => {
                    el.checked = false;
                  });
                  if (reasonsWrap) {
                    reasonsWrap.innerHTML = '';
                  }

                  // Reset widget to closed state
                  if (questions) questions.hidden = true;
                  if (actions) actions.hidden = false;
                  if (statusIcon) statusIcon.hidden = true;
                })
                .finally(() => {
                  btn.removeAttribute('data-submitting');
                  btn.disabled = false;
                  btn.textContent = originalText;
                });
            }, refreshDelay);

            return false;
          }
        } catch (err) {
          // Transition to ERROR state - exception occurred
          widget.setAttribute('data-fw-state', STATE.ERROR);

          // Show error inline
          const statusText = widget.querySelector('.fw-status-text');
          if (statusText) {
            statusText.textContent = validationMessages.submissionFailed;
            statusText.style.color = '#dc3545';
            statusText.style.fontWeight = 'bold';
          }

          // Ensure questions form is visible
          const questions = widget.querySelector('[data-fw-questions]');
          if (questions && questions.hidden) {
            questions.hidden = false;
          }

          btn.removeAttribute('data-submitting');
          btn.disabled = false;
          btn.textContent = originalText;
        }

        // Always remove submitting flag and return false
        btn.removeAttribute('data-submitting');
        return false;
      };
    });
  }

  // GLOBAL HANDLER - Delegated Yes/No click fallback to ensure is_useful is set
  function attachGlobalYesNoHandler() {
    // Attach once to each Yes/No button to set data-fw-useful early
    const yesButtons = document.querySelectorAll('[data-fw-yes]');
    const noButtons = document.querySelectorAll('[data-fw-no]');

    yesButtons.forEach(function(btn) {
      if (!btn || btn.hasAttribute('data-global-yesno-attached')) return;
      btn.setAttribute('data-global-yesno-attached', 'true');
      btn.addEventListener('click', function(e) {
        const widget = btn.closest('[data-feedback-widget]');
        if (!widget) return;
        // Set attribute so validation can read it even if local handlers failed
        widget.setAttribute('data-fw-useful', 'yes');
        // Clear actions error state if any
        const actions = widget.querySelector('[data-fw-actions]');
        if (actions) actions.classList.remove('error');
      }, { capture: true });
    });

    noButtons.forEach(function(btn) {
      if (!btn || btn.hasAttribute('data-global-yesno-attached')) return;
      btn.setAttribute('data-global-yesno-attached', 'true');
      btn.addEventListener('click', function(e) {
        const widget = btn.closest('[data-feedback-widget]');
        if (!widget) return;
        widget.setAttribute('data-fw-useful', 'no');
        const actions = widget.querySelector('[data-fw-actions]');
        if (actions) actions.classList.remove('error');
      }, { capture: true });
    });
  }

  // Helper function to check if text is an error message
  function isErrorMessage(text) {
    if (!text || typeof text !== 'string') return false;
    const errorPatterns = [
      'يرجى', 'Please', 'error', 'خطأ', 'أولاً', 'select', 'اختيار',
      'first', 'أول', 'required', 'مطلوب', 'choose', 'اختر'
    ];
    return errorPatterns.some(function(pattern) {
      return text.includes(pattern);
    });
  }

  // Global helper to get the localized question text outside behavior scope
  function getGlobalQuestionText(statusTextEl) {
    const ds = (typeof drupalSettings !== 'undefined' && drupalSettings.dgaFeedback) ? drupalSettings.dgaFeedback : {};
    const fromSettings = (ds && typeof ds.questionText === 'string') ? ds.questionText.trim() : '';
    const fromAttr = statusTextEl ? (statusTextEl.getAttribute('data-original-question') || '').trim() : '';
    const fromDom = statusTextEl ? (statusTextEl.textContent || '').trim() : '';
    // Prefer non-error values
    const firstValid = (val) => val && !isErrorMessage(val) && val.trim() !== '';
    if (firstValid(fromAttr)) return fromAttr;
    if (firstValid(fromSettings)) return fromSettings;
    if (firstValid(fromDom)) return fromDom;
    return '';
  }

  // Helper function to fix status text if it's an error message
  function fixStatusText(statusText) {
    if (!statusText) return;

    const currentText = statusText.textContent || '';
    const widgetState = statusText.closest('[data-feedback-widget]')?.getAttribute('data-fw-state');

    // Only fix if we're in idle state (not during submission/success)
    if (widgetState && widgetState !== 'idle' && widgetState !== 'error') {
      return; // Don't interfere with active states
    }

    if (isErrorMessage(currentText)) {
      // It's an error message - get question from data attribute
      let originalQuestion = statusText.getAttribute('data-original-question');

      // If data-original-question is also an error, try to get from template default
      if (!originalQuestion || isErrorMessage(originalQuestion)) {
        // Try to get from the template's initial value (might be in HTML comment or elsewhere)
        const widget = statusText.closest('[data-feedback-widget]');
        if (widget) {
          // Check if widget has question in data attributes
          const questionFromWidget = widget.getAttribute('data-question-text');
          if (questionFromWidget && !isErrorMessage(questionFromWidget)) {
            originalQuestion = questionFromWidget;
          }
        }
      }

      // If still no valid question, use default
      if (!originalQuestion || isErrorMessage(originalQuestion)) {
        originalQuestion = getGlobalQuestionText(statusText);
      }

      // Only update if different to avoid infinite loops
      if (statusText.textContent !== originalQuestion) {
        statusText.textContent = originalQuestion;
        statusText.setAttribute('data-original-question', originalQuestion);
        statusText.style.color = '';
        statusText.style.fontWeight = '';
      }
    }
  }

  // CRITICAL: Global widget reset function for anonymous users
  // This ensures widgets always start in the correct state, even with caching
  // MUST run BEFORE Drupal behaviors to fix cached error states
  function resetAllWidgets() {
    const widgets = document.querySelectorAll('[data-feedback-widget]');
    widgets.forEach(function(widget) {
      // Reset state machine
      widget.setAttribute('data-fw-state', 'idle');
      widget.removeAttribute('data-fw-useful');

      // Reset status text to question (never error/success)
      const statusText = widget.querySelector('.fw-status-text');
      if (statusText) {
        // CRITICAL: Always check if current text or stored question is an error message
        const currentText = statusText.textContent || '';
        let storedQuestion = statusText.getAttribute('data-original-question') || '';

        // If stored question is an error, clear it
        if (isErrorMessage(storedQuestion)) {
          statusText.removeAttribute('data-original-question');
          storedQuestion = '';
        }

        // If current text is an error, replace it
        if (isErrorMessage(currentText)) {
          // Try to get valid question from stored value
          if (storedQuestion && !isErrorMessage(storedQuestion) && storedQuestion.trim() !== '') {
            statusText.textContent = storedQuestion;
  } else {
            // Use default question
            const defaultQuestion = getGlobalQuestionText(statusText);
            statusText.textContent = defaultQuestion;
            statusText.setAttribute('data-original-question', defaultQuestion);
          }
        } else {
          // Current text is valid - store it if not already stored
          if (!storedQuestion || storedQuestion.trim() === '') {
            if (currentText.trim() !== '') {
              statusText.setAttribute('data-original-question', currentText);
            } else {
              const defaultQuestion = getGlobalQuestionText(statusText);
              statusText.setAttribute('data-original-question', defaultQuestion);
              statusText.textContent = defaultQuestion;
            }
          }
        }

        // Remove temp error flag
        statusText.removeAttribute('data-temp-error');
        statusText.style.color = '';
        statusText.style.fontWeight = '';
      }

      // Always hide status icon
      const statusIcon = widget.querySelector('.fw-status-icon');
      if (statusIcon) {
        statusIcon.hidden = true;
      }

      // Always show actions (Yes/No buttons)
      const actions = widget.querySelector('[data-fw-actions]');
      if (actions) {
        actions.hidden = false;
      }

      // Always hide questions form initially
      const questions = widget.querySelector('[data-fw-questions]');
      if (questions) {
        questions.hidden = true;
      }

      // Always hide close button initially
      const btnClose = widget.querySelector('[data-fw-close]');
      if (btnClose) {
        btnClose.hidden = true;
      }
    });
  }

  // CRITICAL: Set up MutationObserver to watch for error messages being set
  // This catches cases where error messages are set after page load
  function setupStatusTextWatcher() {
    const widgets = document.querySelectorAll('[data-feedback-widget]');
    widgets.forEach(function(widget) {
      const statusText = widget.querySelector('.fw-status-text');
      if (statusText && !statusText.hasAttribute('data-watcher-attached')) {
        statusText.setAttribute('data-watcher-attached', 'true');

        const observer = new MutationObserver(function(mutations) {
          mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' || mutation.type === 'characterData') {
              const widgetState = widget.getAttribute('data-fw-state');
              // Only fix if in idle state (not during active submission)
              if (widgetState === 'idle' || !widgetState) {
                fixStatusText(statusText);
              }
            }
          });
        });

        observer.observe(statusText, {
          childList: true,
          characterData: true,
          subtree: true
        });
      }
    });
  }

  // CRITICAL: Run reset IMMEDIATELY, before DOMContentLoaded
  // This ensures widgets are reset even if cached HTML contains error states
  function initializeAll() {
    resetAllWidgets();
    setupStatusTextWatcher();
    attachGlobalSubmitHandler();
    attachGlobalYesNoHandler();
  }

  // CRITICAL: Run reset function as early as possible
  // Use DOMContentLoaded or immediate execution if DOM is ready
  function runEarlyReset() {
    try {
      // Only run if DOM elements exist
      const widgets = document.querySelectorAll('[data-feedback-widget]');
      if (widgets.length > 0) {
        resetAllWidgets();
      }
    } catch (e) {
      // Silently fail - will retry later
    }
  }

  // Run immediately if DOM is ready, otherwise wait for DOMContentLoaded
  if (document.readyState === 'complete' || document.readyState === 'interactive') {
    runEarlyReset();
  }

  if (document.readyState === 'loading') {
    // Run immediately (before DOMContentLoaded) - use requestAnimationFrame for earliest execution
    if (window.requestAnimationFrame) {
      requestAnimationFrame(function() {
        resetAllWidgets();
        setupStatusTextWatcher();
      });
    } else {
      resetAllWidgets();
      setupStatusTextWatcher();
    }

    document.addEventListener('DOMContentLoaded', function() {
      initializeAll();
      setTimeout(initializeAll, 50); // Faster timeout
      setTimeout(initializeAll, 200);
    });
  } else {
    // DOM already loaded, run immediately
    initializeAll();
    setTimeout(initializeAll, 50); // Faster timeout
    setTimeout(initializeAll, 200);
  }

  // Also run after delays as fallback
  setTimeout(initializeAll, 500);
  setTimeout(initializeAll, 1000);
  setTimeout(initializeAll, 2000); // Extra fallback for slow connections


})(Drupal, once);
