<?php

namespace Drupal\dga_feedback\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DGA Feedback module translations (English & Arabic).
 */
class FeedbackTranslationForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dga_feedback.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_feedback_translation_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    $form['#attached']['library'][] = 'dga_feedback/admin';

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure all front-end text for the DGA Feedback module in English and Arabic. These translations will be displayed dynamically based on the current site language.') . '</p>',
    ];

    // Closed State Section.
    $form['closed_state'] = [
      '#type' => 'details',
      '#title' => $this->t('Closed State (Default View)'),
      '#description' => $this->t('Text displayed when the feedback widget is closed.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['closed_state']['question_text'] = $this->buildBilingualField(
      'question_text',
      $this->t('Question text'),
      $this->t('The main question displayed (e.g., "Was this page useful?").'),
      $config->get('question_text_en') ?? 'Was this page useful?',
      $config->get('question_text_ar') ?? 'هل كانت هذه الصفحة مفيدة؟',
      TRUE
    );

    $form['closed_state']['yes_button_text'] = $this->buildBilingualField(
      'yes_button_text',
      $this->t('Yes button text'),
      $this->t('Text on the Yes button.'),
      $config->get('yes_button_text_en') ?? 'Yes',
      $config->get('yes_button_text_ar') ?? 'نعم',
      TRUE
    );

    $form['closed_state']['no_button_text'] = $this->buildBilingualField(
      'no_button_text',
      $this->t('No button text'),
      $this->t('Text on the No button.'),
      $config->get('no_button_text_en') ?? 'No',
      $config->get('no_button_text_ar') ?? 'لا',
      TRUE
    );

    $form['closed_state']['stats_text_template'] = $this->buildBilingualField(
      'stats_text_template',
      $this->t('Statistics text template'),
      $this->t('Template for statistics text. Use @percentage for percentage and @count for total count.'),
      $config->get('stats_text_template_en') ?? '@percentage% of users said Yes from @count Feedbacks',
      $config->get('stats_text_template_ar') ?? '@percentage% من المستخدمين قالوا نعم من @count تعليق',
      TRUE
    );

    // Form Section.
    $form['form_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Feedback Form'),
      '#description' => $this->t('Text displayed in the feedback form.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['form_section']['close_button_text'] = $this->buildBilingualField(
      'close_button_text',
      $this->t('Close button text'),
      NULL,
      $config->get('close_button_text_en') ?? 'Close',
      $config->get('close_button_text_ar') ?? 'إغلاق',
      TRUE
    );

    $form['form_section']['reasons_title'] = $this->buildBilingualField(
      'reasons_title',
      $this->t('Reasons title'),
      $this->t('Title for the reasons section (e.g., "Please tell us why").'),
      $config->get('reasons_title_en') ?? 'Please tell us why',
      $config->get('reasons_title_ar') ?? 'يرجى إخبارنا بالسبب',
      TRUE
    );

    $form['form_section']['reasons_instruction'] = $this->buildBilingualField(
      'reasons_instruction',
      $this->t('Reasons instruction'),
      $this->t('Instruction text for reasons (e.g., "you can select multiple options"). Note: Parentheses will be added automatically in the widget.'),
      $config->get('reasons_instruction_en') ?? 'you can select multiple options',
      $config->get('reasons_instruction_ar') ?? 'يمكنك اختيار خيارات متعددة',
      TRUE
    );

    $form['form_section']['reasons_yes'] = $this->buildBilingualTextarea(
      'reasons_yes',
      $this->t('Reasons for Yes'),
      $this->t('One reason per line. These will be shown when user clicks Yes.'),
      $config->get('reasons_yes_en') ?? "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else",
      $config->get('reasons_yes_ar') ?? "المحتوى ذو صلة\nكان مكتوبًا بشكل جيد\nجعل التخطيط القراءة سهلة\nشيء آخر",
      TRUE
    );

    $form['form_section']['reasons_no'] = $this->buildBilingualTextarea(
      'reasons_no',
      $this->t('Reasons for No'),
      $this->t('One reason per line. These will be shown when user clicks No.'),
      $config->get('reasons_no_en') ?? "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else",
      $config->get('reasons_no_ar') ?? "المحتوى غير ذي صلة\nالمحتوى غير دقيق\nالمحتوى طويل جدًا\nشيء آخر",
      TRUE
    );

    $form['form_section']['feedback_label'] = $this->buildBilingualField(
      'feedback_label',
      $this->t('Feedback label'),
      NULL,
      $config->get('feedback_label_en') ?? 'Feedback',
      $config->get('feedback_label_ar') ?? 'التعليقات',
      TRUE
    );

    $form['form_section']['feedback_placeholder'] = $this->buildBilingualField(
      'feedback_placeholder',
      $this->t('Feedback placeholder'),
      $this->t('Placeholder text for the feedback textarea.'),
      $config->get('feedback_placeholder_en') ?? 'text placeholder',
      $config->get('feedback_placeholder_ar') ?? 'نص توضيحي',
      TRUE
    );

    $form['form_section']['gender_label'] = $this->buildBilingualField(
      'gender_label',
      $this->t('Gender label'),
      $this->t('Label for gender section (e.g., "I\'m").'),
      $config->get('gender_label_en') ?? "I'm",
      $config->get('gender_label_ar') ?? 'أنا',
      TRUE
    );

    $form['form_section']['gender_male'] = $this->buildBilingualField(
      'gender_male',
      $this->t('Male option text'),
      NULL,
      $config->get('gender_male_en') ?? 'Male',
      $config->get('gender_male_ar') ?? 'ذكر',
      TRUE
    );

    $form['form_section']['gender_female'] = $this->buildBilingualField(
      'gender_female',
      $this->t('Female option text'),
      NULL,
      $config->get('gender_female_en') ?? 'Female',
      $config->get('gender_female_ar') ?? 'أنثى',
      TRUE
    );

    $form['form_section']['gender_prefer_not_to_say'] = $this->buildBilingualField(
      'gender_prefer_not_to_say',
      $this->t('Prefer not to say option text'),
      NULL,
      $config->get('gender_prefer_not_to_say_en') ?? 'Prefer not to say',
      $config->get('gender_prefer_not_to_say_ar') ?? 'أفضل عدم الإفصاح',
      TRUE
    );

    $form['form_section']['submit_button_text'] = $this->buildBilingualField(
      'submit_button_text',
      $this->t('Submit button text'),
      NULL,
      $config->get('submit_button_text_en') ?? 'Submit',
      $config->get('submit_button_text_ar') ?? 'إرسال',
      TRUE
    );

    // Submitted Section.
    $form['submitted_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Submitted State'),
      '#description' => $this->t('Text displayed after feedback is submitted.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['submitted_section']['submitted_success_text'] = $this->buildBilingualField(
      'submitted_success_text',
      $this->t('Success message'),
      $this->t('Message shown when feedback is successfully submitted.'),
      $config->get('submitted_success_text_en') ?? 'Your feedback is submitted!',
      $config->get('submitted_success_text_ar') ?? 'تم إرسال تعليقك!',
      TRUE
    );

    // Validation Messages Section.
    $form['validation'] = [
      '#type' => 'details',
      '#title' => $this->t('Validation Messages'),
      '#description' => $this->t('Error messages shown when form validation fails. These appear as red text under each field.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['validation']['validation_yes_no_required'] = $this->buildBilingualField(
      'validation_yes_no_required',
      $this->t('Yes/No required message'),
      $this->t('Message shown when the user has not selected Yes or No.'),
      $config->get('validation_yes_no_required_en') ?? 'Please select Yes or No first.',
      $config->get('validation_yes_no_required_ar') ?? 'يرجى اختيار نعم أو لا أولاً.',
      TRUE
    );

    $form['validation']['validation_reason_required'] = $this->buildBilingualField(
      'validation_reason_required',
      $this->t('Reason required message'),
      $this->t('Message shown when no reason is selected.'),
      $config->get('validation_reason_required_en') ?? 'Please select at least one reason',
      $config->get('validation_reason_required_ar') ?? 'يرجى اختيار سبب واحد على الأقل',
      TRUE
    );

    $form['validation']['validation_reason_invalid'] = $this->buildBilingualField(
      'validation_reason_invalid',
      $this->t('Reason invalid message'),
      $this->t('Message shown when no valid reasons remain after filtering.'),
      $config->get('validation_reason_invalid_en') ?? 'At least one valid reason must be selected.',
      $config->get('validation_reason_invalid_ar') ?? 'يجب اختيار سبب واحد صالح على الأقل.',
      TRUE
    );

    $form['validation']['validation_feedback_required'] = $this->buildBilingualField(
      'validation_feedback_required',
      $this->t('Feedback required message'),
      $this->t('Message shown when feedback text is missing.'),
      $config->get('validation_feedback_required_en') ?? 'Please provide feedback text',
      $config->get('validation_feedback_required_ar') ?? 'يرجى تقديم نص التعليق',
      TRUE
    );

    $form['validation']['validation_gender_required'] = $this->buildBilingualField(
      'validation_gender_required',
      $this->t('Gender required message'),
      $this->t('Message shown when gender is not selected.'),
      $config->get('validation_gender_required_en') ?? 'Please select your gender',
      $config->get('validation_gender_required_ar') ?? 'يرجى اختيار جنسك',
      TRUE
    );

    $form['validation']['validation_submission_failed'] = $this->buildBilingualField(
      'validation_submission_failed',
      $this->t('Submission failed message'),
      $this->t('Message shown when the submission fails.'),
      $config->get('validation_submission_failed_en') ?? 'Submission failed. Please try again.',
      $config->get('validation_submission_failed_ar') ?? 'فشل الإرسال. يرجى المحاولة مرة أخرى.',
      TRUE
    );

    $form['validation']['validation_unknown_error'] = $this->buildBilingualField(
      'validation_unknown_error',
      $this->t('Unknown error message'),
      $this->t('Fallback error message shown for unexpected errors.'),
      $config->get('validation_unknown_error_en') ?? 'Unknown error',
      $config->get('validation_unknown_error_ar') ?? 'خطأ غير معروف',
      TRUE
    );

    $form['validation']['button_submitting_text'] = $this->buildBilingualField(
      'button_submitting_text',
      $this->t('Submitting button text'),
      $this->t('Temporary text shown on the submit button while submitting.'),
      $config->get('button_submitting_text_en') ?? 'Submitting...',
      $config->get('button_submitting_text_ar') ?? 'جاري الإرسال...',
      TRUE
    );

    // API / Backend Messages Section.
    $form['api_messages'] = [
      '#type' => 'details',
      '#title' => $this->t('API & Backend Messages'),
      '#description' => $this->t('Messages returned by the backend API. These should align with the validation messages for consistency.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['api_messages']['api_method_not_allowed'] = $this->buildBilingualField(
      'api_method_not_allowed',
      $this->t('Method not allowed'),
      NULL,
      $config->get('api_method_not_allowed_en') ?? 'Method not allowed. Use POST.',
      $config->get('api_method_not_allowed_ar') ?? 'الطريقة غير مسموحة. استخدم POST.',
      TRUE
    );

    $form['api_messages']['api_invalid_json'] = $this->buildBilingualField(
      'api_invalid_json',
      $this->t('Invalid JSON'),
      NULL,
      $config->get('api_invalid_json_en') ?? 'Invalid JSON data',
      $config->get('api_invalid_json_ar') ?? 'بيانات JSON غير صالحة',
      TRUE
    );

    $form['api_messages']['api_invalid_useful'] = $this->buildBilingualField(
      'api_invalid_useful',
      $this->t('Invalid is_useful value'),
      NULL,
      $config->get('api_invalid_useful_en') ?? 'is_useful must be "yes" or "no"',
      $config->get('api_invalid_useful_ar') ?? 'يجب أن يكون is_useful "yes" أو "no"',
      TRUE
    );

    $form['api_messages']['api_rate_limit'] = $this->buildBilingualField(
      'api_rate_limit',
      $this->t('Rate limit reached'),
      NULL,
      $config->get('api_rate_limit_en') ?? 'Too many submissions. Please try again later.',
      $config->get('api_rate_limit_ar') ?? 'عدد كبير جدًا من الإرسالات. يرجى المحاولة مرة أخرى لاحقًا.',
      TRUE
    );

    $form['api_messages']['api_save_failed'] = $this->buildBilingualField(
      'api_save_failed',
      $this->t('Save failed message'),
      NULL,
      $config->get('api_save_failed_en') ?? 'Failed to save feedback.',
      $config->get('api_save_failed_ar') ?? 'فشل حفظ التعليق.',
      TRUE
    );

    $form['api_messages']['api_success_message'] = $this->buildBilingualField(
      'api_success_message',
      $this->t('Success message'),
      $this->t('Message returned after successful submission.'),
      $config->get('api_success_message_en') ?? 'Thank you for your feedback!',
      $config->get('api_success_message_ar') ?? 'شكرًا لك على تعليقك!',
      TRUE
    );

    // Menu Items Section.
    $form['menu_items'] = [
      '#type' => 'details',
      '#title' => $this->t('Menu Items'),
      '#description' => $this->t('Translations for admin menu items.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['menu_items']['menu_title_dga_feedback'] = $this->buildBilingualField(
      'menu_title_dga_feedback',
      $this->t('DGA Feedback (Main Menu)'),
      $this->t('Title for the main DGA Feedback menu item in the sidebar.'),
      $config->get('menu_title_dga_feedback_en') ?? 'DGA Feedback',
      $config->get('menu_title_dga_feedback_ar') ?? 'تعليقات DGA',
      TRUE
    );

    $form['menu_items']['menu_title_dashboard'] = $this->buildBilingualField(
      'menu_title_dashboard',
      $this->t('Feedback Dashboard'),
      $this->t('Title for the Feedback Dashboard menu item/tab.'),
      $config->get('menu_title_dashboard_en') ?? 'Feedback Dashboard',
      $config->get('menu_title_dashboard_ar') ?? 'لوحة تحكم التعليقات',
      TRUE
    );

    $form['menu_items']['menu_title_settings'] = $this->buildBilingualField(
      'menu_title_settings',
      $this->t('Settings'),
      $this->t('Title for the Settings menu item/tab.'),
      $config->get('menu_title_settings_en') ?? 'Settings',
      $config->get('menu_title_settings_ar') ?? 'الإعدادات',
      TRUE
    );

    $form['menu_items']['menu_title_translations'] = $this->buildBilingualField(
      'menu_title_translations',
      $this->t('Translations'),
      $this->t('Title for the Translations menu item/tab.'),
      $config->get('menu_title_translations_en') ?? 'Translations',
      $config->get('menu_title_translations_ar') ?? 'الترجمات',
      TRUE
    );

    return parent::buildForm($form, $form_state);
  }

  /**
   * Builds a bilingual text field with English and Arabic columns.
   */
  protected function buildBilingualField($field_name, $title, $description, $default_en, $default_ar, $required = FALSE) {
    $config = $this->config('dga_feedback.settings');

    $field = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-wrapper']],
      '#tree' => TRUE,
    ];

    $field['label'] = [
      '#type' => 'markup',
      '#markup' => '<div class="bilingual-field-label"><strong>' . $title . '</strong></div>',
    ];

    if ($description) {
      $field['description'] = [
        '#type' => 'markup',
        '#markup' => '<div class="description">' . $description . '</div>',
      ];
    }

    $field['columns'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-columns', 'clearfix']],
      '#tree' => TRUE,
    ];

    $field['columns']['en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('English'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_en') ?? $default_en,
      '#required' => $required,
      '#attributes' => ['class' => ['bilingual-field-en']],
      '#prefix' => '<div class="bilingual-column bilingual-column-en">',
      '#suffix' => '</div>',
    ];

    $field['columns']['ar'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Arabic'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_ar') ?? $default_ar,
      '#required' => $required,
      '#attributes' => ['class' => ['bilingual-field-ar']],
      '#prefix' => '<div class="bilingual-column bilingual-column-ar">',
      '#suffix' => '</div>',
    ];

    return $field;
  }

  /**
   * Builds a bilingual textarea field with English and Arabic columns.
   */
  protected function buildBilingualTextarea($field_name, $title, $description, $default_en, $default_ar, $required = FALSE) {
    $config = $this->config('dga_feedback.settings');

    $field = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-wrapper']],
      '#tree' => TRUE,
    ];

    $field['label'] = [
      '#type' => 'markup',
      '#markup' => '<div class="bilingual-field-label"><strong>' . $title . '</strong></div>',
    ];

    if ($description) {
      $field['description'] = [
        '#type' => 'markup',
        '#markup' => '<div class="description">' . $description . '</div>',
      ];
    }

    $field['columns'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-columns', 'clearfix']],
      '#tree' => TRUE,
    ];

    $field['columns']['en'] = [
      '#type' => 'textarea',
      '#title' => $this->t('English'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_en') ?? $default_en,
      '#required' => $required,
      '#rows' => 4,
      '#attributes' => ['class' => ['bilingual-field-en']],
      '#prefix' => '<div class="bilingual-column bilingual-column-en">',
      '#suffix' => '</div>',
    ];

    $field['columns']['ar'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Arabic'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_ar') ?? $default_ar,
      '#required' => $required,
      '#rows' => 4,
      '#attributes' => ['class' => ['bilingual-field-ar']],
      '#prefix' => '<div class="bilingual-column bilingual-column-ar">',
      '#suffix' => '</div>',
    ];

    return $field;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dga_feedback.settings');

    $getValue = function (array $path) use ($form_state) {
      $value = $form_state->getValue($path);
      return $value !== NULL ? $value : '';
    };

    // Closed state.
    $config->set('question_text_en', $getValue(['closed_state', 'question_text', 'columns', 'en']));
    $config->set('question_text_ar', $getValue(['closed_state', 'question_text', 'columns', 'ar']));
    $config->set('yes_button_text_en', $getValue(['closed_state', 'yes_button_text', 'columns', 'en']));
    $config->set('yes_button_text_ar', $getValue(['closed_state', 'yes_button_text', 'columns', 'ar']));
    $config->set('no_button_text_en', $getValue(['closed_state', 'no_button_text', 'columns', 'en']));
    $config->set('no_button_text_ar', $getValue(['closed_state', 'no_button_text', 'columns', 'ar']));
    $config->set('stats_text_template_en', $getValue(['closed_state', 'stats_text_template', 'columns', 'en']));
    $config->set('stats_text_template_ar', $getValue(['closed_state', 'stats_text_template', 'columns', 'ar']));

    // Form section.
    $config->set('close_button_text_en', $getValue(['form_section', 'close_button_text', 'columns', 'en']));
    $config->set('close_button_text_ar', $getValue(['form_section', 'close_button_text', 'columns', 'ar']));
    $config->set('reasons_title_en', $getValue(['form_section', 'reasons_title', 'columns', 'en']));
    $config->set('reasons_title_ar', $getValue(['form_section', 'reasons_title', 'columns', 'ar']));
    $config->set('reasons_instruction_en', $getValue(['form_section', 'reasons_instruction', 'columns', 'en']));
    $config->set('reasons_instruction_ar', $getValue(['form_section', 'reasons_instruction', 'columns', 'ar']));
    $config->set('reasons_yes_en', $getValue(['form_section', 'reasons_yes', 'columns', 'en']));
    $config->set('reasons_yes_ar', $getValue(['form_section', 'reasons_yes', 'columns', 'ar']));
    $config->set('reasons_no_en', $getValue(['form_section', 'reasons_no', 'columns', 'en']));
    $config->set('reasons_no_ar', $getValue(['form_section', 'reasons_no', 'columns', 'ar']));
    $config->set('feedback_label_en', $getValue(['form_section', 'feedback_label', 'columns', 'en']));
    $config->set('feedback_label_ar', $getValue(['form_section', 'feedback_label', 'columns', 'ar']));
    $config->set('feedback_placeholder_en', $getValue(['form_section', 'feedback_placeholder', 'columns', 'en']));
    $config->set('feedback_placeholder_ar', $getValue(['form_section', 'feedback_placeholder', 'columns', 'ar']));
    $config->set('gender_label_en', $getValue(['form_section', 'gender_label', 'columns', 'en']));
    $config->set('gender_label_ar', $getValue(['form_section', 'gender_label', 'columns', 'ar']));
    $config->set('gender_male_en', $getValue(['form_section', 'gender_male', 'columns', 'en']));
    $config->set('gender_male_ar', $getValue(['form_section', 'gender_male', 'columns', 'ar']));
    $config->set('gender_female_en', $getValue(['form_section', 'gender_female', 'columns', 'en']));
    $config->set('gender_female_ar', $getValue(['form_section', 'gender_female', 'columns', 'ar']));
    $config->set('gender_prefer_not_to_say_en', $getValue(['form_section', 'gender_prefer_not_to_say', 'columns', 'en']));
    $config->set('gender_prefer_not_to_say_ar', $getValue(['form_section', 'gender_prefer_not_to_say', 'columns', 'ar']));
    $config->set('submit_button_text_en', $getValue(['form_section', 'submit_button_text', 'columns', 'en']));
    $config->set('submit_button_text_ar', $getValue(['form_section', 'submit_button_text', 'columns', 'ar']));

    // Submitted section.
    $config->set('submitted_success_text_en', $getValue(['submitted_section', 'submitted_success_text', 'columns', 'en']));
    $config->set('submitted_success_text_ar', $getValue(['submitted_section', 'submitted_success_text', 'columns', 'ar']));

    // Validation messages.
    $config->set('validation_yes_no_required_en', $getValue(['validation', 'validation_yes_no_required', 'columns', 'en']));
    $config->set('validation_yes_no_required_ar', $getValue(['validation', 'validation_yes_no_required', 'columns', 'ar']));
    $config->set('validation_reason_required_en', $getValue(['validation', 'validation_reason_required', 'columns', 'en']));
    $config->set('validation_reason_required_ar', $getValue(['validation', 'validation_reason_required', 'columns', 'ar']));
    $config->set('validation_reason_invalid_en', $getValue(['validation', 'validation_reason_invalid', 'columns', 'en']));
    $config->set('validation_reason_invalid_ar', $getValue(['validation', 'validation_reason_invalid', 'columns', 'ar']));
    $config->set('validation_feedback_required_en', $getValue(['validation', 'validation_feedback_required', 'columns', 'en']));
    $config->set('validation_feedback_required_ar', $getValue(['validation', 'validation_feedback_required', 'columns', 'ar']));
    $config->set('validation_gender_required_en', $getValue(['validation', 'validation_gender_required', 'columns', 'en']));
    $config->set('validation_gender_required_ar', $getValue(['validation', 'validation_gender_required', 'columns', 'ar']));
    $config->set('validation_submission_failed_en', $getValue(['validation', 'validation_submission_failed', 'columns', 'en']));
    $config->set('validation_submission_failed_ar', $getValue(['validation', 'validation_submission_failed', 'columns', 'ar']));
    $config->set('validation_unknown_error_en', $getValue(['validation', 'validation_unknown_error', 'columns', 'en']));
    $config->set('validation_unknown_error_ar', $getValue(['validation', 'validation_unknown_error', 'columns', 'ar']));
    $config->set('button_submitting_text_en', $getValue(['validation', 'button_submitting_text', 'columns', 'en']));
    $config->set('button_submitting_text_ar', $getValue(['validation', 'button_submitting_text', 'columns', 'ar']));

    // API messages.
    $config->set('api_method_not_allowed_en', $getValue(['api_messages', 'api_method_not_allowed', 'columns', 'en']));
    $config->set('api_method_not_allowed_ar', $getValue(['api_messages', 'api_method_not_allowed', 'columns', 'ar']));
    $config->set('api_invalid_json_en', $getValue(['api_messages', 'api_invalid_json', 'columns', 'en']));
    $config->set('api_invalid_json_ar', $getValue(['api_messages', 'api_invalid_json', 'columns', 'ar']));
    $config->set('api_invalid_useful_en', $getValue(['api_messages', 'api_invalid_useful', 'columns', 'en']));
    $config->set('api_invalid_useful_ar', $getValue(['api_messages', 'api_invalid_useful', 'columns', 'ar']));
    $config->set('api_rate_limit_en', $getValue(['api_messages', 'api_rate_limit', 'columns', 'en']));
    $config->set('api_rate_limit_ar', $getValue(['api_messages', 'api_rate_limit', 'columns', 'ar']));
    $config->set('api_save_failed_en', $getValue(['api_messages', 'api_save_failed', 'columns', 'en']));
    $config->set('api_save_failed_ar', $getValue(['api_messages', 'api_save_failed', 'columns', 'ar']));
    $config->set('api_success_message_en', $getValue(['api_messages', 'api_success_message', 'columns', 'en']));
    $config->set('api_success_message_ar', $getValue(['api_messages', 'api_success_message', 'columns', 'ar']));

    // Menu items.
    $config->set('menu_title_dga_feedback_en', $getValue(['menu_items', 'menu_title_dga_feedback', 'columns', 'en']));
    $config->set('menu_title_dga_feedback_ar', $getValue(['menu_items', 'menu_title_dga_feedback', 'columns', 'ar']));
    $config->set('menu_title_dashboard_en', $getValue(['menu_items', 'menu_title_dashboard', 'columns', 'en']));
    $config->set('menu_title_dashboard_ar', $getValue(['menu_items', 'menu_title_dashboard', 'columns', 'ar']));
    $config->set('menu_title_settings_en', $getValue(['menu_items', 'menu_title_settings', 'columns', 'en']));
    $config->set('menu_title_settings_ar', $getValue(['menu_items', 'menu_title_settings', 'columns', 'ar']));
    $config->set('menu_title_translations_en', $getValue(['menu_items', 'menu_title_translations', 'columns', 'en']));
    $config->set('menu_title_translations_ar', $getValue(['menu_items', 'menu_title_translations', 'columns', 'ar']));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}

