<?php

namespace Drupal\dga_feedback\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\dga_feedback\Service\DgaFeedbackService;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a DGA Feedback widget block.
 *
 * @Block(
 *   id = "dga_feedback_block",
 *   admin_label = @Translation("DGA Feedback Widget"),
 *   category = @Translation("Custom")
 * )
 */
class DgaFeedbackBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The feedback service.
   *
   * @var \Drupal\dga_feedback\Service\DgaFeedbackService
   */
  protected $feedbackService;

  /**
   * Constructs a DgaFeedbackBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dga_feedback\Service\DgaFeedbackService $feedback_service
   *   The feedback service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DgaFeedbackService $feedback_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->feedbackService = $feedback_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dga_feedback.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get current URL path - use canonical path, not alias, for consistency.
    $current_path = \Drupal::service('path.current')->getPath();

    // Get the current request to extract the actual URL path
    $request = \Drupal::request();
    $request_path = $request->getPathInfo();

    // Get the alias
    $url_alias = \Drupal::service('path_alias.manager')->getAliasByPath($current_path);

    // IMPORTANT: Use alias (without language prefix) for consistency
    if ($url_alias !== $current_path && $url_alias !== '/') {
      $url = rtrim($url_alias, '/');
    } else {
      // If no alias, strip language prefix from request path
      $url = $request_path;
      if (preg_match('#^/[a-z]{2}/(.+)$#', $url, $matches)) {
        $url = '/' . $matches[1];
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Ensure we don't have language prefix in the final URL
    if (preg_match('#^/[a-z]{2}/#', $url)) {
      $url = preg_replace('#^/[a-z]{2}#', '', $url);
      $url = rtrim($url, '/') ?: '/';
    }

    // Try to get entity context if available.
    $entity_type = NULL;
    $entity_id = NULL;
    $route_match = \Drupal::routeMatch();

    // Check if we're on a node page.
    $node = $route_match->getParameter('node');
    if ($node instanceof Node) {
      $entity_type = 'node';
      $entity_id = $node->id();
      if ($entity_id > 0) {
        $stats = $this->feedbackService->getStatistics($entity_type, $entity_id, $url);
      } else {
        $stats = $this->feedbackService->getStatisticsByUrl($url);
      }
    }
    else {
      // Fallback to URL-based statistics.
      $stats = $this->feedbackService->getStatisticsByUrl($url);
    }

    // Format the yes percentage
    $yes_percentage = $stats['yes_percentage'] > 0 ? number_format($stats['yes_percentage'], 0, '.', '') : '0';
    $total_count = (int) $stats['total_count'];

    // Ensure count is always a number (not string).
    $total_count = max(0, $total_count);

    // Load configuration for text strings.
    $config = \Drupal::config('dga_feedback.settings');

    // Determine current language - use multiple methods to ensure correct detection for anonymous users
    $language_manager = \Drupal::languageManager();
    $request = \Drupal::request();
    $route_match = \Drupal::routeMatch();
    
    // CRITICAL: For anonymous users, language detection must be more robust
    // Method 1: Get language from language manager FIRST (most reliable for current page context)
    $current_language = $language_manager->getCurrentLanguage(LanguageInterface::TYPE_INTERFACE);
    $current_lang = $current_language->getId();
    
    // Method 2: Try to get language from URL path (important for anonymous users)
    $path = $request->getPathInfo();
    $lang_from_path = NULL;
    if (preg_match('#^/([a-z]{2})(/|$)#', $path, $matches)) {
      $lang_from_path = $matches[1];
    }
    
    // Method 2b: Also check the full URL (including query string)
    if (!$lang_from_path) {
      $full_url = $request->getUri();
      if (preg_match('#/([a-z]{2})(/|$|\?)#', $full_url, $url_matches)) {
        $lang_from_path = $url_matches[1];
      }
    }
    
    // Method 2c: Check query parameter (some setups use ?lang=ar)
    if (!$lang_from_path) {
      $lang_from_query = $request->query->get('lang');
      if ($lang_from_query && in_array($lang_from_query, ['en', 'ar'])) {
        $lang_from_path = $lang_from_query;
      }
    }
    
    // Method 3: Try to get from route parameters
    $lang_from_route = NULL;
    if ($route_match && $route_match->getRouteObject()) {
      $route_lang = $route_match->getParameter('langcode');
      if ($route_lang && in_array($route_lang, ['en', 'ar'])) {
        $lang_from_route = $route_lang;
      }
    }
    
    // Method 4: Check if we're on a node and get its language
    $lang_from_entity = NULL;
    if (isset($node) && $node instanceof Node) {
      $node_lang = $node->language()->getId();
      if (in_array($node_lang, ['en', 'ar'])) {
        $lang_from_entity = $node_lang;
      }
    }
    
    // CRITICAL: Priority order for anonymous users:
    // 1. Entity language (if on a node page)
    // 2. URL path (most reliable indicator for anonymous)
    // 3. Route parameter
    // 4. Language manager (fallback)
    if ($lang_from_entity && in_array($lang_from_entity, ['en', 'ar'])) {
      $current_lang = $lang_from_entity;
    } elseif ($lang_from_path && in_array($lang_from_path, ['en', 'ar'])) {
      $current_lang = $lang_from_path;
    } elseif ($lang_from_route && in_array($lang_from_route, ['en', 'ar'])) {
      $current_lang = $lang_from_route;
    }
    // If current_lang is still not set or invalid, use language manager result
    
    // Ensure we have a valid language code
    if (!in_array($current_lang, ['en', 'ar'])) {
      $current_lang = 'en'; // Default fallback
    }
    
    $is_arabic = ($current_lang === 'ar');

    $getTranslation = function ($base_key, $default_en, $default_ar = NULL) use ($config, $is_arabic) {
      $value_en = $config->get($base_key . '_en');
      $value_ar = $config->get($base_key . '_ar');

      // Check if values are set and not null (allow empty strings as valid values)
      $has_en = $value_en !== NULL && $value_en !== '';
      $has_ar = $value_ar !== NULL && $value_ar !== '';

      if ($is_arabic) {
        // CRITICAL: For Arabic view, ALWAYS prefer Arabic text
        // Priority: Arabic config > Arabic default > English config > English default
        if ($has_ar) {
          return $value_ar;
        }
        // If we have a default Arabic string, use it (NEVER use English when in Arabic view)
        if ($default_ar !== NULL && $default_ar !== '') {
          return $default_ar;
        }
        // Last resort: fall back to English config only if no Arabic default provided
        // This should rarely happen, but ensures we always return something
        if ($has_en) {
          return $value_en;
        }
        return $default_en;
      }

      // English: prefer English value, fallback to default
      if ($has_en) {
        return $value_en;
      }
      return $default_en;
    };

    $reasons_yes_raw = $getTranslation(
      'reasons_yes',
      "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else",
      "المحتوى ذو صلة\nكان مكتوبًا بشكل جيد\nجعل التخطيط القراءة سهلة\nشيء آخر"
    );
    $reasons_no_raw = $getTranslation(
      'reasons_no',
      "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else",
      "المحتوى غير ذي صلة\nالمحتوى غير دقيق\nالمحتوى طويل جدًا\nشيء آخر"
    );

    $reasons_yes_translated = [];
    $reasons_no_translated = [];

    foreach (explode("\n", $reasons_yes_raw) as $reason) {
      $reason = trim($reason);
      if ($reason !== '') {
        $reasons_yes_translated[] = $reason;
      }
    }

    foreach (explode("\n", $reasons_no_raw) as $reason) {
      $reason = trim($reason);
      if ($reason !== '') {
        $reasons_no_translated[] = $reason;
      }
    }

    // CRITICAL: Validate that reasons are in the correct language
    // If we're in Arabic view but reasons contain English text, force Arabic defaults
    if ($is_arabic) {
      // Check if reasons contain English text (common English words in reasons)
      $english_indicators = ['Content is', 'It was', 'The layout', 'Something else', 'not relevant', 'not accurate', 'too long'];
      $has_english = false;
      
      // Check yes reasons
      foreach ($reasons_yes_translated as $reason) {
        foreach ($english_indicators as $indicator) {
          if (stripos($reason, $indicator) !== false) {
            $has_english = true;
            break 2;
          }
        }
      }
      
      // Check no reasons
      if (!$has_english) {
        foreach ($reasons_no_translated as $reason) {
          foreach ($english_indicators as $indicator) {
            if (stripos($reason, $indicator) !== false) {
              $has_english = true;
              break 2;
            }
          }
        }
      }
      
      // If English text detected in Arabic view, force Arabic defaults
      if ($has_english) {
        $reasons_yes_translated = [
          "المحتوى ذو صلة",
          "كان مكتوبًا بشكل جيد",
          "جعل التخطيط القراءة سهلة",
          "شيء آخر"
        ];
        $reasons_no_translated = [
          "المحتوى غير ذي صلة",
          "المحتوى غير دقيق",
          "المحتوى طويل جدًا",
          "شيء آخر"
        ];
      }
    }

    // Build texts array - ensure all values are set (never empty/null)
    // Force non-empty values to prevent missing translations
    $texts_array = [
      'question_text' => $getTranslation('question_text', 'Was this page useful?', 'هل كانت هذه الصفحة مفيدة؟') ?: ($is_arabic ? 'هل كانت هذه الصفحة مفيدة؟' : 'Was this page useful?'),
      'yes_button_text' => $getTranslation('yes_button_text', 'Yes', 'نعم') ?: ($is_arabic ? 'نعم' : 'Yes'),
      'no_button_text' => $getTranslation('no_button_text', 'No', 'لا') ?: ($is_arabic ? 'لا' : 'No'),
      'stats_text_template' => $getTranslation('stats_text_template', '@percentage% of users said Yes from @count Feedbacks', '@percentage% من المستخدمين قالوا نعم من @count تعليق') ?: ($is_arabic ? '@percentage% من المستخدمين قالوا نعم من @count تعليق' : '@percentage% of users said Yes from @count Feedbacks'),
      'close_button_text' => $getTranslation('close_button_text', 'Close', 'إغلاق') ?: ($is_arabic ? 'إغلاق' : 'Close'),
      'reasons_title' => $getTranslation('reasons_title', 'Please tell us why', 'يرجى إخبارنا بالسبب') ?: ($is_arabic ? 'يرجى إخبارنا بالسبب' : 'Please tell us why'),
      'reasons_instruction' => $getTranslation('reasons_instruction', 'you can select multiple options', 'يمكنك اختيار خيارات متعددة') ?: ($is_arabic ? 'يمكنك اختيار خيارات متعددة' : 'you can select multiple options'),
      'feedback_label' => $getTranslation('feedback_label', 'Feedback', 'التعليقات') ?: ($is_arabic ? 'التعليقات' : 'Feedback'),
      'feedback_placeholder' => $getTranslation('feedback_placeholder', 'text placeholder', 'نص توضيحي') ?: ($is_arabic ? 'نص توضيحي' : 'text placeholder'),
      'gender_label' => $getTranslation('gender_label', "I'm", 'أنا') ?: ($is_arabic ? 'أنا' : "I'm"),
      'gender_male' => $getTranslation('gender_male', 'Male', 'ذكر') ?: ($is_arabic ? 'ذكر' : 'Male'),
      'gender_female' => $getTranslation('gender_female', 'Female', 'أنثى') ?: ($is_arabic ? 'أنثى' : 'Female'),
      'gender_prefer_not_to_say' => $getTranslation('gender_prefer_not_to_say', 'Prefer not to say', 'أفضل عدم الإفصاح') ?: ($is_arabic ? 'أفضل عدم الإفصاح' : 'Prefer not to say'),
      'submit_button_text' => $getTranslation('submit_button_text', 'Submit', 'إرسال') ?: ($is_arabic ? 'إرسال' : 'Submit'),
      'submitted_success_text' => $getTranslation('submitted_success_text', 'Your feedback is submitted!', 'تم إرسال تعليقك!') ?: ($is_arabic ? 'تم إرسال تعليقك!' : 'Your feedback is submitted!'),
      // CRITICAL: Ensure reasons are always in the correct language
      // If we're in Arabic view but got English reasons, force Arabic defaults
      'reasons_yes' => !empty($reasons_yes_translated) ? implode("\n", $reasons_yes_translated) : ($is_arabic ? "المحتوى ذو صلة\nكان مكتوبًا بشكل جيد\nجعل التخطيط القراءة سهلة\nشيء آخر" : "Content is relevant\nIt was well written\nThe layout made it easy to read\nSomething else"),
      'reasons_no' => !empty($reasons_no_translated) ? implode("\n", $reasons_no_translated) : ($is_arabic ? "المحتوى غير ذي صلة\nالمحتوى غير دقيق\nالمحتوى طويل جدًا\nشيء آخر" : "Content is not relevant\nContent is not accurate\nContent is too long\nSomething else"),
      'validation_yes_no_required' => $getTranslation('validation_yes_no_required', 'Please select Yes or No first.', 'يرجى اختيار نعم أو لا أولاً.') ?: ($is_arabic ? 'يرجى اختيار نعم أو لا أولاً.' : 'Please select Yes or No first.'),
      'validation_reason_required' => $getTranslation('validation_reason_required', 'Please select at least one reason', 'يرجى اختيار سبب واحد على الأقل') ?: ($is_arabic ? 'يرجى اختيار سبب واحد على الأقل' : 'Please select at least one reason'),
      'validation_reason_invalid' => $getTranslation('validation_reason_invalid', 'At least one valid reason must be selected.', 'يجب اختيار سبب واحد صالح على الأقل.') ?: ($is_arabic ? 'يجب اختيار سبب واحد صالح على الأقل.' : 'At least one valid reason must be selected.'),
      'validation_feedback_required' => $getTranslation('validation_feedback_required', 'Please provide feedback text', 'يرجى تقديم نص التعليق') ?: ($is_arabic ? 'يرجى تقديم نص التعليق' : 'Please provide feedback text'),
      'validation_gender_required' => $getTranslation('validation_gender_required', 'Please select your gender', 'يرجى اختيار جنسك') ?: ($is_arabic ? 'يرجى اختيار جنسك' : 'Please select your gender'),
      'validation_submission_failed' => $getTranslation('validation_submission_failed', 'Submission failed. Please try again.', 'فشل الإرسال. يرجى المحاولة مرة أخرى.') ?: ($is_arabic ? 'فشل الإرسال. يرجى المحاولة مرة أخرى.' : 'Submission failed. Please try again.'),
      'validation_unknown_error' => $getTranslation('validation_unknown_error', 'Unknown error', 'خطأ غير معروف') ?: ($is_arabic ? 'خطأ غير معروف' : 'Unknown error'),
      'button_submitting_text' => $getTranslation('button_submitting_text', 'Submitting...', 'جاري الإرسال...') ?: ($is_arabic ? 'جاري الإرسال...' : 'Submitting...'),
    ];

    // Build render array.
    $build = [
      '#theme' => 'dga_feedback_widget',
      '#yes_percentage' => $yes_percentage,
      '#total_count' => $total_count,
      '#url' => $url,
      '#entity_type' => $entity_type,
      '#entity_id' => $entity_id,
      '#texts' => $texts_array,
      '#attached' => [
        'library' => [
          'dga_feedback/feedback_widget',
        ],
        'drupalSettings' => [
          'dgaFeedback' => [
            // Ensure language-aware URLs by explicitly setting the interface language
            'submitUrl' => \Drupal\Core\Url::fromRoute('dga_feedback.submit', [], ['language' => \Drupal::languageManager()->getCurrentLanguage(\Drupal\Core\Language\LanguageInterface::TYPE_INTERFACE)])->toString(),
            'statsUrl' => \Drupal\Core\Url::fromRoute('dga_feedback.stats', [], ['language' => \Drupal::languageManager()->getCurrentLanguage(\Drupal\Core\Language\LanguageInterface::TYPE_INTERFACE)])->toString(),
            'refreshBlockUrl' => \Drupal\Core\Url::fromRoute('dga_feedback.refresh_block', [], ['language' => \Drupal::languageManager()->getCurrentLanguage(\Drupal\Core\Language\LanguageInterface::TYPE_INTERFACE)])->toString(),
            'questionText' => $texts_array['question_text'],
            'submittedSuccessText' => $texts_array['submitted_success_text'],
            'errorMessage' => $texts_array['validation_submission_failed'],
            'unknownMessage' => $texts_array['validation_unknown_error'],
            'buttonSubmittingText' => $texts_array['button_submitting_text'],
            'apiMessages' => [
              'methodNotAllowed' => $getTranslation('api_method_not_allowed', 'Method not allowed. Use POST.', 'الطريقة غير مسموحة. استخدم POST.') ?: ($is_arabic ? 'الطريقة غير مسموحة. استخدم POST.' : 'Method not allowed. Use POST.'),
              'invalidJson' => $getTranslation('api_invalid_json', 'Invalid JSON data', 'بيانات JSON غير صالحة') ?: ($is_arabic ? 'بيانات JSON غير صالحة' : 'Invalid JSON data'),
              'invalidUseful' => $getTranslation('api_invalid_useful', 'is_useful must be "yes" or "no"', 'يجب أن يكون is_useful "yes" أو "no"') ?: ($is_arabic ? 'يجب أن يكون is_useful "yes" أو "no"' : 'is_useful must be "yes" or "no"'),
              'rateLimit' => $getTranslation('api_rate_limit', 'Too many submissions. Please try again later.', 'عدد كبير جدًا من الإرسالات. يرجى المحاولة مرة أخرى لاحقًا.') ?: ($is_arabic ? 'عدد كبير جدًا من الإرسالات. يرجى المحاولة مرة أخرى لاحقًا.' : 'Too many submissions. Please try again later.'),
              'saveFailed' => $getTranslation('api_save_failed', 'Failed to save feedback.', 'فشل حفظ التعليق.') ?: ($is_arabic ? 'فشل حفظ التعليق.' : 'Failed to save feedback.'),
              'success' => $getTranslation('api_success_message', 'Thank you for your feedback!', 'شكرًا لك على تعليقك!') ?: ($is_arabic ? 'شكرًا لك على تعليقك!' : 'Thank you for your feedback!'),
            ],
            'refreshDelay' => $config->get('refresh_delay') ?? 3000,
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0, // Disable cache for dynamic content - CRITICAL for anonymous users
        'contexts' => [
          'url.path',
          'url.query_args',
          'route',
          'user.roles:anonymous', // Cache per anonymous role
          'user.roles:authenticated', // Also cache per authenticated role
          'languages:language_interface', // Critical: Cache per language for proper translations
          'languages:language_content', // Also cache per content language
          'session', // CRITICAL: Cache per session to prevent stale states for anonymous users
        ],
        'tags' => [
          'dga_feedback:submissions',
          'config:dga_feedback.settings',
        ],
      ],
    ];

    return $build;
  }

}

