<?php

namespace Drupal\dga_rating\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\dga_rating\Service\DgaRatingService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for the admin submissions page.
 */
class DgaRatingAdminController extends ControllerBase {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * Constructs a DgaRatingAdminController object.
   *
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(DgaRatingService $rating_service) {
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_rating.service')
    );
  }

  /**
   * Displays all rating submissions and statistics.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   Render array for the admin page.
   */
  public function listSubmissions(Request $request) {
    // Get pagination parameters.
    $page = $request->query->get('page', 0);
    $limit = 50;
    $offset = $page * $limit;

    // Get filter parameters.
    $filters = [];
    if ($entity_type = $request->query->get('entity_type')) {
      $filters['entity_type'] = $entity_type;
    }
    if ($entity_id = $request->query->get('entity_id')) {
      $filters['entity_id'] = $entity_id;
    }
    if ($url = $request->query->get('url')) {
      $filters['url'] = $url;
    }
    if ($rating = $request->query->get('rating')) {
      $filters['rating'] = $rating;
    }

    // Get all submissions.
    $submissions = $this->ratingService->getAllSubmissions($limit, $offset, $filters);
    $total_count = $this->ratingService->getSubmissionsCount($filters);
    // Ensure total_count is always an integer.
    $total_count = (int) ($total_count ?? 0);

    // Get overall statistics.
    $overall_stats = $this->ratingService->getOverallStatistics();
    // Ensure stats are properly formatted and never NULL.
    $overall_stats = [
      'average' => $overall_stats['average'] ?? 0.0,
      'count' => $overall_stats['count'] ?? 0,
    ];

    $rating_distribution = $this->ratingService->getRatingDistribution();
    // Ensure distribution values are integers, not NULL.
    $rating_distribution = array_map(function($count) {
      return (int) ($count ?? 0);
    }, $rating_distribution);

    // Process submissions for display.
    // Ensure all values are strings or have defaults to prevent NULL issues in Views/templates.
    $processed_submissions = [];
    foreach ($submissions as $submission) {
      $processed_submissions[] = [
        'id' => $submission['id'] ?? 0,
        'rating' => $submission['rating'] ?? 0,
        'feedback' => $submission['feedback'] ?? '',
        'url' => $submission['url'] ?? '/',
        'entity_type' => $submission['entity_type'] ?? '',
        'entity_id' => $submission['entity_id'] ?? 0,
        'user_id' => $submission['user_id'] ?? NULL,
        'ip_address' => $submission['ip_address'] ?? '',
        'created' => $submission['created'] ?? time(),
        'created_formatted' => isset($submission['created']) ? date('Y-m-d H:i:s', $submission['created']) : date('Y-m-d H:i:s'),
      ];
    }

    // Build pagination.
    $pager_manager = \Drupal::service('pager.manager');
    $pager_manager->createPager($total_count, $limit);

    return [
      '#theme' => 'dga_rating_admin',
      '#overall_stats' => $overall_stats,
      '#rating_distribution' => $rating_distribution,
      '#submissions' => $processed_submissions,
      '#total_count' => $total_count,
      '#current_page' => $page,
      '#limit' => $limit,
      '#filters' => $filters,
      '#attached' => [
        'library' => [
          'dga_rating/admin',
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

}

