<?php

namespace Drupal\dga_rating\Service;

use Drupal\Core\Database\Connection;

/**
 * Service for managing rating data and calculations.
 */
class DgaRatingService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a DgaRatingService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * Saves a rating submission.
   *
   * @param array $data
   *   Array containing:
   *   - entity_type: The entity type (default: 'node')
   *   - entity_id: The entity ID
   *   - rating: The rating value (1-5)
   *   - feedback: Optional feedback text
   *   - url: The URL where rating was submitted
   *   - user_id: Optional user ID
   *   - ip_address: Optional IP address
   *
   * @return int|false
   *   The ID of the inserted record, or FALSE on failure.
   */
  public function saveRating(array $data) {
    try {
      // Normalize URL - controller already normalizes, but be safe here too.
      $url = $data['url'] ?? '/';
      $url = trim($url);
      if (empty($url) || $url === '/') {
        $url = '/';
      } else {
        // Remove language prefix if still present.
        if (preg_match('#^/[a-z]{2}(/.*)?$#', $url, $matches)) {
          $url = isset($matches[1]) ? $matches[1] : '/';
          if (empty($url)) {
            $url = '/';
          }
        }
        // Remove trailing slash.
        $url = rtrim($url, '/');
        if (empty($url)) {
          $url = '/';
        }
      }
      
      $fields = [
        'entity_type' => $data['entity_type'] ?? 'node',
        'entity_id' => (int) ($data['entity_id'] ?? 0),
        'rating' => (int) ($data['rating'] ?? 0),
        'feedback' => isset($data['feedback']) ? trim($data['feedback']) : '',
        'url' => $url,
        'user_id' => $data['user_id'] ?? NULL, // NULL is valid for anonymous
        'ip_address' => $data['ip_address'] ?? NULL,
        'created' => time(),
      ];
      
      // Validate rating value.
      if ($fields['rating'] < 1 || $fields['rating'] > 5) {
        return FALSE;
      }
      
      // Validate required fields
      if (empty($fields['url'])) {
        return FALSE;
      }

      try {
        // Direct insert - Drupal's database layer auto-commits
        // Transaction wrapper can cause issues with anonymous users
        $rating_id = $this->database->insert('dga_rating')
          ->fields($fields)
          ->execute();

        if ($rating_id) {
          // Invalidate cache tags after saving a rating.
          \Drupal::service('cache_tags.invalidator')->invalidateTags(['dga_rating:submissions']);
        }
      } catch (\Exception $e) {
        \Drupal::logger('dga_rating')->error('Exception saving rating: @message', [
          '@message' => $e->getMessage(),
        ]);
        throw $e;
      }

      return $rating_id;
    }
    catch (\Exception $e) {
      \Drupal::logger('dga_rating')->error('Error saving rating: @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return FALSE;
    }
    catch (\Throwable $e) {
      \Drupal::logger('dga_rating')->error('Fatal error saving rating: @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets rating statistics for a specific entity or URL.
   *
   * @param string|null $entity_type
   *   The entity type to filter by (optional).
   * @param int|null $entity_id
   *   The entity ID to filter by (optional).
   * @param string|null $url
   *   The URL to filter by (optional).
   *
   * @return array
   *   Array with:
   *   - average: Average rating (float)
   *   - count: Total number of ratings (int)
   */
  public function getStatistics($entity_type = NULL, $entity_id = NULL, $url = NULL) {
    $query = $this->database->select('dga_rating', 'r');
    $query->addExpression('AVG(r.rating)', 'average');
    $query->addExpression('COUNT(r.id)', 'count');

    // IMPORTANT: When both entity and URL are provided, match by EITHER:
    // 1. entity_type + entity_id (for authenticated users or saved with entity)
    // 2. OR URL (for anonymous users saved with entity_id=0)
    // This ensures we capture all ratings for a page.
    if ($entity_type && $entity_id !== NULL && $entity_id > 0 && $url) {
      // Match by entity OR URL - this handles both authenticated and anonymous ratings.
      $or_group = $query->orConditionGroup();
      $or_group->condition(
        $query->andConditionGroup()
          ->condition('r.entity_type', $entity_type)
          ->condition('r.entity_id', $entity_id)
      );
      
      // Also match by URL (normalized)
      $normalized_url = rtrim($url, '/') ?: '/';
      if ($normalized_url === '/') {
        $or_group->condition('r.url', '/');
      } else {
        $or_group->condition(
          $query->orConditionGroup()
            ->condition('r.url', $normalized_url)
            ->condition('r.url', $normalized_url . '/')
        );
      }
      
      $query->condition($or_group);
    }
    // Only match by entity if entity_id is provided and > 0 (valid entity) and no URL.
    elseif ($entity_type && $entity_id !== NULL && $entity_id > 0) {
      $query->condition('r.entity_type', $entity_type);
      $query->condition('r.entity_id', $entity_id);
    }
    elseif ($url) {
      // Normalize URL: remove trailing slash and ensure consistent format.
      $normalized_url = rtrim($url, '/') ?: '/';
      
      // CRITICAL: Query for ALL URL variations to match what was saved
      // Anonymous users save with normalized URL (no /en/ prefix)
      // But pages might be accessed with /en/ prefix
      $conditions = $query->orConditionGroup();
      
      // Match exact normalized URL (what should be saved)
      $conditions->condition('r.url', $normalized_url);
      
      // Match with trailing slash
      if ($normalized_url !== '/') {
        $conditions->condition('r.url', $normalized_url . '/');
      }
      
      // If URL has language prefix (/en/xxx), also match without prefix (what anonymous saves)
      if (preg_match('#^/[a-z]{2}/(.+)$#', $normalized_url, $matches)) {
        $without_lang = '/' . $matches[1];
        $conditions->condition('r.url', $without_lang);
        if ($without_lang !== '/') {
          $conditions->condition('r.url', $without_lang . '/');
        }
      }
      // If URL doesn't have language prefix, also match with common prefixes
      else if ($normalized_url !== '/') {
        $lang_prefixes = ['/en', '/ar']; // Add more if needed
        foreach ($lang_prefixes as $prefix) {
          $with_lang = $prefix . $normalized_url;
          $conditions->condition('r.url', $with_lang);
          $conditions->condition('r.url', $with_lang . '/');
        }
      }
      
      $query->condition($conditions);
    }

    $result = $query->execute()->fetchObject();

    // Handle empty result set.
    if (!$result || $result->count == 0) {
      return [
        'average' => 0.0,
        'count' => 0,
      ];
    }

    return [
      'average' => $result->average ? (float) $result->average : 0.0,
      'count' => (int) $result->count,
    ];
  }

  /**
   * Gets rating statistics for the current page URL.
   *
   * @param string $url
   *   The URL path.
   *
   * @return array
   *   Array with average and count.
   */
  public function getStatisticsByUrl($url) {
    return $this->getStatistics(NULL, NULL, $url);
  }

  /**
   * Gets rating statistics for a specific entity.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return array
   *   Array with average and count.
   */
  public function getStatisticsByEntity($entity_type, $entity_id) {
    return $this->getStatistics($entity_type, $entity_id, NULL);
  }

  /**
   * Gets overall statistics across all ratings.
   *
   * @return array
   *   Array with average and count for all ratings.
   */
  public function getOverallStatistics() {
    return $this->getStatistics(NULL, NULL, NULL);
  }

  /**
   * Gets all rating submissions with pagination.
   *
   * @param int $limit
   *   Number of items per page.
   * @param int $offset
   *   Offset for pagination.
   * @param array $filters
   *   Optional filters (entity_type, entity_id, url, rating, date_from, date_to).
   *
   * @return array
   *   Array of rating submissions.
   */
  public function getAllSubmissions($limit = 50, $offset = 0, array $filters = []) {
    $query = $this->database->select('dga_rating', 'r');
    $query->fields('r');
    $query->orderBy('r.created', 'DESC');
    $query->range($offset, $limit);

    // Apply filters.
    if (!empty($filters['entity_type'])) {
      $query->condition('r.entity_type', $filters['entity_type']);
    }
    if (!empty($filters['entity_id'])) {
      $query->condition('r.entity_id', (int) $filters['entity_id']);
    }
    if (!empty($filters['url'])) {
      $query->condition('r.url', '%' . $this->database->escapeLike($filters['url']) . '%', 'LIKE');
    }
    if (!empty($filters['rating'])) {
      $query->condition('r.rating', (int) $filters['rating']);
    }
    if (!empty($filters['date_from'])) {
      $query->condition('r.created', $filters['date_from'], '>=');
    }
    if (!empty($filters['date_to'])) {
      $query->condition('r.created', $filters['date_to'], '<=');
    }

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets total count of submissions with optional filters.
   *
   * @param array $filters
   *   Optional filters (same as getAllSubmissions).
   *
   * @return int
   *   Total count of submissions.
   */
  public function getSubmissionsCount(array $filters = []) {
    $query = $this->database->select('dga_rating', 'r');
    $query->addExpression('COUNT(r.id)', 'count');

    // Apply same filters as getAllSubmissions.
    if (!empty($filters['entity_type'])) {
      $query->condition('r.entity_type', $filters['entity_type']);
    }
    if (!empty($filters['entity_id'])) {
      $query->condition('r.entity_id', (int) $filters['entity_id']);
    }
    if (!empty($filters['url'])) {
      $query->condition('r.url', '%' . $this->database->escapeLike($filters['url']) . '%', 'LIKE');
    }
    if (!empty($filters['rating'])) {
      $query->condition('r.rating', (int) $filters['rating']);
    }
    if (!empty($filters['date_from'])) {
      $query->condition('r.created', $filters['date_from'], '>=');
    }
    if (!empty($filters['date_to'])) {
      $query->condition('r.created', $filters['date_to'], '<=');
    }

    $result = $query->execute()->fetchField();
    return (int) $result;
  }

  /**
   * Gets statistics grouped by rating value (1-5).
   *
   * @return array
   *   Array keyed by rating value (1-5) with count for each.
   */
  public function getRatingDistribution() {
    $query = $this->database->select('dga_rating', 'r');
    $query->addField('r', 'rating');
    $query->addExpression('COUNT(r.id)', 'count');
    $query->groupBy('r.rating');
    $query->orderBy('r.rating', 'ASC');

    $results = $query->execute()->fetchAll();
    $distribution = [1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0];

    foreach ($results as $result) {
      $rating = (int) $result->rating;
      if ($rating >= 1 && $rating <= 5) {
        $distribution[$rating] = (int) $result->count;
      }
    }

    return $distribution;
  }

}


