<?php

namespace Drupal\dga_rating\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dga_rating\Service\DgaRatingService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for editing a rating submission.
 */
class DgaRatingEditForm extends FormBase {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * The submission ID.
   *
   * @var int
   */
  protected $submissionId;

  /**
   * Constructs a DgaRatingEditForm object.
   *
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(DgaRatingService $rating_service) {
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_rating.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_rating_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->submissionId = $id;
    
    $submission = $this->ratingService->getSubmissionById($id);
    
    if (!$submission) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    $form['rating'] = [
      '#type' => 'select',
      '#title' => $this->t('Rating'),
      '#options' => [
        1 => $this->t('1 star'),
        2 => $this->t('2 stars'),
        3 => $this->t('3 stars'),
        4 => $this->t('4 stars'),
        5 => $this->t('5 stars'),
      ],
      '#default_value' => $submission['rating'] ?? 0,
      '#required' => TRUE,
    ];

    $form['feedback'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Feedback'),
      '#default_value' => $submission['feedback'] ?? '',
      '#rows' => 5,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dga_rating.admin'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    
    $success = $this->ratingService->updateSubmission($this->submissionId, [
      'rating' => $values['rating'],
      'feedback' => $values['feedback'],
    ]);

    if ($success) {
      $this->messenger()->addMessage($this->t('Rating submission updated successfully.'));
      $form_state->setRedirect('dga_rating.admin');
    } else {
      $this->messenger()->addError($this->t('Failed to update rating submission.'));
    }
  }

}

