<?php

namespace Drupal\dga_rating\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\dga_rating\Service\DgaRatingService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for the admin submissions page.
 */
class DgaRatingAdminController extends ControllerBase {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * Constructs a DgaRatingAdminController object.
   *
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(DgaRatingService $rating_service) {
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_rating.service')
    );
  }

  /**
   * Displays all rating submissions and statistics.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   Render array for the admin page.
   */
  public function listSubmissions(Request $request) {
    // Get pagination parameters.
    $page = $request->query->get('page', 0);
    $limit = 50;
    $offset = $page * $limit;

    // Get sort parameters.
    $sort_by = $request->query->get('sort', 'created');
    $sort_direction = $request->query->get('order', 'DESC');

    // Get filter parameters.
    $filters = [];
    if ($id = $request->query->get('id')) {
      $filters['id'] = (int) $id;
    }
    if ($entity_type = $request->query->get('entity_type')) {
      $filters['entity_type'] = $entity_type;
    }
    if ($entity_id = $request->query->get('entity_id')) {
      $filters['entity_id'] = (int) $entity_id;
    }
    if ($url = $request->query->get('url')) {
      $filters['url'] = $url;
    }
    if ($rating = $request->query->get('rating')) {
      $filters['rating'] = (int) $rating;
    }
    if ($feedback = $request->query->get('feedback')) {
      $filters['feedback'] = $feedback;
    }
    if ($user_id = $request->query->get('user_id')) {
      $filters['user_id'] = (int) $user_id;
    }
    if ($ip_address = $request->query->get('ip_address')) {
      $filters['ip_address'] = $ip_address;
    }
    if ($date_from = $request->query->get('date_from')) {
      $filters['date_from'] = strtotime($date_from);
    }
    if ($date_to = $request->query->get('date_to')) {
      $filters['date_to'] = strtotime($date_to . ' 23:59:59');
    }

    // Get URL statistics grouping parameters.
    $url_stats_limit = (int) $request->query->get('url_stats_limit', 20);
    $url_stats_order_by = $request->query->get('url_stats_order', 'count');
    $url_stats_order_direction = $request->query->get('url_stats_order_dir', 'DESC');

    // Get all submissions.
    $submissions = $this->ratingService->getAllSubmissions($limit, $offset, $filters, $sort_by, $sort_direction);
    $total_count = $this->ratingService->getSubmissionsCount($filters);
    // Ensure total_count is always an integer.
    $total_count = (int) ($total_count ?? 0);

    // Get overall statistics.
    $overall_stats = $this->ratingService->getOverallStatistics();
    // Ensure stats are properly formatted and never NULL.
    $overall_stats = [
      'average' => $overall_stats['average'] ?? 0.0,
      'count' => $overall_stats['count'] ?? 0,
    ];

    $rating_distribution = $this->ratingService->getRatingDistribution();
    // Ensure distribution values are integers, not NULL.
    $rating_distribution = array_map(function($count) {
      return (int) ($count ?? 0);
    }, $rating_distribution);

    // Get statistics grouped by URL.
    $url_statistics = $this->ratingService->getStatisticsGroupedByUrl($url_stats_limit, $url_stats_order_by, $url_stats_order_direction);
    
    // Get total count of unique URLs for average calculation.
    $unique_url_count = $this->ratingService->getUniqueUrlCount();

    // Get additional statistics for advanced blocks.
    $top_rated_page = $this->ratingService->getTopRatedPage();
    $most_reviewed_page = $this->ratingService->getMostReviewedPage();
    $recent_activity = $this->ratingService->getRecentActivity();
    $positive_percentage = $this->ratingService->getPositiveRatingsPercentage();
    $user_type_stats = $this->ratingService->getSubmissionsByUserType();

    // Process submissions for display.
    // Ensure all values are strings or have defaults to prevent NULL issues in Views/templates.
    $processed_submissions = [];
    foreach ($submissions as $submission) {
      $processed_submissions[] = [
        'id' => $submission['id'] ?? 0,
        'rating' => $submission['rating'] ?? 0,
        'feedback' => $submission['feedback'] ?? '',
        'url' => $submission['url'] ?? '/',
        'entity_type' => $submission['entity_type'] ?? '',
        'entity_id' => $submission['entity_id'] ?? 0,
        'user_id' => $submission['user_id'] ?? NULL,
        'ip_address' => $submission['ip_address'] ?? '',
        'created' => $submission['created'] ?? time(),
        'created_formatted' => isset($submission['created']) ? date('Y-m-d H:i:s', $submission['created']) : date('Y-m-d H:i:s'),
      ];
    }

    // Build pagination.
    $pager_manager = \Drupal::service('pager.manager');
    $pager_manager->createPager($total_count, $limit);

    return [
      '#theme' => 'dga_rating_admin',
      '#overall_stats' => $overall_stats,
      '#rating_distribution' => $rating_distribution,
      '#url_statistics' => $url_statistics,
      '#unique_url_count' => $unique_url_count,
      '#submissions' => $processed_submissions,
      '#total_count' => $total_count,
      '#current_page' => $page,
      '#limit' => $limit,
      '#filters' => $filters,
      '#sort_by' => $sort_by,
      '#sort_direction' => $sort_direction,
      '#url_stats_limit' => $url_stats_limit,
      '#url_stats_order_by' => $url_stats_order_by,
      '#url_stats_order_direction' => $url_stats_order_direction,
      '#top_rated_page' => $top_rated_page,
      '#most_reviewed_page' => $most_reviewed_page,
      '#recent_activity' => $recent_activity,
      '#positive_percentage' => $positive_percentage,
      '#user_type_stats' => $user_type_stats,
      '#attached' => [
        'library' => [
          'dga_rating/admin',
        ],
        'drupalSettings' => [
          'dgaRating' => [
            'bulkDeleteUrl' => Url::fromRoute('dga_rating.admin.bulk_delete')->toString(),
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

  /**
   * Displays edit form for a rating submission.
   *
   * @param int $id
   *   The rating submission ID.
   *
   * @return array
   *   Render array for the edit form.
   */
  public function editSubmission($id) {
    // Build form using Drupal's form API.
    $form = \Drupal::formBuilder()->getForm('\Drupal\dga_rating\Form\DgaRatingEditForm', $id);
    
    return $form;
  }

  /**
   * Handles bulk delete operation.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to the admin page.
   */
  public function bulkDelete(Request $request) {
    // Get submissions array from request
    // When form inputs have name="submissions[]", Symfony parses them as an array
    $ids = [];
    $request_data = $request->request->all();
    
    if (isset($request_data['submissions'])) {
      $submissions = $request_data['submissions'];
      if (is_array($submissions)) {
        $ids = $submissions;
      } elseif (is_scalar($submissions)) {
        // Handle case where it might be a single value
        $ids = [$submissions];
      }
    }
    
    if (empty($ids)) {
      $this->messenger()->addError($this->t('No submissions selected for deletion.'));
      return new RedirectResponse(Url::fromRoute('dga_rating.admin')->toString());
    }

    $deleted_count = $this->ratingService->bulkDeleteSubmissions($ids);

    if ($deleted_count > 0) {
      $this->messenger()->addMessage($this->t('Deleted @count submission(s).', ['@count' => $deleted_count]));
    } else {
      $this->messenger()->addError($this->t('Failed to delete submissions.'));
    }

    return new RedirectResponse(Url::fromRoute('dga_rating.admin')->toString());
  }

}

