<?php

namespace Drupal\dga_rating\Service;

use Drupal\Core\Database\Connection;

/**
 * Service for managing rating data and calculations.
 */
class DgaRatingService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a DgaRatingService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * Saves a rating submission.
   *
   * @param array $data
   *   Array containing:
   *   - entity_type: The entity type (default: 'node')
   *   - entity_id: The entity ID
   *   - rating: The rating value (1-5)
   *   - feedback: Optional feedback text
   *   - url: The URL where rating was submitted
   *   - user_id: Optional user ID
   *   - ip_address: Optional IP address
   *
   * @return int|false
   *   The ID of the inserted record, or FALSE on failure.
   */
  public function saveRating(array $data) {
    try {
      // Normalize URL - controller already normalizes, but be safe here too.
      $url = $data['url'] ?? '/';
      $url = trim($url);
      if (empty($url) || $url === '/') {
        $url = '/';
      } else {
        // Remove language prefix if still present.
        if (preg_match('#^/[a-z]{2}(/.*)?$#', $url, $matches)) {
          $url = isset($matches[1]) ? $matches[1] : '/';
          if (empty($url)) {
            $url = '/';
          }
        }
        // Remove trailing slash.
        $url = rtrim($url, '/');
        if (empty($url)) {
          $url = '/';
        }
      }
      
      // SECURITY: Sanitize feedback text (already sanitized in controller, but double-check here)
      // Get configurable limits from settings
      $config = \Drupal::config('dga_rating.settings');
      $feedback_max_length = (int) ($config->get('feedback_max_length') ?? 5000);
      
      $feedback_text = isset($data['feedback']) ? trim($data['feedback']) : '';
      $feedback_text = strip_tags($feedback_text);
      if (strlen($feedback_text) > $feedback_max_length) {
        $feedback_text = substr($feedback_text, 0, $feedback_max_length);
      }

      $fields = [
        'entity_type' => $data['entity_type'] ?? 'node',
        'entity_id' => (int) ($data['entity_id'] ?? 0),
        'rating' => (int) ($data['rating'] ?? 0),
        'feedback' => $feedback_text,
        'url' => $url,
        'user_id' => $data['user_id'] ?? NULL, // NULL is valid for anonymous
        'ip_address' => $data['ip_address'] ?? NULL,
        'created' => time(),
      ];
      
      // Validate rating value.
      if ($fields['rating'] < 1 || $fields['rating'] > 5) {
        return FALSE;
      }
      
      // Validate required fields
      if (empty($fields['url'])) {
        return FALSE;
      }

      try {
        // Direct insert - Drupal's database layer auto-commits
        // Transaction wrapper can cause issues with anonymous users
        $rating_id = $this->database->insert('dga_rating')
          ->fields($fields)
          ->execute();

        if ($rating_id) {
          // Invalidate cache tags after saving a rating.
          \Drupal::service('cache_tags.invalidator')->invalidateTags(['dga_rating:submissions']);
        }
      } catch (\Exception $e) {
        \Drupal::logger('dga_rating')->error('Exception saving rating: @message', [
          '@message' => $e->getMessage(),
        ]);
        throw $e;
      }

      return $rating_id;
    }
    catch (\Exception $e) {
      \Drupal::logger('dga_rating')->error('Error saving rating: @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return FALSE;
    }
    catch (\Throwable $e) {
      \Drupal::logger('dga_rating')->error('Fatal error saving rating: @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets rating statistics for a specific entity or URL.
   *
   * @param string|null $entity_type
   *   The entity type to filter by (optional).
   * @param int|null $entity_id
   *   The entity ID to filter by (optional).
   * @param string|null $url
   *   The URL to filter by (optional).
   *
   * @return array
   *   Array with:
   *   - average: Average rating (float)
   *   - count: Total number of ratings (int)
   */
  public function getStatistics($entity_type = NULL, $entity_id = NULL, $url = NULL) {
    $query = $this->database->select('dga_rating', 'r');
    $query->addExpression('AVG(r.rating)', 'average');
    $query->addExpression('COUNT(r.id)', 'count');

    // IMPORTANT: When both entity and URL are provided, match by EITHER:
    // 1. entity_type + entity_id (for authenticated users or saved with entity)
    // 2. OR URL (for anonymous users saved with entity_id=0)
    // This ensures we capture all ratings for a page.
    if ($entity_type && $entity_id !== NULL && $entity_id > 0 && $url) {
      // Match by entity OR URL - this handles both authenticated and anonymous ratings.
      $or_group = $query->orConditionGroup();
      $or_group->condition(
        $query->andConditionGroup()
          ->condition('r.entity_type', $entity_type)
          ->condition('r.entity_id', $entity_id)
      );
      
      // Also match by URL (normalized)
      $normalized_url = rtrim($url, '/') ?: '/';
      if ($normalized_url === '/') {
        $or_group->condition('r.url', '/');
      } else {
        $or_group->condition(
          $query->orConditionGroup()
            ->condition('r.url', $normalized_url)
            ->condition('r.url', $normalized_url . '/')
        );
      }
      
      $query->condition($or_group);
    }
    // Only match by entity if entity_id is provided and > 0 (valid entity) and no URL.
    elseif ($entity_type && $entity_id !== NULL && $entity_id > 0) {
      $query->condition('r.entity_type', $entity_type);
      $query->condition('r.entity_id', $entity_id);
    }
    elseif ($url) {
      // Normalize URL: remove trailing slash and ensure consistent format.
      $normalized_url = rtrim($url, '/') ?: '/';
      
      // CRITICAL: Query for ALL URL variations to match what was saved
      // Anonymous users save with normalized URL (no /en/ prefix)
      // But pages might be accessed with /en/ prefix
      $conditions = $query->orConditionGroup();
      
      // Match exact normalized URL (what should be saved)
      $conditions->condition('r.url', $normalized_url);
      
      // Match with trailing slash
      if ($normalized_url !== '/') {
        $conditions->condition('r.url', $normalized_url . '/');
      }
      
      // If URL has language prefix (/en/xxx), also match without prefix (what anonymous saves)
      if (preg_match('#^/[a-z]{2}/(.+)$#', $normalized_url, $matches)) {
        $without_lang = '/' . $matches[1];
        $conditions->condition('r.url', $without_lang);
        if ($without_lang !== '/') {
          $conditions->condition('r.url', $without_lang . '/');
        }
      }
      // If URL doesn't have language prefix, also match with common prefixes
      else if ($normalized_url !== '/') {
        $lang_prefixes = ['/en', '/ar']; // Add more if needed
        foreach ($lang_prefixes as $prefix) {
          $with_lang = $prefix . $normalized_url;
          $conditions->condition('r.url', $with_lang);
          $conditions->condition('r.url', $with_lang . '/');
        }
      }
      
      $query->condition($conditions);
    }

    $result = $query->execute()->fetchObject();

    // Handle empty result set.
    if (!$result || $result->count == 0) {
      return [
        'average' => 0.0,
        'count' => 0,
      ];
    }

    return [
      'average' => $result->average ? (float) $result->average : 0.0,
      'count' => (int) $result->count,
    ];
  }

  /**
   * Gets rating statistics for the current page URL.
   *
   * @param string $url
   *   The URL path.
   *
   * @return array
   *   Array with average and count.
   */
  public function getStatisticsByUrl($url) {
    return $this->getStatistics(NULL, NULL, $url);
  }

  /**
   * Gets rating statistics for a specific entity.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return array
   *   Array with average and count.
   */
  public function getStatisticsByEntity($entity_type, $entity_id) {
    return $this->getStatistics($entity_type, $entity_id, NULL);
  }

  /**
   * Gets overall statistics across all ratings.
   *
   * @return array
   *   Array with average and count for all ratings.
   */
  public function getOverallStatistics() {
    return $this->getStatistics(NULL, NULL, NULL);
  }

  /**
   * Gets all rating submissions with pagination.
   *
   * @param int $limit
   *   Number of items per page.
   * @param int $offset
   *   Offset for pagination.
   * @param array $filters
   *   Optional filters (entity_type, entity_id, url, rating, date_from, date_to).
   * @param string $sort_by
   *   Field to sort by (default: 'created').
   * @param string $sort_direction
   *   Sort direction: 'ASC' or 'DESC' (default: 'DESC').
   *
   * @return array
   *   Array of rating submissions.
   */
  public function getAllSubmissions($limit = 50, $offset = 0, array $filters = [], $sort_by = 'created', $sort_direction = 'DESC') {
    $query = $this->database->select('dga_rating', 'r');
    $query->fields('r');
    
    // Validate sort parameters.
    $valid_sort_fields = ['id', 'rating', 'created', 'url', 'entity_type', 'entity_id', 'user_id'];
    if (!in_array($sort_by, $valid_sort_fields)) {
      $sort_by = 'created';
    }
    
    $sort_direction = strtoupper($sort_direction);
    if ($sort_direction !== 'ASC' && $sort_direction !== 'DESC') {
      $sort_direction = 'DESC';
    }
    
    $query->orderBy('r.' . $sort_by, $sort_direction);
    $query->range($offset, $limit);

    // Apply filters.
    if (isset($filters['id']) && $filters['id'] > 0) {
      $query->condition('r.id', $filters['id']);
    }
    if (!empty($filters['entity_type'])) {
      $query->condition('r.entity_type', $filters['entity_type']);
    }
    if (isset($filters['entity_id']) && $filters['entity_id'] > 0) {
      $query->condition('r.entity_id', (int) $filters['entity_id']);
    }
    if (!empty($filters['url'])) {
      $query->condition('r.url', '%' . $this->database->escapeLike($filters['url']) . '%', 'LIKE');
    }
    if (isset($filters['rating']) && $filters['rating'] > 0) {
      $query->condition('r.rating', (int) $filters['rating']);
    }
    if (!empty($filters['feedback'])) {
      $query->condition('r.feedback', '%' . $this->database->escapeLike($filters['feedback']) . '%', 'LIKE');
    }
    if (isset($filters['user_id']) && $filters['user_id'] !== NULL && $filters['user_id'] !== '') {
      $query->condition('r.user_id', (int) $filters['user_id']);
    }
    if (!empty($filters['ip_address'])) {
      $query->condition('r.ip_address', '%' . $this->database->escapeLike($filters['ip_address']) . '%', 'LIKE');
    }
    if (!empty($filters['date_from'])) {
      $query->condition('r.created', $filters['date_from'], '>=');
    }
    if (!empty($filters['date_to'])) {
      $query->condition('r.created', $filters['date_to'], '<=');
    }

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets total count of submissions with optional filters.
   *
   * @param array $filters
   *   Optional filters (same as getAllSubmissions).
   *
   * @return int
   *   Total count of submissions.
   */
  public function getSubmissionsCount(array $filters = []) {
    $query = $this->database->select('dga_rating', 'r');
    $query->addExpression('COUNT(r.id)', 'count');

    // Apply same filters as getAllSubmissions.
    if (isset($filters['id']) && $filters['id'] > 0) {
      $query->condition('r.id', $filters['id']);
    }
    if (!empty($filters['entity_type'])) {
      $query->condition('r.entity_type', $filters['entity_type']);
    }
    if (isset($filters['entity_id']) && $filters['entity_id'] > 0) {
      $query->condition('r.entity_id', (int) $filters['entity_id']);
    }
    if (!empty($filters['url'])) {
      $query->condition('r.url', '%' . $this->database->escapeLike($filters['url']) . '%', 'LIKE');
    }
    if (isset($filters['rating']) && $filters['rating'] > 0) {
      $query->condition('r.rating', (int) $filters['rating']);
    }
    if (!empty($filters['feedback'])) {
      $query->condition('r.feedback', '%' . $this->database->escapeLike($filters['feedback']) . '%', 'LIKE');
    }
    if (isset($filters['user_id']) && $filters['user_id'] !== NULL && $filters['user_id'] !== '') {
      $query->condition('r.user_id', (int) $filters['user_id']);
    }
    if (!empty($filters['ip_address'])) {
      $query->condition('r.ip_address', '%' . $this->database->escapeLike($filters['ip_address']) . '%', 'LIKE');
    }
    if (!empty($filters['date_from'])) {
      $query->condition('r.created', $filters['date_from'], '>=');
    }
    if (!empty($filters['date_to'])) {
      $query->condition('r.created', $filters['date_to'], '<=');
    }

    $result = $query->execute()->fetchField();
    return (int) $result;
  }

  /**
   * Gets statistics grouped by rating value (1-5).
   *
   * @return array
   *   Array keyed by rating value (1-5) with count for each.
   */
  public function getRatingDistribution() {
    $query = $this->database->select('dga_rating', 'r');
    $query->addField('r', 'rating');
    $query->addExpression('COUNT(r.id)', 'count');
    $query->groupBy('r.rating');
    $query->orderBy('r.rating', 'ASC');

    $results = $query->execute()->fetchAll();
    $distribution = [1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0];

    foreach ($results as $result) {
      $rating = (int) $result->rating;
      if ($rating >= 1 && $rating <= 5) {
        $distribution[$rating] = (int) $result->count;
      }
    }

    return $distribution;
  }

  /**
   * Gets a single submission by ID.
   *
   * @param int $id
   *   The submission ID.
   *
   * @return array|false
   *   Array of submission data or FALSE if not found.
   */
  public function getSubmissionById($id) {
    $query = $this->database->select('dga_rating', 'r');
    $query->fields('r');
    $query->condition('r.id', (int) $id);
    $result = $query->execute()->fetchAssoc();
    return $result ? $result : FALSE;
  }

  /**
   * Deletes a submission by ID.
   *
   * @param int $id
   *   The submission ID.
   *
   * @return bool
   *   TRUE if deleted, FALSE otherwise.
   */
  public function deleteSubmission($id) {
    try {
      $deleted = $this->database->delete('dga_rating')
        ->condition('id', (int) $id)
        ->execute();
      
      if ($deleted) {
        // Invalidate cache tags after deleting.
        \Drupal::service('cache_tags.invalidator')->invalidateTags(['dga_rating:submissions']);
      }
      
      return $deleted > 0;
    } catch (\Exception $e) {
      \Drupal::logger('dga_rating')->error('Error deleting rating: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Deletes multiple submissions by IDs.
   *
   * @param array $ids
   *   Array of submission IDs.
   *
   * @return int
   *   Number of submissions deleted.
   */
  public function bulkDeleteSubmissions(array $ids) {
    if (empty($ids)) {
      return 0;
    }

    try {
      // Convert all IDs to integers and filter out invalid values.
      $ids = array_filter(array_map('intval', $ids), function($id) {
        return $id > 0;
      });

      if (empty($ids)) {
        return 0;
      }

      $deleted = $this->database->delete('dga_rating')
        ->condition('id', $ids, 'IN')
        ->execute();
      
      if ($deleted > 0) {
        // Invalidate cache tags after deleting.
        \Drupal::service('cache_tags.invalidator')->invalidateTags(['dga_rating:submissions']);
      }
      
      return $deleted;
    } catch (\Exception $e) {
      \Drupal::logger('dga_rating')->error('Error bulk deleting ratings: @message', [
        '@message' => $e->getMessage(),
      ]);
      return 0;
    }
  }

  /**
   * Updates a submission.
   *
   * @param int $id
   *   The submission ID.
   * @param array $data
   *   Array of fields to update.
   *
   * @return bool
   *   TRUE if updated, FALSE otherwise.
   */
  public function updateSubmission($id, array $data) {
    try {
      $fields = [];
      if (isset($data['rating'])) {
        $fields['rating'] = (int) $data['rating'];
        if ($fields['rating'] < 1 || $fields['rating'] > 5) {
          return FALSE;
        }
      }
      if (isset($data['feedback'])) {
        $fields['feedback'] = trim($data['feedback']);
      }
      
      if (empty($fields)) {
        return FALSE;
      }
      
      $updated = $this->database->update('dga_rating')
        ->fields($fields)
        ->condition('id', (int) $id)
        ->execute();
      
      if ($updated !== FALSE) {
        // Invalidate cache tags after updating.
        \Drupal::service('cache_tags.invalidator')->invalidateTags(['dga_rating:submissions']);
      }
      
      return $updated !== FALSE;
    } catch (\Exception $e) {
      \Drupal::logger('dga_rating')->error('Error updating rating: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets the total count of unique URLs.
   *
   * @return int
   *   Total number of unique URLs.
   */
  public function getUniqueUrlCount() {
    $query = $this->database->select('dga_rating', 'r');
    $query->addExpression('COUNT(DISTINCT r.url)', 'count');
    $result = $query->execute()->fetchField();
    return (int) $result;
  }

  /**
   * Gets statistics grouped by URL.
   *
   * @param int $limit
   *   Maximum number of URLs to return (default: 50).
   * @param string $order_by
   *   Field to order by: 'count', 'average', or 'url' (default: 'count').
   * @param string $order_direction
   *   Order direction: 'ASC' or 'DESC' (default: 'DESC').
   *
   * @return array
   *   Array of URL statistics, each with:
   *   - url: The URL path
   *   - average: Average rating (float)
   *   - count: Total number of ratings (int)
   *   - distribution: Array of rating counts by value (1-5)
   */
  public function getStatisticsGroupedByUrl($limit = 50, $order_by = 'count', $order_direction = 'DESC') {
    $query = $this->database->select('dga_rating', 'r');
    $query->addField('r', 'url');
    $query->addExpression('AVG(r.rating)', 'average');
    $query->addExpression('COUNT(r.id)', 'count');
    $query->groupBy('r.url');

    // Validate order_by parameter.
    $valid_order_fields = ['count', 'average', 'url'];
    if (!in_array($order_by, $valid_order_fields)) {
      $order_by = 'count';
    }

    // Validate order direction.
    $order_direction = strtoupper($order_direction);
    if ($order_direction !== 'ASC' && $order_direction !== 'DESC') {
      $order_direction = 'DESC';
    }

    // Order by the specified field.
    if ($order_by === 'url') {
      $query->orderBy('r.url', $order_direction);
    } else {
      $query->orderBy($order_by, $order_direction);
    }

    $query->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $url_stats = [];
    foreach ($results as $result) {
      $url = $result->url ?? '/';
      $average = $result->average ? (float) $result->average : 0.0;
      $count = (int) $result->count;

      // Get rating distribution for this URL.
      $distribution_query = $this->database->select('dga_rating', 'r2');
      $distribution_query->addField('r2', 'rating');
      $distribution_query->addExpression('COUNT(r2.id)', 'rating_count');
      $distribution_query->condition('r2.url', $url);
      $distribution_query->groupBy('r2.rating');
      $distribution_query->orderBy('r2.rating', 'ASC');

      $distribution_results = $distribution_query->execute()->fetchAll();
      $distribution = [1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0];

      foreach ($distribution_results as $dist_result) {
        $rating = (int) $dist_result->rating;
        if ($rating >= 1 && $rating <= 5) {
          $distribution[$rating] = (int) $dist_result->rating_count;
        }
      }

      $url_stats[] = [
        'url' => $url,
        'average' => $average,
        'count' => $count,
        'distribution' => $distribution,
      ];
    }

    return $url_stats;
  }

  /**
   * Gets top rated page (highest average rating).
   *
   * @return array|null
   *   Array with url, average, and count, or NULL if no ratings exist.
   */
  public function getTopRatedPage() {
    $query = $this->database->select('dga_rating', 'r');
    $query->addField('r', 'url');
    $query->addExpression('AVG(r.rating)', 'average');
    $query->addExpression('COUNT(r.id)', 'count');
    $query->groupBy('r.url');
    // Filter out URLs with less than 1 rating (though this should always be true)
    // We'll handle this in PHP instead to avoid HAVING clause issues
    $query->orderBy('average', 'DESC');
    $query->orderBy('count', 'DESC');
    $query->range(0, 1);

    $result = $query->execute()->fetchObject();
    if ($result && $result->count > 0) {
      return [
        'url' => $result->url ?? '/',
        'average' => (float) $result->average,
        'count' => (int) $result->count,
      ];
    }
    return NULL;
  }

  /**
   * Gets most reviewed page (highest count).
   *
   * @return array|null
   *   Array with url, average, and count, or NULL if no ratings exist.
   */
  public function getMostReviewedPage() {
    $query = $this->database->select('dga_rating', 'r');
    $query->addField('r', 'url');
    $query->addExpression('AVG(r.rating)', 'average');
    $query->addExpression('COUNT(r.id)', 'count');
    $query->groupBy('r.url');
    $query->orderBy('count', 'DESC');
    $query->orderBy('average', 'DESC');
    $query->range(0, 1);

    $result = $query->execute()->fetchObject();
    if ($result && $result->count > 0) {
      return [
        'url' => $result->url ?? '/',
        'average' => (float) $result->average,
        'count' => (int) $result->count,
      ];
    }
    return NULL;
  }

  /**
   * Gets recent activity statistics (last 7 days, last 30 days).
   *
   * @return array
   *   Array with:
   *   - last_7_days: count and average for last 7 days
   *   - last_30_days: count and average for last 30 days
   */
  public function getRecentActivity() {
    $now = time();
    $seven_days_ago = $now - (7 * 24 * 60 * 60);
    $thirty_days_ago = $now - (30 * 24 * 60 * 60);

    // Last 7 days
    $query_7 = $this->database->select('dga_rating', 'r');
    $query_7->addExpression('AVG(r.rating)', 'average');
    $query_7->addExpression('COUNT(r.id)', 'count');
    $query_7->condition('r.created', $seven_days_ago, '>=');
    $result_7 = $query_7->execute()->fetchObject();

    // Last 30 days
    $query_30 = $this->database->select('dga_rating', 'r');
    $query_30->addExpression('AVG(r.rating)', 'average');
    $query_30->addExpression('COUNT(r.id)', 'count');
    $query_30->condition('r.created', $thirty_days_ago, '>=');
    $result_30 = $query_30->execute()->fetchObject();

    return [
      'last_7_days' => [
        'count' => (int) ($result_7->count ?? 0),
        'average' => $result_7->average ? (float) $result_7->average : 0.0,
      ],
      'last_30_days' => [
        'count' => (int) ($result_30->count ?? 0),
        'average' => $result_30->average ? (float) $result_30->average : 0.0,
      ],
    ];
  }

  /**
   * Gets positive ratings percentage (4-5 stars).
   *
   * @return float
   *   Percentage of positive ratings (0-100).
   */
  public function getPositiveRatingsPercentage() {
    $total_query = $this->database->select('dga_rating', 'r');
    $total_query->addExpression('COUNT(r.id)', 'count');
    $total = $total_query->execute()->fetchField();

    if ($total == 0) {
      return 0.0;
    }

    $positive_query = $this->database->select('dga_rating', 'r');
    $positive_query->addExpression('COUNT(r.id)', 'count');
    $positive_query->condition('r.rating', [4, 5], 'IN');
    $positive = $positive_query->execute()->fetchField();

    return ($positive / $total) * 100;
  }

  /**
   * Gets submissions count by user type (anonymous vs authenticated).
   *
   * @return array
   *   Array with:
   *   - anonymous: count
   *   - authenticated: count
   */
  public function getSubmissionsByUserType() {
    $anonymous_query = $this->database->select('dga_rating', 'r');
    $anonymous_query->addExpression('COUNT(r.id)', 'count');
    $anonymous_query->isNull('r.user_id');
    $anonymous_count = (int) $anonymous_query->execute()->fetchField();

    $auth_query = $this->database->select('dga_rating', 'r');
    $auth_query->addExpression('COUNT(r.id)', 'count');
    $auth_query->isNotNull('r.user_id');
    $auth_count = (int) $auth_query->execute()->fetchField();

    return [
      'anonymous' => $anonymous_count,
      'authenticated' => $auth_count,
    ];
  }

}


