/**
 * @file
 * Admin JavaScript for DGA Rating submissions page.
 */

(function (Drupal, once) {
  'use strict';

  /**
   * Initialize table sorting functionality.
   */
  function initTableSorting() {
    const table = document.getElementById('submissions-table');
    if (!table) {
      return;
    }

    const headers = table.querySelectorAll('th.sortable');
    const tbody = table.querySelector('tbody');
    if (!tbody) {
      return;
    }

    headers.forEach(function (header) {
      header.addEventListener('click', function (e) {
        // Don't trigger if clicking on sort indicator
        if (e.target.classList.contains('sort-indicator')) {
          return;
        }

        const sortField = header.getAttribute('data-sort');
        if (!sortField) {
          return;
        }

        // Get current sort parameters from URL
        const urlParams = new URLSearchParams(window.location.search);
        const currentSort = urlParams.get('sort');
        const currentOrder = urlParams.get('order') || 'DESC';

        // Determine new sort direction
        let newOrder = 'DESC';
        if (currentSort === sortField && currentOrder === 'DESC') {
          newOrder = 'ASC';
        }

        // Update URL parameters
        urlParams.set('sort', sortField);
        urlParams.set('order', newOrder);
        urlParams.delete('page'); // Reset to first page when sorting

        // Redirect with new parameters
        window.location.search = urlParams.toString();
      });
    });
  }

  /**
   * Initialize URL statistics controls.
   */
  function initUrlStatsControls() {
    const orderSelect = document.querySelector('[data-url-stats-order]');
    const orderDirSelect = document.querySelector('[data-url-stats-order-dir]');
    const limitSelect = document.querySelector('[data-url-stats-limit]');

    if (orderSelect) {
      orderSelect.addEventListener('change', function () {
        updateUrlStatsParams();
      });
    }

    if (orderDirSelect) {
      orderDirSelect.addEventListener('change', function () {
        updateUrlStatsParams();
      });
    }

    if (limitSelect) {
      limitSelect.addEventListener('change', function () {
        updateUrlStatsParams();
      });
    }
  }

  /**
   * Update URL parameters for URL statistics and reload.
   */
  function updateUrlStatsParams() {
    const urlParams = new URLSearchParams(window.location.search);
    const orderSelect = document.querySelector('[data-url-stats-order]');
    const orderDirSelect = document.querySelector('[data-url-stats-order-dir]');
    const limitSelect = document.querySelector('[data-url-stats-limit]');

    if (orderSelect) {
      urlParams.set('url_stats_order', orderSelect.value);
    }
    if (orderDirSelect) {
      urlParams.set('url_stats_order_dir', orderDirSelect.value);
    }
    if (limitSelect) {
      urlParams.set('url_stats_limit', limitSelect.value);
    }

    // Reset to first page when changing URL stats parameters
    urlParams.delete('page');

    window.location.search = urlParams.toString();
  }

  /**
   * Initialize filter form enhancements.
   */
  function initFilterForm() {
    const form = document.getElementById('rating-filters-form');
    if (!form) {
      return;
    }

    // Preserve URL stats parameters when submitting filters
    form.addEventListener('submit', function (e) {
      const urlParams = new URLSearchParams(window.location.search);
      const urlStatsOrder = urlParams.get('url_stats_order');
      const urlStatsOrderDir = urlParams.get('url_stats_order_dir');
      const urlStatsLimit = urlParams.get('url_stats_limit');

      if (urlStatsOrder) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'url_stats_order';
        input.value = urlStatsOrder;
        form.appendChild(input);
      }

      if (urlStatsOrderDir) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'url_stats_order_dir';
        input.value = urlStatsOrderDir;
        form.appendChild(input);
      }

      if (urlStatsLimit) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'url_stats_limit';
        input.value = urlStatsLimit;
        form.appendChild(input);
      }
    });
  }

  /**
   * Initialize dropbutton functionality.
   */
  function initDropbuttons() {
    const dropbuttons = document.querySelectorAll('.dropbutton-wrapper');
    
    dropbuttons.forEach(function (wrapper) {
      const toggle = wrapper.querySelector('.dropbutton__toggle');
      const widget = wrapper.querySelector('.dropbutton-widget');
      const editLink = wrapper.querySelector('.dropbutton-action:first-child a');
      
      if (!toggle || !widget) {
        return;
      }
      
      // Remove any existing event listeners by cloning and replacing
      const toggleClone = toggle.cloneNode(true);
      toggle.parentNode.replaceChild(toggleClone, toggle);
      
      // Add click handler to toggle button (arrow/caret)
      toggleClone.addEventListener('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        
        const isOpen = widget.classList.contains('open');
        
        // Close all other dropbuttons first
        document.querySelectorAll('.dropbutton-widget.open').forEach(function (openWidget) {
          if (openWidget !== widget) {
            openWidget.classList.remove('open');
            const openToggle = openWidget.querySelector('.dropbutton__toggle');
            if (openToggle) {
              openToggle.setAttribute('aria-expanded', 'false');
            }
          }
        });
        
        // Toggle current dropbutton
        if (isOpen) {
          widget.classList.remove('open');
          toggleClone.setAttribute('aria-expanded', 'false');
        } else {
          widget.classList.add('open');
          toggleClone.setAttribute('aria-expanded', 'true');
        }
      }, true); // Use capture phase to ensure it fires first
      
      // Ensure edit link doesn't interfere with toggle
      if (editLink) {
        editLink.addEventListener('click', function (e) {
          // Don't prevent default - let edit link work normally
          // But stop propagation if clicking on toggle area
          if (e.target.closest('.dropbutton-toggle') || e.target.closest('.dropbutton__toggle')) {
            e.preventDefault();
            e.stopPropagation();
            // Manually trigger toggle click
            toggleClone.click();
          }
        });
      }
    });
    
    // Close dropbuttons when clicking outside
    document.addEventListener('click', function (e) {
      if (!e.target.closest('.dropbutton-wrapper')) {
        document.querySelectorAll('.dropbutton-widget.open').forEach(function (widget) {
          widget.classList.remove('open');
          const toggle = widget.querySelector('.dropbutton__toggle');
          if (toggle) {
            toggle.setAttribute('aria-expanded', 'false');
          }
        });
      }
    });
  }

  /**
   * Initialize bulk operations functionality.
   */
  function initBulkOperations() {
    const selectAllCheckbox = document.getElementById('select-all-checkbox');
    const checkboxes = document.querySelectorAll('.submission-checkbox');
    const bulkOperationsWrapper = document.getElementById('bulk-operations-wrapper');
    const bulkOperationsCount = document.getElementById('bulk-operations-count');
    const bulkActionSelect = document.getElementById('bulk-action-select');
    const bulkActionSubmit = document.getElementById('bulk-action-submit');

    if (!selectAllCheckbox || !bulkOperationsWrapper) {
      return;
    }

    // Update bulk operations UI
    function updateBulkOperations() {
      const checked = document.querySelectorAll('.submission-checkbox:checked');
      const count = checked.length;

      if (count > 0) {
        bulkOperationsWrapper.style.display = 'block';
        bulkOperationsCount.textContent = count;
        
        // Update select all checkbox state
        if (count === checkboxes.length) {
          selectAllCheckbox.checked = true;
          selectAllCheckbox.indeterminate = false;
        } else if (count > 0) {
          selectAllCheckbox.checked = false;
          selectAllCheckbox.indeterminate = true;
        } else {
          selectAllCheckbox.checked = false;
          selectAllCheckbox.indeterminate = false;
        }

        // Enable/disable submit button based on action selection
        if (bulkActionSelect && bulkActionSubmit) {
          bulkActionSubmit.disabled = !bulkActionSelect.value;
        }
      } else {
        bulkOperationsWrapper.style.display = 'none';
        selectAllCheckbox.checked = false;
        selectAllCheckbox.indeterminate = false;
        if (bulkActionSubmit) {
          bulkActionSubmit.disabled = true;
        }
      }
    }

    // Select all checkbox handler
    if (selectAllCheckbox) {
      selectAllCheckbox.addEventListener('change', function () {
        checkboxes.forEach(function (checkbox) {
          checkbox.checked = selectAllCheckbox.checked;
        });
        updateBulkOperations();
      });
    }

    // Individual checkbox handlers
    checkboxes.forEach(function (checkbox) {
      checkbox.addEventListener('change', function () {
        updateBulkOperations();
      });
    });

    // Bulk action select handler
    if (bulkActionSelect && bulkActionSubmit) {
      bulkActionSelect.addEventListener('change', function () {
        bulkActionSubmit.disabled = !this.value;
      });
    }

    // Bulk action submit handler
    if (bulkActionSubmit) {
      bulkActionSubmit.addEventListener('click', function (e) {
        e.preventDefault();

        const action = bulkActionSelect ? bulkActionSelect.value : '';
        if (!action) {
          return;
        }

        const checked = document.querySelectorAll('.submission-checkbox:checked');
        if (checked.length === 0) {
          alert(Drupal.t('No items selected.'));
          return;
        }

        // Collect selected IDs
        const ids = [];
        checked.forEach(function (checkbox) {
          ids.push(checkbox.value);
        });

        if (action === 'delete') {
          const confirmMsg = Drupal.formatPlural(ids.length, 'Are you sure you want to delete @count selected item?', 'Are you sure you want to delete @count selected items?', {'@count': ids.length});
          if (confirm(confirmMsg)) {
            // Get bulk delete URL from settings
            const bulkDeleteUrl = (typeof drupalSettings !== 'undefined' && 
                                  drupalSettings.dgaRating && 
                                  drupalSettings.dgaRating.bulkDeleteUrl) 
                                  ? drupalSettings.dgaRating.bulkDeleteUrl 
                                  : '/admin/content/dga-rating/bulk-delete';

            // Create and submit form
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = bulkDeleteUrl;
            form.style.display = 'none';

            // Get CSRF token from settings
            let csrfToken = '';
            if (typeof drupalSettings !== 'undefined' && drupalSettings.csrfToken) {
              csrfToken = drupalSettings.csrfToken;
            }

            // Add CSRF token if available
            if (csrfToken) {
              const csrfInput = document.createElement('input');
              csrfInput.type = 'hidden';
              csrfInput.name = 'form_token';
              csrfInput.value = csrfToken;
              form.appendChild(csrfInput);
            }

            // Add submission IDs
            ids.forEach(function (id) {
              const input = document.createElement('input');
              input.type = 'hidden';
              input.name = 'submissions[]';
              input.value = id;
              form.appendChild(input);
            });

            document.body.appendChild(form);
            form.submit();
          }
        }
      });
    }
  }

  /**
   * Initialize all admin functionality.
   */
  Drupal.behaviors.dgaRatingAdmin = {
    attach: function (context) {
      once('dga-rating-admin', 'body', context).forEach(function () {
        initTableSorting();
        initUrlStatsControls();
        initFilterForm();
        initDropbuttons();
        initBulkOperations();
        initUrlStatsCollapsible();
        initUrlStatsSearch();
        initViewModeToggle();
        initFiltersToggle();
      });
    },
  };

  /**
   * Initialize collapsible URL statistics blocks.
   */
  function initUrlStatsCollapsible() {
    const toggles = document.querySelectorAll('.url-statistics-toggle');
    
    toggles.forEach(function (toggle) {
      toggle.addEventListener('click', function () {
        const contentId = toggle.getAttribute('aria-controls');
        const content = document.getElementById(contentId);
        const isExpanded = toggle.getAttribute('aria-expanded') === 'true';
        
        if (content) {
          if (isExpanded) {
            content.style.display = 'none';
            toggle.setAttribute('aria-expanded', 'false');
          } else {
            content.style.display = 'block';
            toggle.setAttribute('aria-expanded', 'true');
          }
        }
        
        updateExpandCollapseButtons();
      });
    });
    
    // Expand All / Collapse All buttons
    const expandAllBtn = document.getElementById('url-stats-expand-all');
    const collapseAllBtn = document.getElementById('url-stats-collapse-all');
    
    if (expandAllBtn) {
      expandAllBtn.addEventListener('click', function () {
        const contents = document.querySelectorAll('.url-statistics-content');
        const toggles = document.querySelectorAll('.url-statistics-toggle');
        
        contents.forEach(function (content) {
          content.style.display = 'block';
        });
        
        toggles.forEach(function (toggle) {
          toggle.setAttribute('aria-expanded', 'true');
        });
        
        updateExpandCollapseButtons();
      });
    }
    
    if (collapseAllBtn) {
      collapseAllBtn.addEventListener('click', function () {
        const contents = document.querySelectorAll('.url-statistics-content');
        const toggles = document.querySelectorAll('.url-statistics-toggle');
        
        contents.forEach(function (content) {
          content.style.display = 'none';
        });
        
        toggles.forEach(function (toggle) {
          toggle.setAttribute('aria-expanded', 'false');
        });
        
        updateExpandCollapseButtons();
      });
    }
    
    function updateExpandCollapseButtons() {
      const contents = document.querySelectorAll('.url-statistics-content:not([style*="display: none"])');
      const hasExpanded = contents.length > 0;
      
      if (expandAllBtn && collapseAllBtn) {
        if (hasExpanded) {
          expandAllBtn.style.display = 'none';
          collapseAllBtn.style.display = 'inline-block';
        } else {
          expandAllBtn.style.display = 'inline-block';
          collapseAllBtn.style.display = 'none';
        }
      }
    }
  }

  /**
   * Initialize URL statistics search/filter.
   */
  function initUrlStatsSearch() {
    const searchInput = document.getElementById('url-stats-search');
    if (!searchInput) {
      return;
    }
    
    searchInput.addEventListener('input', function () {
      const searchTerm = this.value.toLowerCase().trim();
      const blocks = document.querySelectorAll('.url-statistics-block');
      
      blocks.forEach(function (block) {
        const url = block.getAttribute('data-url') || '';
        const urlText = block.querySelector('.url-link-main')?.textContent?.toLowerCase() || '';
        
        if (searchTerm === '' || url.includes(searchTerm) || urlText.includes(searchTerm)) {
          block.classList.remove('hidden');
        } else {
          block.classList.add('hidden');
        }
      });
    });
  }

  /**
   * Initialize view mode toggle (Compact/Detailed).
   */
  function initViewModeToggle() {
    const compactBtn = document.getElementById('view-mode-compact');
    const detailedBtn = document.getElementById('view-mode-detailed');
    const adminWrapper = document.querySelector('.dga-rating-admin-wrapper');
    
    if (!compactBtn || !detailedBtn || !adminWrapper) {
      return;
    }
    
    // Set default to compact view
    adminWrapper.classList.add('view-mode-compact');
    
    compactBtn.addEventListener('click', function () {
      adminWrapper.classList.remove('view-mode-detailed');
      adminWrapper.classList.add('view-mode-compact');
      compactBtn.classList.add('active');
      detailedBtn.classList.remove('active');
      localStorage.setItem('dga-rating-view-mode', 'compact');
    });
    
    detailedBtn.addEventListener('click', function () {
      adminWrapper.classList.remove('view-mode-compact');
      adminWrapper.classList.add('view-mode-detailed');
      detailedBtn.classList.add('active');
      compactBtn.classList.remove('active');
      localStorage.setItem('dga-rating-view-mode', 'detailed');
    });
    
    // Restore saved preference
    const savedMode = localStorage.getItem('dga-rating-view-mode');
    if (savedMode === 'detailed') {
      detailedBtn.click();
    }
  }

  /**
   * Initialize filters toggle.
   */
  function initFiltersToggle() {
    const toggleBtn = document.getElementById('toggle-filters');
    const filtersContent = document.getElementById('filters-content');
    
    if (!toggleBtn || !filtersContent) {
      return;
    }
    
    // Remove any existing event listeners by cloning the button
    const toggleBtnClone = toggleBtn.cloneNode(true);
    toggleBtn.parentNode.replaceChild(toggleBtnClone, toggleBtn);
    
    // Get the toggle text element from the cloned button
    const toggleText = toggleBtnClone.querySelector('.toggle-text');
    
    toggleBtnClone.addEventListener('click', function (e) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
      
      const isExpanded = toggleBtnClone.getAttribute('aria-expanded') === 'true';
      
      if (isExpanded) {
        filtersContent.style.display = 'none';
        toggleBtnClone.setAttribute('aria-expanded', 'false');
        if (toggleText) {
          toggleText.textContent = Drupal.t('Show Filters');
        }
      } else {
        filtersContent.style.display = 'block';
        toggleBtnClone.setAttribute('aria-expanded', 'true');
        if (toggleText) {
          toggleText.textContent = Drupal.t('Hide Filters');
        }
      }
    });
  }

  // Initialize on page load
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function () {
      once('dga-rating-admin-init', 'body', document).forEach(function () {
        initTableSorting();
        initUrlStatsControls();
        initFilterForm();
        initDropbuttons();
        initBulkOperations();
        initUrlStatsCollapsible();
        initUrlStatsSearch();
        initViewModeToggle();
        initFiltersToggle();
      });
    });
  } else {
    once('dga-rating-admin-init', 'body', document).forEach(function () {
      initTableSorting();
      initUrlStatsControls();
      initFilterForm();
      initDropbuttons();
      initBulkOperations();
      initUrlStatsCollapsible();
      initUrlStatsSearch();
      initViewModeToggle();
      initFiltersToggle();
    });
  }

})(Drupal, once);
