<?php

namespace Drupal\dga_rating\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\dga_rating\Service\DgaRatingService;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a DGA Rating widget block.
 *
 * @Block(
 *   id = "dga_rating_block",
 *   admin_label = @Translation("DGA Rating Widget"),
 *   category = @Translation("Custom")
 * )
 */
class DgaRatingBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * Constructs a DgaRatingBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DgaRatingService $rating_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dga_rating.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get current URL path - use canonical path, not alias, for consistency.
    $current_path = \Drupal::service('path.current')->getPath();

    // Get the current request to extract the actual URL path
    $request = \Drupal::request();
    $request_path = $request->getPathInfo();

    // Get the alias
    $url_alias = \Drupal::service('path_alias.manager')->getAliasByPath($current_path);

    // IMPORTANT: Use alias (without language prefix) for consistency
    // This matches how URLs are saved (without /en/ prefix)
    // Example: /calculator not /en/calculator
    if ($url_alias !== $current_path && $url_alias !== '/') {
      $url = rtrim($url_alias, '/');
    } else {
      // If no alias, strip language prefix from request path
      $url = $request_path;
      if (preg_match('#^/[a-z]{2}/(.+)$#', $url, $matches)) {
        $url = '/' . $matches[1];
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Ensure we don't have language prefix in the final URL
    if (preg_match('#^/[a-z]{2}/#', $url)) {
      $url = preg_replace('#^/[a-z]{2}#', '', $url);
      $url = rtrim($url, '/') ?: '/';
    }

    // Try to get entity context if available.
    $entity_type = NULL;
    $entity_id = NULL;
    $route_match = \Drupal::routeMatch();

    // Check if we're on a node page.
    $node = $route_match->getParameter('node');
    if ($node instanceof Node) {
      $entity_type = 'node';
      $entity_id = $node->id();
      // IMPORTANT: Query by BOTH entity AND URL simultaneously.
      // Anonymous ratings are saved with entity_id=0 and rely on URL.
      // Authenticated ratings may be saved with entity_id > 0.
      // By querying both, we capture all ratings for this page.
      if ($entity_id > 0) {
        // Query using getStatistics with both entity and URL to match all ratings.
        $stats = $this->ratingService->getStatistics($entity_type, $entity_id, $url);
      } else {
        // Fallback to URL-based statistics if entity_id is invalid.
        $stats = $this->ratingService->getStatisticsByUrl($url);
      }
    }
    else {
      // Fallback to URL-based statistics.
      $stats = $this->ratingService->getStatisticsByUrl($url);
    }

    // Format the average rating to one decimal place.
    // Ensure we always show numeric values, defaulting to 0.0 and 0 if no data.
    $average = $stats['average'] > 0 ? number_format($stats['average'], 1, '.', '') : '0.0';
    $count = (int) $stats['count'];

    // Ensure count is always a number (not string).
    $count = max(0, $count);

    // Load configuration for text strings.
    $config = \Drupal::config('dga_rating.settings');

    // Build render array with translated strings.
    $build = [
      '#theme' => 'dga_rating_widget',
      '#average' => $average,
      '#count' => $count,
      '#url' => $url,
      '#entity_type' => $entity_type,
      '#entity_id' => $entity_id,
      '#texts' => [
        'average_text' => $this->t($config->get('average_text') ?? 'This service is rated with an average of')->render(),
        'reviews_text' => $this->t($config->get('reviews_text') ?? 'reviews')->render(),
        'rate_button_text' => $this->t($config->get('rate_button_text') ?? 'Rate this service')->render(),
        'form_title' => $this->t($config->get('form_title') ?? 'Tell us what you think of this service')->render(),
        'form_description' => $this->t($config->get('form_description') ?? "Please don't include personal or financial information. Your review will be submitted and recorded to improve services.")->render(),
        'close_button_text' => $this->t($config->get('close_button_text') ?? 'Close')->render(),
        'rating_question' => $this->t($config->get('rating_question') ?? 'How would you rate this service?')->render(),
        'rating_instructions' => $this->t($config->get('rating_instructions') ?? 'Rate your experience from (1) poor to (5) excellent')->render(),
        'rating_aria_label' => $this->t($config->get('rating_aria_label') ?? 'Select rating from 1 to 5')->render(),
        'feedback_label' => $this->t($config->get('feedback_label') ?? 'Feedback')->render(),
        'submit_button_text' => $this->t($config->get('submit_button_text') ?? 'Submit')->render(),
        'submitted_rating_text' => $this->t($config->get('submitted_rating_text') ?? 'You rated this service as')->render(),
        'submitted_success_text' => $this->t($config->get('submitted_success_text') ?? 'Your feedback is submitted!')->render(),
        'star_singular' => $this->t($config->get('star_singular') ?? 'star')->render(),
        'star_plural' => $this->t($config->get('star_plural') ?? 'stars')->render(),
      ],
      '#attached' => [
        'library' => [
          'dga_rating/rating_widget',
        ],
      ],
      '#cache' => [
        'max-age' => 0, // Disable cache for dynamic content.
        'contexts' => [
          'url.path',
          'route',
          'user.roles:anonymous',
        ],
        'tags' => [
          'dga_rating:submissions',
          'config:dga_rating.settings',
        ],
      ],
    ];


    return $build;
  }

}


