(function (Drupal, once) {
  "use strict";

  // DGA Rating Widget Script

  function pctForRating(r) {
    return (Math.max(0, Math.min(5, Number(r) || 0)) / 5) * 100;
  }

  Drupal.behaviors.ratingWidget = {
    attach: function(context, settings) {
      // Find widgets - use direct query if once() fails
      let widgets = [];
      try {
        widgets = once("ratingWidget", "[data-rating-widget]", context);
      } catch (e) {
        // Fallback to direct query
      }

      if (widgets.length === 0) {
        const direct = (context.querySelectorAll || document.querySelectorAll).call(context === document ? document : context, "[data-rating-widget]");
        widgets = Array.from(direct || []);
      }

      if (widgets.length === 0) {
        return;
      }

      widgets.forEach((root) => {
        if (root.hasAttribute('data-rw-initialized')) {
          return;
        }
        root.setAttribute('data-rw-initialized', 'true');

        const rowClosed = root.querySelector("[data-rw-closed]");
        const rowOpen = root.querySelector("[data-rw-opened]");
        const rowSubmitted = root.querySelector("[data-rw-submitted]");
        const btnOpen = root.querySelector("[data-rw-open]");
        const btnClose = root.querySelector("[data-rw-close]");
        const btnSubmit = root.querySelector("[data-rw-submit]");
        const avgText = root.querySelector("[data-rw-average-text]");
        const avgFill = root.querySelector("[data-rw-average-fill]");
        const reviewsEl = root.querySelector("[data-rw-reviews]");
        const starWrap = root.querySelector("[data-rw-stars]");
        const starBtns = root.querySelectorAll("[data-rw-star]");
        const feedbackEl = root.querySelector("[data-rw-feedback]");
        const yourRatingText = root.querySelector("[data-rw-your-rating]");
        const yourFill = root.querySelector("[data-rw-your-fill]");

        let average = Number(root.getAttribute("data-average")) || 0;
        let reviews = root.getAttribute("data-reviews") || "0";
        let url = root.getAttribute("data-url") || window.location.pathname;
        url = url.trim().replace(/\/+$/, '') || '/';
        if (url.match(/^\/[a-z]{2}\//)) {
          url = url.replace(/^\/[a-z]{2}/, '');
          url = url || '/';
        }

        const entityType = root.getAttribute("data-entity-type") || "";
        const entityId = root.getAttribute("data-entity-id") || "";

        const state = { rating: 0, feedback: "", open: false, submitted: false };

        function setClosedView() {
          if (rowClosed) rowClosed.hidden = false;
          if (rowOpen) rowOpen.hidden = true;
          if (rowSubmitted) rowSubmitted.hidden = true;
        }
        function setOpenView() {
          if (rowClosed) rowClosed.hidden = true;
          if (rowOpen) rowOpen.hidden = false;
          if (rowSubmitted) rowSubmitted.hidden = true;
        }
        function setSubmittedView() {
          if (rowClosed) rowClosed.hidden = true;
          if (rowOpen) rowOpen.hidden = true;
          if (rowSubmitted) rowSubmitted.hidden = false;
        }
        function paintAverage() {
          if (avgText) avgText.textContent = average.toFixed(1);
          if (avgFill) avgFill.style.width = pctForRating(average) + "%";
          if (reviewsEl) {
            const reviewsText = reviews === "1" 
              ? (typeof Drupal !== 'undefined' ? Drupal.t('review') : 'review')
              : (typeof Drupal !== 'undefined' ? Drupal.t('reviews') : 'reviews');
            reviewsEl.textContent = reviews + " " + reviewsText;
          }
        }
        function paintYourRating() {
          if (yourRatingText) yourRatingText.textContent = String(state.rating);
          if (yourFill) yourFill.style.width = pctForRating(state.rating) + "%";
          starBtns.forEach((btn) => {
            const v = Number(btn.getAttribute("data-value"));
            btn.setAttribute("aria-checked", v === state.rating ? "true" : "false");
            btn.classList.toggle("is-active", v <= state.rating);
          });
        }

        // Events
        if (btnOpen) {
          btnOpen.addEventListener("click", (e) => {
            e.preventDefault();
            state.open = true;
            setOpenView();
          });
        }

        if (btnClose) {
          btnClose.addEventListener("click", () => {
            state.open = false;
            setClosedView();
          });
        }

        if (feedbackEl) {
          feedbackEl.addEventListener("input", (e) => {
            state.feedback = e.target.value;
            // Clear feedback error when user starts typing
            const feedbackLabel = root.querySelector('[data-rw-feedback-label]');
            if (feedbackLabel) {
              const errorMsg = feedbackLabel.querySelector('.rw-error-message');
              if (errorMsg) {
                errorMsg.remove();
              }
              feedbackEl.classList.remove('error');
            }
          });
        }

        starBtns.forEach((btn) => {
          btn.addEventListener("click", () => {
            state.rating = Number(btn.getAttribute("data-value"));
            paintYourRating();
            // Clear rating error when user selects a rating
            const ratingTitle = root.querySelector('[data-rw-rating-title]');
            if (ratingTitle) {
              const errorMsg = ratingTitle.querySelector('.rw-error-message');
              if (errorMsg) {
                errorMsg.remove();
              }
            }
          });
        });

        // The global handler below will attach to the submit button
        // No need for widget-specific handler since global one works reliably
      });
    },
  };

  // Fallback for delayed content
  setTimeout(() => {
    if (typeof Drupal !== 'undefined' && Drupal.behaviors && Drupal.behaviors.ratingWidget) {
      Drupal.behaviors.ratingWidget.attach(document, Drupal.settings || {});
    }
  }, 1000);

  // GLOBAL HANDLER - Direct button handler that works independently
  function attachGlobalSubmitHandler() {
    const submitButtons = document.querySelectorAll('[data-rw-submit]');

    submitButtons.forEach(function(btn) {
      if (btn.hasAttribute('data-global-handler-attached')) {
        return;
      }

      btn.setAttribute('data-global-handler-attached', 'true');

      btn.onclick = function(e) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();

        // Find widget root
        const widget = btn.closest('[data-rating-widget]');
        if (!widget) {
          console.error('DGA Rating: Widget not found');
          alert(typeof Drupal !== 'undefined' ? Drupal.t('ERROR: Widget not found') : 'ERROR: Widget not found');
          return false;
        }

        // Get rating from stars
        let rating = 0;
        const activeStars = widget.querySelectorAll('[data-rw-star].is-active');
        if (activeStars.length > 0) {
          rating = activeStars.length;
        } else {
          const checkedStar = widget.querySelector('[data-rw-star][aria-checked="true"]');
          if (checkedStar) {
            rating = Number(checkedStar.getAttribute('data-value')) || 0;
          }
        }

        // Get validation messages from widget data attributes
        const validationRatingRequired = widget.getAttribute('data-validation-rating-required') || 'Please select a rating';
        const validationFeedbackRequired = widget.getAttribute('data-validation-feedback-required') || 'Please provide feedback';
        const validationFeedbackTooLong = widget.getAttribute('data-validation-feedback-too-long') || 'Feedback is too long';

        // Helper function to show inline error message
        function showError(element, message) {
          // Remove existing error message
          const existingError = element.querySelector('.rw-error-message');
          if (existingError) {
            existingError.remove();
          }
          
          // Create error span
          const errorSpan = document.createElement('span');
          errorSpan.className = 'rw-error-message';
          errorSpan.textContent = ' - ' + message;
          element.appendChild(errorSpan);
          
          // Add error styling to input if it's a form field
          const input = element.querySelector('input, textarea');
          if (input) {
            input.classList.add('error');
            input.focus();
          }
        }

        // Helper function to clear error message
        function clearError(element) {
          const errorMsg = element.querySelector('.rw-error-message');
          if (errorMsg) {
            errorMsg.remove();
          }
          const input = element.querySelector('input, textarea');
          if (input) {
            input.classList.remove('error');
          }
        }

        // Validate rating
        if (rating < 1 || rating > 5) {
          const ratingTitle = widget.querySelector('[data-rw-rating-title]');
          if (ratingTitle) {
            showError(ratingTitle, validationRatingRequired);
          }
          return false;
        } else {
          // Clear rating error if exists
          const ratingTitle = widget.querySelector('[data-rw-rating-title]');
          if (ratingTitle) {
            clearError(ratingTitle);
          }
        }

        // Get feedback and validate
        const feedbackEl = widget.querySelector('[data-rw-feedback]');
        const feedback = feedbackEl ? feedbackEl.value.trim() : '';
        const feedbackLabel = widget.querySelector('[data-rw-feedback-label]');
        
        if (!feedback || feedback.length === 0) {
          if (feedbackLabel) {
            showError(feedbackLabel, validationFeedbackRequired);
          }
          return false;
        } else {
          // Clear feedback error if exists
          if (feedbackLabel) {
            clearError(feedbackLabel);
          }
        }

        // Get URL
        let url = widget.getAttribute('data-url') || window.location.pathname;
        url = url.trim().replace(/\/+$/, '') || '/';
        if (url.match(/^\/[a-z]{2}\//)) {
          url = url.replace(/^\/[a-z]{2}/, '');
          url = url || '/';
        }

        // Disable button
        const originalText = btn.textContent;
        btn.disabled = true;
        btn.textContent = typeof Drupal !== 'undefined' ? Drupal.t('Submitting...') : 'Submitting...';

        // Prepare payload
        const payload = {
          rating: rating,
          feedback: feedback || '',
          url: url || '/',
          entity_type: 'node',
          entity_id: 0
        };

        // Send request
        const xhr = new XMLHttpRequest();
        xhr.open('POST', '/dga-rating/submit', false);
        xhr.setRequestHeader('Content-Type', 'application/json');

        try {
          xhr.send(JSON.stringify(payload));

          let data;
          try {
            data = JSON.parse(xhr.responseText);
          } catch (e) {
            console.error('DGA Rating: JSON Parse Error:', e);
            const errorMsg = typeof Drupal !== 'undefined' ? Drupal.t('ERROR: Invalid JSON response: @response', {'@response': xhr.responseText.substring(0, 200)}) : 'ERROR: Invalid JSON response: ' + xhr.responseText.substring(0, 200);
            alert(errorMsg);
            btn.disabled = false;
            btn.textContent = originalText;
            return false;
          }

          // Validate response
          if (xhr.status === 200 && data.success && data.rating_id) {
            // Clear any existing errors before showing success
            const ratingTitle = widget.querySelector('[data-rw-rating-title]');
            const feedbackLabel = widget.querySelector('[data-rw-feedback-label]');
            if (ratingTitle) {
              const errorMsg = ratingTitle.querySelector('.rw-error-message');
              if (errorMsg) errorMsg.remove();
            }
            if (feedbackLabel) {
              const errorMsg = feedbackLabel.querySelector('.rw-error-message');
              if (errorMsg) errorMsg.remove();
            }

            // Show submitted state with widget message
            const rowClosed = widget.querySelector('[data-rw-closed]');
            const rowOpen = widget.querySelector('[data-rw-opened]');
            const rowSubmitted = widget.querySelector('[data-rw-submitted]');

            if (rowClosed) rowClosed.hidden = true;
            if (rowOpen) rowOpen.hidden = true;
            if (rowSubmitted) {
              rowSubmitted.hidden = false;
              const yourRating = rowSubmitted.querySelector('[data-rw-your-rating]');
              if (yourRating) yourRating.textContent = rating;
            }

            // Get refresh settings
            const refreshDelay = (typeof drupalSettings !== 'undefined' && 
                                 drupalSettings.dgaRating && 
                                 drupalSettings.dgaRating.refreshDelay) 
                                 ? drupalSettings.dgaRating.refreshDelay 
                                 : 3000; // Default 3 seconds

            const refreshBlockUrl = (typeof drupalSettings !== 'undefined' && 
                                    drupalSettings.dgaRating && 
                                    drupalSettings.dgaRating.refreshBlockUrl) 
                                    ? drupalSettings.dgaRating.refreshBlockUrl 
                                    : '/dga-rating/refresh-block';

            // After delay, refresh block via AJAX and reset to closed state
            setTimeout(() => {
              // Get current widget data
              const widgetUrl = widget.getAttribute('data-url') || window.location.pathname;
              const entityType = widget.getAttribute('data-entity-type') || '';
              const entityId = widget.getAttribute('data-entity-id') || '';

              // Build refresh URL
              let refreshUrl = refreshBlockUrl + '?url=' + encodeURIComponent(widgetUrl);
              if (entityType) {
                refreshUrl += '&entity_type=' + encodeURIComponent(entityType);
              }
              if (entityId) {
                refreshUrl += '&entity_id=' + encodeURIComponent(entityId);
              }

              // Fetch updated statistics
              fetch(refreshUrl, {
                method: 'GET',
                headers: {
                  'Accept': 'application/json',
                },
              })
              .then(response => response.json())
              .then(refreshData => {
                // Update statistics
                if (refreshData.success && refreshData.statistics) {
                  const avgText = widget.querySelector('[data-rw-average-text]');
                  const avgFill = widget.querySelector('[data-rw-average-fill]');
                  const reviewsEl = widget.querySelector('[data-rw-reviews]');
                  
                  if (avgText) avgText.textContent = refreshData.average || '0.0';
                  if (avgFill) {
                    const avgValue = parseFloat(refreshData.average || 0);
                    avgFill.style.width = ((avgValue / 5) * 100) + '%';
                  }
                  if (reviewsEl) {
                    const reviewsText = refreshData.count === 1 
                      ? (typeof Drupal !== 'undefined' ? Drupal.t('review') : 'review')
                      : (typeof Drupal !== 'undefined' ? Drupal.t('reviews') : 'reviews');
                    reviewsEl.textContent = refreshData.count + ' ' + reviewsText;
                  }
                  
                  // Update widget data attributes
                  widget.setAttribute('data-average', refreshData.average || '0.0');
                  widget.setAttribute('data-reviews', refreshData.count || '0');
                }
                
                // Reset widget to closed state
                if (rowClosed) rowClosed.hidden = false;
                if (rowOpen) rowOpen.hidden = true;
                if (rowSubmitted) rowSubmitted.hidden = true;
                
                // Clear form data
                const feedbackEl = widget.querySelector('[data-rw-feedback]');
                if (feedbackEl) {
                  feedbackEl.value = '';
                }
                
                // Reset star selection
                const starBtns = widget.querySelectorAll('[data-rw-star]');
                starBtns.forEach((starBtn) => {
                  starBtn.classList.remove('is-active');
                  starBtn.setAttribute('aria-checked', 'false');
                });
                
                // Store reset state on widget element for future reference
                widget.setAttribute('data-rw-rating', '0');
                widget.setAttribute('data-rw-feedback', '');
              })
              .catch(error => {
                console.error('DGA Rating: Error refreshing block:', error);
                // Fallback: just reset to closed state manually with data from submission
                if (rowClosed) rowClosed.hidden = false;
                if (rowOpen) rowOpen.hidden = true;
                if (rowSubmitted) rowSubmitted.hidden = true;
                
                // Update statistics if available from submission response
                if (data.statistics) {
                  const avgText = widget.querySelector('[data-rw-average-text]');
                  const avgFill = widget.querySelector('[data-rw-average-fill]');
                  const reviewsEl = widget.querySelector('[data-rw-reviews]');
                  
                  if (avgText) avgText.textContent = data.average || '0.0';
                  if (avgFill) {
                    const avgValue = parseFloat(data.average || 0);
                    avgFill.style.width = ((avgValue / 5) * 100) + '%';
                  }
                  if (reviewsEl) {
                    const reviewsText = data.count === 1 
                      ? (typeof Drupal !== 'undefined' ? Drupal.t('review') : 'review')
                      : (typeof Drupal !== 'undefined' ? Drupal.t('reviews') : 'reviews');
                    reviewsEl.textContent = data.count + ' ' + reviewsText;
                  }
                  
                  // Update widget data attributes
                  widget.setAttribute('data-average', data.average || '0.0');
                  widget.setAttribute('data-reviews', data.count || '0');
                }
                
                // Clear form
                const feedbackEl = widget.querySelector('[data-rw-feedback]');
                if (feedbackEl) {
                  feedbackEl.value = '';
                }
                
                // Reset stars
                const starBtns = widget.querySelectorAll('[data-rw-star]');
                starBtns.forEach((starBtn) => {
                  starBtn.classList.remove('is-active');
                  starBtn.setAttribute('aria-checked', 'false');
                });
              });
            }, refreshDelay);

            // Re-enable button
            btn.disabled = false;
            btn.textContent = originalText;
          } else {
            console.error('DGA Rating: Submission failed - Status:', xhr.status, 'Success:', data.success, 'Rating ID:', data.rating_id);
            const errorMsg = data.message || (typeof Drupal !== 'undefined' ? Drupal.t('Unknown error') : 'Unknown error');
            
            // Determine which field the error relates to and show inline
            const errorLower = errorMsg.toLowerCase();
            let targetElement = null;
            
            if (errorLower.includes('rating') || errorLower.includes('تقييم')) {
              targetElement = widget.querySelector('[data-rw-rating-title]');
            } else if (errorLower.includes('feedback') || errorLower.includes('تعليق') || errorLower.includes('long') || errorLower.includes('طويل')) {
              targetElement = widget.querySelector('[data-rw-feedback-label]');
            }
            
            // If we can't determine the field, show in form title
            if (!targetElement) {
              targetElement = widget.querySelector('.rw-head-left .rw-title');
            }
            
            if (targetElement) {
              const existingError = targetElement.querySelector('.rw-error-message');
              if (existingError) {
                existingError.remove();
              }
              const errorSpan = document.createElement('span');
              errorSpan.className = 'rw-error-message';
              errorSpan.textContent = ' - ' + errorMsg;
              targetElement.appendChild(errorSpan);
              
              // Add error styling to input if it's a form field
              const input = widget.querySelector('[data-rw-feedback]');
              if (input && (errorLower.includes('feedback') || errorLower.includes('تعليق'))) {
                input.classList.add('error');
                input.focus();
              }
            }
            
            btn.disabled = false;
            btn.textContent = originalText;
          }
        } catch (err) {
          console.error('DGA Rating: Exception during submission:', err);
          const errorMsg = typeof Drupal !== 'undefined'
            ? Drupal.t('EXCEPTION: @message\nStatus: @status', {'@message': err.message, '@status': xhr.status})
            : 'EXCEPTION: ' + err.message + '\nStatus: ' + xhr.status;
          alert(errorMsg);
          btn.disabled = false;
          btn.textContent = originalText;
        }

        return false;
      };
    });
  }

  // Attach on DOM ready and once after delay
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', attachGlobalSubmitHandler);
  } else {
    attachGlobalSubmitHandler();
  }

  // Single delayed attach as fallback
  setTimeout(attachGlobalSubmitHandler, 1000);

})(Drupal, once);
