<?php

namespace Drupal\dga_rating\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dga_rating\Service\DgaRatingService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for deleting a rating submission.
 */
class DgaRatingDeleteForm extends ConfirmFormBase {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * The submission ID.
   *
   * @var int
   */
  protected $submissionId;

  /**
   * The submission data.
   *
   * @var array|false
   */
  protected $submission;

  /**
   * Constructs a DgaRatingDeleteForm object.
   *
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(DgaRatingService $rating_service) {
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dga_rating.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_rating_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete this rating submission?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('dga_rating.admin');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    if ($this->submission) {
      return $this->t('This action cannot be undone. Rating ID: @id, Rating: @rating stars', [
        '@id' => $this->submissionId,
        '@rating' => $this->submission['rating'] ?? 0,
      ]);
    }
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->submissionId = $id;
    $this->submission = $this->ratingService->getSubmissionById($id);
    
    if (!$this->submission) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $success = $this->ratingService->deleteSubmission($this->submissionId);

    if ($success) {
      $this->messenger()->addMessage($this->t('Rating submission deleted successfully.'));
      $form_state->setRedirect('dga_rating.admin');
    } else {
      $this->messenger()->addError($this->t('Failed to delete rating submission.'));
    }
  }

}

