<?php

namespace Drupal\dga_rating\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\dga_rating\Service\DgaRatingService;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a DGA Rating widget block.
 *
 * @Block(
 *   id = "dga_rating_block",
 *   admin_label = @Translation("DGA Rating Widget"),
 *   category = @Translation("Custom")
 * )
 */
class DgaRatingBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The rating service.
   *
   * @var \Drupal\dga_rating\Service\DgaRatingService
   */
  protected $ratingService;

  /**
   * Constructs a DgaRatingBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dga_rating\Service\DgaRatingService $rating_service
   *   The rating service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DgaRatingService $rating_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->ratingService = $rating_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dga_rating.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get current URL path - use canonical path, not alias, for consistency.
    $current_path = \Drupal::service('path.current')->getPath();

    // Get the current request to extract the actual URL path
    $request = \Drupal::request();
    $request_path = $request->getPathInfo();

    // Get the alias
    $url_alias = \Drupal::service('path_alias.manager')->getAliasByPath($current_path);

    // IMPORTANT: Use alias (without language prefix) for consistency
    // This matches how URLs are saved (without /en/ prefix)
    // Example: /calculator not /en/calculator
    if ($url_alias !== $current_path && $url_alias !== '/') {
      $url = rtrim($url_alias, '/');
    } else {
      // If no alias, strip language prefix from request path
      $url = $request_path;
      if (preg_match('#^/[a-z]{2}/(.+)$#', $url, $matches)) {
        $url = '/' . $matches[1];
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Ensure we don't have language prefix in the final URL
    if (preg_match('#^/[a-z]{2}/#', $url)) {
      $url = preg_replace('#^/[a-z]{2}#', '', $url);
      $url = rtrim($url, '/') ?: '/';
    }

    // Try to get entity context if available.
    $entity_type = NULL;
    $entity_id = NULL;
    $route_match = \Drupal::routeMatch();

    // Check if we're on a node page.
    $node = $route_match->getParameter('node');
    if ($node instanceof Node) {
      $entity_type = 'node';
      $entity_id = $node->id();
      // IMPORTANT: Query by BOTH entity AND URL simultaneously.
      // Anonymous ratings are saved with entity_id=0 and rely on URL.
      // Authenticated ratings may be saved with entity_id > 0.
      // By querying both, we capture all ratings for this page.
      if ($entity_id > 0) {
        // Query using getStatistics with both entity and URL to match all ratings.
        $stats = $this->ratingService->getStatistics($entity_type, $entity_id, $url);
      } else {
        // Fallback to URL-based statistics if entity_id is invalid.
        $stats = $this->ratingService->getStatisticsByUrl($url);
      }
    }
    else {
      // Fallback to URL-based statistics.
      $stats = $this->ratingService->getStatisticsByUrl($url);
    }

    // Format the average rating to one decimal place.
    // Ensure we always show numeric values, defaulting to 0.0 and 0 if no data.
    $average = $stats['average'] > 0 ? number_format($stats['average'], 1, '.', '') : '0.0';
    $count = (int) $stats['count'];

    // Ensure count is always a number (not string).
    $count = max(0, $count);

    // Load configuration for text strings.
    $config = \Drupal::config('dga_rating.settings');
    
    // Get current language.
    $language_manager = \Drupal::languageManager();
    $current_lang = $language_manager->getCurrentLanguage()->getId();
    $is_arabic = ($current_lang === 'ar');
    
    // Helper function to get translation with fallback.
    $getTranslation = function($key_en, $key_ar, $default_en, $default_ar = NULL) use ($config, $is_arabic) {
      if ($is_arabic) {
        $value = $config->get($key_ar);
        if (!empty($value)) {
          return $value;
        }
        // Fallback to English if Arabic is empty.
        $value = $config->get($key_en);
        if (!empty($value)) {
          return $value;
        }
        return $default_ar ?? $default_en;
      }
      else {
        $value = $config->get($key_en);
        if (!empty($value)) {
          return $value;
        }
        return $default_en;
      }
    };

    // Build render array with translated strings.
    $build = [
      '#theme' => 'dga_rating_widget',
      '#average' => $average,
      '#count' => $count,
      '#url' => $url,
      '#entity_type' => $entity_type,
      '#entity_id' => $entity_id,
      '#texts' => [
        'average_text' => $getTranslation('average_text_en', 'average_text_ar', 'This service is rated with an average of', 'تم تقييم هذه الخدمة بمتوسط'),
        'reviews_text' => $getTranslation('reviews_text_en', 'reviews_text_ar', 'reviews', 'تقييم'),
        'rate_button_text' => $getTranslation('rate_button_text_en', 'rate_button_text_ar', 'Rate this service', 'قيّم هذه الخدمة'),
        'form_title' => $getTranslation('form_title_en', 'form_title_ar', 'Tell us what you think of this service', 'أخبرنا برأيك في هذه الخدمة'),
        'form_description' => $getTranslation('form_description_en', 'form_description_ar', "Please don't include personal or financial information. Your review will be submitted and recorded to improve services.", 'يرجى عدم تضمين معلومات شخصية أو مالية. سيتم إرسال مراجعتك وتسجيلها لتحسين الخدمات.'),
        'close_button_text' => $getTranslation('close_button_text_en', 'close_button_text_ar', 'Close', 'إغلاق'),
        'rating_question' => $getTranslation('rating_question_en', 'rating_question_ar', 'How would you rate this service?', 'كيف تقيم هذه الخدمة؟'),
        'rating_instructions' => $getTranslation('rating_instructions_en', 'rating_instructions_ar', 'Rate your experience from (1) poor to (5) excellent', 'قيم تجربتك من (1) ضعيف إلى (5) ممتاز'),
        'rating_aria_label' => $getTranslation('rating_aria_label_en', 'rating_aria_label_ar', 'Select rating from 1 to 5', 'اختر التقييم من 1 إلى 5'),
        'feedback_label' => $getTranslation('feedback_label_en', 'feedback_label_ar', 'Feedback', 'التعليقات'),
        'submit_button_text' => $getTranslation('submit_button_text_en', 'submit_button_text_ar', 'Submit', 'إرسال'),
        'submitted_rating_text' => $getTranslation('submitted_rating_text_en', 'submitted_rating_text_ar', 'You rated this service as', 'قيمت هذه الخدمة بـ'),
        'submitted_success_text' => $getTranslation('submitted_success_text_en', 'submitted_success_text_ar', 'Your feedback is submitted!', 'تم إرسال ملاحظاتك!'),
        'star_singular' => $getTranslation('star_singular_en', 'star_singular_ar', 'star', 'نجمة'),
        'star_plural' => $getTranslation('star_plural_en', 'star_plural_ar', 'stars', 'نجوم'),
        'validation_rating_required' => $getTranslation('validation_rating_required_en', 'validation_rating_required_ar', 'Please select a rating', 'يرجى اختيار التقييم'),
        'validation_feedback_required' => $getTranslation('validation_feedback_required_en', 'validation_feedback_required_ar', 'Please provide feedback', 'يرجى تقديم التعليقات'),
        'validation_feedback_too_long' => $getTranslation('validation_feedback_too_long_en', 'validation_feedback_too_long_ar', 'Feedback is too long', 'التعليقات طويلة جداً'),
      ],
      '#attached' => [
        'library' => [
          'dga_rating/rating_widget',
        ],
        'drupalSettings' => [
          'dgaRating' => [
            'refreshBlockUrl' => \Drupal\Core\Url::fromRoute('dga_rating.refresh_block')->toString(),
            'refreshDelay' => $config->get('refresh_delay') ?? 3000, // Configurable delay before refresh
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0, // Disable cache for dynamic content.
        'contexts' => [
          'url.path',
          'route',
          'user.roles:anonymous',
        ],
        'tags' => [
          'dga_rating:submissions',
          'config:dga_rating.settings',
        ],
      ],
    ];


    return $build;
  }

}


