<?php

namespace Drupal\dga_rating\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\dga_rating\Service\DgaRatingService;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for handling rating AJAX requests.
 */
class DgaRatingController extends ControllerBase {

  protected $ratingService;

  public function __construct(DgaRatingService $rating_service) {
    $this->ratingService = $rating_service;
  }

  public static function create(ContainerInterface $container) {
    return new static($container->get('dga_rating.service'));
  }

  public function submitRating(Request $request) {
    // Handle CORS preflight (OPTIONS) requests
    if ($request->getMethod() === 'OPTIONS') {
      $response = new JsonResponse([]);
      $response->headers->set('Access-Control-Allow-Origin', '*');
      $response->headers->set('Access-Control-Allow-Methods', 'POST, OPTIONS');
      $response->headers->set('Access-Control-Allow-Headers', 'Content-Type, X-CSRF-Token');
      return $response;
    }
    
    // Reject GET requests - this endpoint is POST only
    if ($request->getMethod() !== 'POST') {
      \Drupal::logger('dga_rating')->warning('Invalid method @method for submit endpoint', [
        '@method' => $request->getMethod(),
      ]);
      return new JsonResponse(['success' => FALSE, 'message' => $this->t('Method not allowed. Use POST.')->render()], 405);
    }
    
    $user = $this->currentUser();
    $is_anon = !$user->isAuthenticated();
    
    // Get current language for error messages.
    $language_manager = \Drupal::languageManager();
    $current_lang = $language_manager->getCurrentLanguage()->getId();
    $is_arabic = ($current_lang === 'ar');
    $config = $this->config('dga_rating.settings');
    
    // Helper to get error message.
    $getErrorMessage = function($default_en, $default_ar = NULL) use ($config, $is_arabic) {
      $error_msg = $is_arabic 
        ? ($config->get('error_message_ar') ?: $config->get('error_message_en') ?: ($default_ar ?? $default_en))
        : ($config->get('error_message_en') ?: $default_en);
      return $error_msg;
    };

    // Parse JSON
    $data = json_decode($request->getContent(), TRUE);
    if (!$data) {
      \Drupal::logger('dga_rating')->error('Invalid JSON data received');
      $error_msg = $is_arabic ? 'بيانات JSON غير صالحة' : 'Invalid JSON data';
      return new JsonResponse(['success' => FALSE, 'message' => $getErrorMessage('Invalid JSON data', $error_msg)], 400);
    }

    // Validate rating
    if (!isset($data['rating']) || !is_numeric($data['rating'])) {
      $error_msg = $is_arabic ? 'التقييم مطلوب' : 'Rating required';
      return new JsonResponse(['success' => FALSE, 'message' => $getErrorMessage('Rating required', $error_msg)], 400);
    }

    $rating = (int) $data['rating'];
    if ($rating < 1 || $rating > 5) {
      $error_msg = $is_arabic ? 'يجب أن يكون التقييم من 1 إلى 5' : 'Rating must be 1-5';
      return new JsonResponse(['success' => FALSE, 'message' => $getErrorMessage('Rating must be 1-5', $error_msg)], 400);
    }

    // Validate feedback is required
    $feedback_text = isset($data['feedback']) ? trim($data['feedback']) : '';
    if (empty($feedback_text)) {
      $validation_msg = $is_arabic 
        ? ($config->get('validation_feedback_required_ar') ?: $config->get('validation_feedback_required_en') ?: 'Feedback is required')
        : ($config->get('validation_feedback_required_en') ?: 'Feedback is required');
      return new JsonResponse(['success' => FALSE, 'message' => $validation_msg], 400);
    }

    // CSRF token check skipped for anonymous users

    // Normalize URL
    $url = isset($data['url']) ? trim($data['url']) : '/';
    if (empty($url) || $url === '/') {
      $url = '/';
    } else {
      if (preg_match('#^/[a-z]{2}(/.*)?$#', $url, $matches)) {
        $url = isset($matches[1]) ? $matches[1] : '/';
        if (empty($url)) $url = '/';
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Get configurable limits from settings (config already loaded above)
    $feedback_max_length = (int) ($config->get('feedback_max_length') ?? 5000);
    
    // SECURITY: Sanitize feedback text (already validated as non-empty above)
    $feedback_text = strip_tags($feedback_text);
    if (strlen($feedback_text) > $feedback_max_length) {
      $validation_msg = $is_arabic 
        ? ($config->get('validation_feedback_too_long_ar') ?: $config->get('validation_feedback_too_long_en') ?: 'Feedback is too long')
        : ($config->get('validation_feedback_too_long_en') ?: 'Feedback is too long');
      // Replace @max placeholder if present
      $validation_msg = str_replace('@max', (string) $feedback_max_length, $validation_msg);
      return new JsonResponse(['success' => FALSE, 'message' => $validation_msg], 400);
    }

    // SECURITY: Rate limiting for anonymous users (configurable)
    if ($is_anon) {
      $rate_limit_max = (int) ($config->get('rate_limit_max_submissions') ?? 20);
      $rate_limit_window = (int) ($config->get('rate_limit_time_window') ?? 3600);
      
      // Only apply rate limiting if enabled (max > 0)
      if ($rate_limit_max > 0) {
        $ip_address = $request->getClientIp();
        $recent_count = \Drupal::database()
          ->select('dga_rating', 'r')
          ->condition('r.ip_address', $ip_address)
          ->condition('r.created', time() - $rate_limit_window, '>=')
          ->countQuery()
          ->execute()
          ->fetchField();
        
        if ($recent_count >= $rate_limit_max) {
          $error_msg = $is_arabic ? 'عدد كبير جداً من الإرسالات. يرجى المحاولة مرة أخرى لاحقاً.' : 'Too many submissions. Please try again later.';
          return new JsonResponse(['success' => FALSE, 'message' => $getErrorMessage('Too many submissions. Please try again later.', $error_msg)], 429);
        }
      }
    }

    // Save data
    $entity_id = 0;
    if ($user->isAuthenticated() && isset($data['entity_id']) && (int) $data['entity_id'] > 0) {
      $entity_id = (int) $data['entity_id'];
    }

    $save_data = [
      'entity_type' => $data['entity_type'] ?? 'node',
      'entity_id' => $entity_id,
      'rating' => $rating,
      'feedback' => $feedback_text,
      'url' => $url,
      'user_id' => $is_anon ? NULL : $user->id(),
      'ip_address' => $request->getClientIp(),
    ];

    // Save rating
    $rating_id = $this->ratingService->saveRating($save_data);

    // Handle the case where saveRating returns FALSE or invalid value
    if ($rating_id === FALSE || !is_numeric($rating_id) || (int) $rating_id <= 0) {
      \Drupal::logger('dga_rating')->error('Save FAILED - Returned: @result', [
        '@result' => var_export($rating_id, TRUE),
      ]);
      
      // FOR ANONYMOUS: Try direct database insert as fallback
      if ($is_anon) {
        try {
          $db = \Drupal::database();
          $save_data['created'] = time();
          $direct_id = $db->insert('dga_rating')
            ->fields($save_data)
            ->execute();
          
          if ($direct_id && is_numeric($direct_id) && (int) $direct_id > 0) {
            $rating_id = $direct_id;
          }
        } catch (\Exception $e) {
          \Drupal::logger('dga_rating')->error('Direct DB insert failed: @msg', ['@msg' => $e->getMessage()]);
        }
      }
      
      // If still failed after fallback, return error
      if ($rating_id === FALSE || !is_numeric($rating_id) || (int) $rating_id <= 0) {
        $error_msg = $is_arabic ? 'فشل حفظ التقييم.' : 'Failed to save rating.';
        return new JsonResponse([
          'success' => FALSE,
          'message' => $getErrorMessage('Failed to save rating.', $error_msg),
        ], 500);
      }
    }

    // Verify the record exists - but don't fail if there's a slight delay
    $db = \Drupal::database();
    
    // Try verification up to 3 times with small delay (in case of replication lag)
    $verify = NULL;
    for ($i = 0; $i < 3; $i++) {
      $verify = $db->select('dga_rating', 'r')
        ->fields('r', ['id', 'rating', 'url', 'user_id'])
        ->condition('r.id', (int) $rating_id)
        ->execute()
        ->fetchObject();
      
      if ($verify) {
        break;
      }
      
      // Small delay before retry
      if ($i < 2) {
        usleep(100000); // 0.1 second
      }
    }

    // Verification is optional - if record not found immediately, continue anyway
    // (may be due to replication lag or timing)

    // Get stats
    $route_match = \Drupal::routeMatch();
    $node = $route_match->getParameter('node');
    $stats = ['average' => 0.0, 'count' => 0];

    if ($node && $node instanceof Node) {
      $entity_id_for_query = $node->id();
      if ($entity_id_for_query > 0) {
        $stats = $this->ratingService->getStatistics('node', $entity_id_for_query, $url);
      }
    }

    if (($stats['count'] ?? 0) == 0) {
      $stats = $this->ratingService->getStatisticsByUrl($url);
    }

    // Get current language for thank you message.
    $language_manager = \Drupal::languageManager();
    $current_lang = $language_manager->getCurrentLanguage()->getId();
    $is_arabic = ($current_lang === 'ar');
    
    // Load thank you message from config.
    $config = $this->config('dga_rating.settings');
    $thank_you_message = $is_arabic 
      ? ($config->get('thank_you_ar') ?: $config->get('thank_you_en') ?: 'Thank you for your feedback!')
      : ($config->get('thank_you_en') ?: 'Thank you for your feedback!');

    return new JsonResponse([
      'success' => TRUE,
      'message' => $thank_you_message,
      'rating' => $rating,
      'rating_id' => (int) $rating_id,
      'statistics' => [
        'average' => (float) ($stats['average'] ?? 0.0),
        'count' => (int) ($stats['count'] ?? 0),
      ],
      'average' => number_format($stats['average'] ?? 0.0, 1, '.', ''),
      'count' => (int) ($stats['count'] ?? 0),
    ]);
  }

  /**
   * Refreshes the rating block statistics via AJAX.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with updated statistics.
   */
  public function refreshBlock(Request $request) {
    $url = $request->query->get('url');
    $entity_type = $request->query->get('entity_type');
    $entity_id = $request->query->get('entity_id');

    if (!$url) {
      $url = \Drupal::service('path.current')->getPath();
    }

    // Normalize URL
    $url = trim($url);
    if (empty($url) || $url === '/') {
      $url = '/';
    } else {
      if (preg_match('#^/[a-z]{2}(/.*)?$#', $url, $matches)) {
        $url = isset($matches[1]) ? $matches[1] : '/';
        if (empty($url)) $url = '/';
      }
      $url = rtrim($url, '/') ?: '/';
    }

    // Get updated statistics
    $stats = ['average' => 0.0, 'count' => 0];
    if ($entity_type && $entity_id && (int) $entity_id > 0) {
      $stats = $this->ratingService->getStatistics($entity_type, (int) $entity_id, $url);
    } else {
      $stats = $this->ratingService->getStatisticsByUrl($url);
    }

    return new JsonResponse([
      'success' => TRUE,
      'statistics' => [
        'average' => (float) ($stats['average'] ?? 0.0),
        'count' => (int) ($stats['count'] ?? 0),
      ],
      'average' => number_format($stats['average'] ?? 0.0, 1, '.', ''),
      'count' => (int) ($stats['count'] ?? 0),
    ]);
  }

  public function getStats(Request $request) {
    $url = $request->query->get('url');
    $entity_type = $request->query->get('entity_type');
    $entity_id = $request->query->get('entity_id');

    $stats = ['average' => 0.0, 'count' => 0];

    if ($entity_type && $entity_id) {
      if ($url) {
        $stats = $this->ratingService->getStatistics($entity_type, (int) $entity_id, $url);
      } else {
        $stats = $this->ratingService->getStatisticsByEntity($entity_type, (int) $entity_id);
      }
    } elseif ($url) {
      $stats = $this->ratingService->getStatisticsByUrl($url);
    }

    return new JsonResponse(['success' => TRUE, 'statistics' => $stats]);
  }
}
