<?php

namespace Drupal\dga_rating\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DGA Rating widget text settings.
 */
class DgaRatingSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dga_rating.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_rating_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dga_rating.settings');

    $form['closed_state'] = [
      '#type' => 'details',
      '#title' => $this->t('Closed State (Default View)'),
      '#description' => $this->t('Text displayed when the rating widget is closed.'),
      '#open' => TRUE,
    ];

    $form['closed_state']['average_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Average rating text'),
      '#description' => $this->t('Text before the average rating number.'),
      '#default_value' => $config->get('average_text') ?? 'This service is rated with an average of',
      '#required' => TRUE,
    ];

    $form['closed_state']['reviews_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reviews label'),
      '#description' => $this->t('Label for the number of reviews (e.g., "reviews", "ratings").'),
      '#default_value' => $config->get('reviews_text') ?? 'reviews',
      '#required' => TRUE,
    ];

    $form['closed_state']['rate_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rate button text'),
      '#description' => $this->t('Text on the button to open the rating form.'),
      '#default_value' => $config->get('rate_button_text') ?? 'Rate this service',
      '#required' => TRUE,
    ];

    $form['form_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Rating Form'),
      '#description' => $this->t('Text displayed in the rating form.'),
      '#open' => TRUE,
    ];

    $form['form_section']['form_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Form title'),
      '#description' => $this->t('Main title of the rating form.'),
      '#default_value' => $config->get('form_title') ?? 'Tell us what you think of this service',
      '#required' => TRUE,
    ];

    $form['form_section']['form_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Form description'),
      '#description' => $this->t('Instructions or description shown below the form title.'),
      '#default_value' => $config->get('form_description') ?? "Please don't include personal or financial information. Your review will be submitted and recorded to improve services.",
      '#required' => TRUE,
    ];

    $form['form_section']['close_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Close button text'),
      '#default_value' => $config->get('close_button_text') ?? 'Close',
      '#required' => TRUE,
    ];

    $form['form_section']['rating_question'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rating question'),
      '#description' => $this->t('Question asking users to rate the service.'),
      '#default_value' => $config->get('rating_question') ?? 'How would you rate this service?',
      '#required' => TRUE,
    ];

    $form['form_section']['rating_instructions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rating instructions'),
      '#description' => $this->t('Instructions explaining the rating scale.'),
      '#default_value' => $config->get('rating_instructions') ?? 'Rate your experience from (1) poor to (5) excellent',
      '#required' => TRUE,
    ];

    $form['form_section']['rating_aria_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rating ARIA label'),
      '#description' => $this->t('Accessibility label for the rating stars.'),
      '#default_value' => $config->get('rating_aria_label') ?? 'Select rating from 1 to 5',
      '#required' => TRUE,
    ];

    $form['form_section']['feedback_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Feedback label'),
      '#default_value' => $config->get('feedback_label') ?? 'Feedback',
      '#required' => TRUE,
    ];

    $form['form_section']['submit_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit button text'),
      '#default_value' => $config->get('submit_button_text') ?? 'Submit',
      '#required' => TRUE,
    ];

    $form['submitted_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Submitted State'),
      '#description' => $this->t('Text displayed after a rating is submitted.'),
      '#open' => FALSE,
    ];

    $form['submitted_section']['submitted_rating_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submitted rating text'),
      '#description' => $this->t('Text shown before the user\'s rating.'),
      '#default_value' => $config->get('submitted_rating_text') ?? 'You rated this service as',
      '#required' => TRUE,
    ];

    $form['submitted_section']['submitted_success_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Success message'),
      '#description' => $this->t('Message shown when feedback is successfully submitted.'),
      '#default_value' => $config->get('submitted_success_text') ?? 'Your feedback is submitted!',
      '#required' => TRUE,
    ];

    $form['labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Rating Labels'),
      '#description' => $this->t('Labels for individual star ratings.'),
      '#open' => FALSE,
    ];

    $form['labels']['star_singular'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Star (singular)'),
      '#default_value' => $config->get('star_singular') ?? 'star',
      '#required' => TRUE,
    ];

    $form['labels']['star_plural'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stars (plural)'),
      '#default_value' => $config->get('star_plural') ?? 'stars',
      '#required' => TRUE,
    ];

    $form['security_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Security & Limits'),
      '#description' => $this->t('Configure security settings and input limits.'),
      '#open' => FALSE,
    ];

    $form['security_section']['rate_limit_max_submissions'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Maximum submissions per IP'),
      '#description' => $this->t('Maximum number of submissions allowed per IP address within the time window. Set to 0 to disable rate limiting.'),
      '#default_value' => $config->get('rate_limit_max_submissions') ?? 20,
      '#min' => 0,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    $form['security_section']['rate_limit_time_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate limit: Time window (seconds)'),
      '#description' => $this->t('Time window in seconds for rate limiting (e.g., 3600 = 1 hour).'),
      '#default_value' => $config->get('rate_limit_time_window') ?? 3600,
      '#min' => 60,
      '#max' => 86400,
      '#required' => TRUE,
      '#suffix' => '<div class="description">' . $this->t('Common values: 3600 (1 hour), 7200 (2 hours), 86400 (24 hours)') . '</div>',
    ];

    $form['security_section']['feedback_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum feedback text length'),
      '#description' => $this->t('Maximum number of characters allowed in feedback text.'),
      '#default_value' => $config->get('feedback_max_length') ?? 5000,
      '#min' => 100,
      '#max' => 50000,
      '#required' => TRUE,
    ];

    $form['behavior_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Widget Behavior'),
      '#description' => $this->t('Configure widget behavior after submission.'),
      '#open' => FALSE,
    ];

    $form['behavior_section']['refresh_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Refresh delay (seconds)'),
      '#description' => $this->t('Time in seconds to wait before refreshing the widget after a successful submission. Default: 3 seconds.'),
      '#default_value' => ($config->get('refresh_delay') ?? 3000) / 1000, // Convert ms to seconds for display
      '#min' => 1,
      '#max' => 30,
      '#step' => 0.5,
      '#required' => TRUE,
      '#suffix' => '<div class="description">' . $this->t('Common values: 2 seconds, 3 seconds, 5 seconds') . '</div>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dga_rating.settings');

    // Closed state
    $config->set('average_text', $form_state->getValue('average_text'));
    $config->set('reviews_text', $form_state->getValue('reviews_text'));
    $config->set('rate_button_text', $form_state->getValue('rate_button_text'));

    // Form section
    $config->set('form_title', $form_state->getValue('form_title'));
    $config->set('form_description', $form_state->getValue('form_description'));
    $config->set('close_button_text', $form_state->getValue('close_button_text'));
    $config->set('rating_question', $form_state->getValue('rating_question'));
    $config->set('rating_instructions', $form_state->getValue('rating_instructions'));
    $config->set('rating_aria_label', $form_state->getValue('rating_aria_label'));
    $config->set('feedback_label', $form_state->getValue('feedback_label'));
    $config->set('submit_button_text', $form_state->getValue('submit_button_text'));

    // Submitted section
    $config->set('submitted_rating_text', $form_state->getValue('submitted_rating_text'));
    $config->set('submitted_success_text', $form_state->getValue('submitted_success_text'));

    // Labels
    $config->set('star_singular', $form_state->getValue('star_singular'));
    $config->set('star_plural', $form_state->getValue('star_plural'));

    // Security section
    $config->set('rate_limit_max_submissions', (int) $form_state->getValue('rate_limit_max_submissions'));
    $config->set('rate_limit_time_window', (int) $form_state->getValue('rate_limit_time_window'));
    $config->set('feedback_max_length', (int) $form_state->getValue('feedback_max_length'));

    // Behavior section - convert seconds to milliseconds for storage
    $refresh_delay_seconds = (float) $form_state->getValue('refresh_delay');
    $config->set('refresh_delay', (int) ($refresh_delay_seconds * 1000));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}

