<?php

namespace Drupal\dga_rating\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DGA Rating module translations (English & Arabic).
 */
class RatingTranslationForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dga_rating.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dga_rating_translation_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dga_rating.settings');

    $form['#attached']['library'][] = 'dga_rating/admin';

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure all front-end text for the DGA Rating module in English and Arabic. These translations will be displayed dynamically based on the current site language.') . '</p>',
    ];

    // Closed State Section
    $form['closed_state'] = [
      '#type' => 'details',
      '#title' => $this->t('Closed State (Default View)'),
      '#description' => $this->t('Text displayed when the rating widget is closed.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['closed_state']['average_text'] = $this->buildBilingualField(
      'average_text',
      $this->t('Average rating text'),
      $this->t('Text before the average rating number.'),
      $config->get('average_text_en') ?? 'This service is rated with an average of',
      $config->get('average_text_ar') ?? 'تم تقييم هذه الخدمة بمتوسط',
      TRUE
    );

    $form['closed_state']['reviews_text'] = $this->buildBilingualField(
      'reviews_text',
      $this->t('Reviews label'),
      $this->t('Label for the number of reviews (e.g., "reviews", "ratings").'),
      $config->get('reviews_text_en') ?? 'reviews',
      $config->get('reviews_text_ar') ?? 'تقييم',
      TRUE
    );

    $form['closed_state']['rate_button_text'] = $this->buildBilingualField(
      'rate_button_text',
      $this->t('Rate button text'),
      $this->t('Text on the button to open the rating form.'),
      $config->get('rate_button_text_en') ?? 'Rate this service',
      $config->get('rate_button_text_ar') ?? 'قيّم هذه الخدمة',
      TRUE
    );

    // Form Section
    $form['form_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Rating Form'),
      '#description' => $this->t('Text displayed in the rating form.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['form_section']['form_title'] = $this->buildBilingualField(
      'form_title',
      $this->t('Form title'),
      $this->t('Main title of the rating form.'),
      $config->get('form_title_en') ?? 'Tell us what you think of this service',
      $config->get('form_title_ar') ?? 'أخبرنا برأيك في هذه الخدمة',
      TRUE
    );

    $form['form_section']['form_description'] = $this->buildBilingualTextarea(
      'form_description',
      $this->t('Form description'),
      $this->t('Instructions or description shown below the form title.'),
      $config->get('form_description_en') ?? "Please don't include personal or financial information. Your review will be submitted and recorded to improve services.",
      $config->get('form_description_ar') ?? 'يرجى عدم تضمين معلومات شخصية أو مالية. سيتم إرسال مراجعتك وتسجيلها لتحسين الخدمات.',
      TRUE
    );

    $form['form_section']['close_button_text'] = $this->buildBilingualField(
      'close_button_text',
      $this->t('Close button text'),
      $this->t('Text on the close button.'),
      $config->get('close_button_text_en') ?? 'Close',
      $config->get('close_button_text_ar') ?? 'إغلاق',
      TRUE
    );

    $form['form_section']['rating_question'] = $this->buildBilingualField(
      'rating_question',
      $this->t('Rating question'),
      $this->t('Question asking users to rate the service.'),
      $config->get('rating_question_en') ?? 'How would you rate this service?',
      $config->get('rating_question_ar') ?? 'كيف تقيم هذه الخدمة؟',
      TRUE
    );

    $form['form_section']['rating_instructions'] = $this->buildBilingualField(
      'rating_instructions',
      $this->t('Rating instructions'),
      $this->t('Instructions explaining the rating scale.'),
      $config->get('rating_instructions_en') ?? 'Rate your experience from (1) poor to (5) excellent',
      $config->get('rating_instructions_ar') ?? 'قيم تجربتك من (1) ضعيف إلى (5) ممتاز',
      TRUE
    );

    $form['form_section']['rating_aria_label'] = $this->buildBilingualField(
      'rating_aria_label',
      $this->t('Rating ARIA label'),
      $this->t('Accessibility label for the rating stars.'),
      $config->get('rating_aria_label_en') ?? 'Select rating from 1 to 5',
      $config->get('rating_aria_label_ar') ?? 'اختر التقييم من 1 إلى 5',
      TRUE
    );

    $form['form_section']['feedback_label'] = $this->buildBilingualField(
      'feedback_label',
      $this->t('Feedback label'),
      $this->t('Label for the feedback textarea.'),
      $config->get('feedback_label_en') ?? 'Feedback',
      $config->get('feedback_label_ar') ?? 'التعليقات',
      TRUE
    );

    $form['form_section']['submit_button_text'] = $this->buildBilingualField(
      'submit_button_text',
      $this->t('Submit button text'),
      $this->t('Text on the submit button.'),
      $config->get('submit_button_text_en') ?? 'Submit',
      $config->get('submit_button_text_ar') ?? 'إرسال',
      TRUE
    );

    // Submitted Section
    $form['submitted_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Submitted State'),
      '#description' => $this->t('Text displayed after a rating is submitted.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['submitted_section']['submitted_rating_text'] = $this->buildBilingualField(
      'submitted_rating_text',
      $this->t('Submitted rating text'),
      $this->t('Text shown before the user\'s rating.'),
      $config->get('submitted_rating_text_en') ?? 'You rated this service as',
      $config->get('submitted_rating_text_ar') ?? 'قيمت هذه الخدمة بـ',
      TRUE
    );

    $form['submitted_section']['submitted_success_text'] = $this->buildBilingualField(
      'submitted_success_text',
      $this->t('Success message'),
      $this->t('Message shown when feedback is successfully submitted.'),
      $config->get('submitted_success_text_en') ?? 'Your feedback is submitted!',
      $config->get('submitted_success_text_ar') ?? 'تم إرسال ملاحظاتك!',
      TRUE
    );

    $form['submitted_section']['thank_you'] = $this->buildBilingualField(
      'thank_you',
      $this->t('Thank you message'),
      $this->t('Thank you message shown after submission.'),
      $config->get('thank_you_en') ?? 'Thank you for your feedback!',
      $config->get('thank_you_ar') ?? 'شكراً لملاحظاتك!',
      TRUE
    );

    // Labels Section
    $form['labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Rating Labels'),
      '#description' => $this->t('Labels for individual star ratings.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['labels']['star_singular'] = $this->buildBilingualField(
      'star_singular',
      $this->t('Star (singular)'),
      $this->t('Text for a single star.'),
      $config->get('star_singular_en') ?? 'star',
      $config->get('star_singular_ar') ?? 'نجمة',
      TRUE
    );

    $form['labels']['star_plural'] = $this->buildBilingualField(
      'star_plural',
      $this->t('Stars (plural)'),
      $this->t('Text for multiple stars.'),
      $config->get('star_plural_en') ?? 'stars',
      $config->get('star_plural_ar') ?? 'نجوم',
      TRUE
    );

    // Error Messages Section
    $form['errors'] = [
      '#type' => 'details',
      '#title' => $this->t('Error Messages'),
      '#description' => $this->t('Error messages displayed to users.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['errors']['error_message'] = $this->buildBilingualField(
      'error_message',
      $this->t('General error message'),
      $this->t('General error message shown when something goes wrong.'),
      $config->get('error_message_en') ?? 'An error occurred. Please try again.',
      $config->get('error_message_ar') ?? 'حدث خطأ. يرجى المحاولة مرة أخرى.',
      TRUE
    );

    // Validation Messages Section
    $form['validation'] = [
      '#type' => 'details',
      '#title' => $this->t('Validation Messages'),
      '#description' => $this->t('Error messages shown when form validation fails. These appear as red text under each field.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['validation']['validation_rating_required'] = $this->buildBilingualField(
      'validation_rating_required',
      $this->t('Rating required message'),
      $this->t('Message shown when user tries to submit without selecting a rating.'),
      $config->get('validation_rating_required_en') ?? 'Please select a rating',
      $config->get('validation_rating_required_ar') ?? 'يرجى اختيار التقييم',
      TRUE
    );

    $form['validation']['validation_feedback_required'] = $this->buildBilingualField(
      'validation_feedback_required',
      $this->t('Feedback required message'),
      $this->t('Message shown when user tries to submit without providing feedback.'),
      $config->get('validation_feedback_required_en') ?? 'Please provide feedback',
      $config->get('validation_feedback_required_ar') ?? 'يرجى تقديم التعليقات',
      TRUE
    );

    $form['validation']['validation_feedback_too_long'] = $this->buildBilingualField(
      'validation_feedback_too_long',
      $this->t('Feedback too long message'),
      $this->t('Message shown when feedback exceeds the maximum length.'),
      $config->get('validation_feedback_too_long_en') ?? 'Feedback is too long',
      $config->get('validation_feedback_too_long_ar') ?? 'التعليقات طويلة جداً',
      TRUE
    );

    // Menu Items Section
    $form['menu_items'] = [
      '#type' => 'details',
      '#title' => $this->t('Menu Items'),
      '#description' => $this->t('Translations for admin menu items.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['menu_items']['menu_title_dga_rating'] = $this->buildBilingualField(
      'menu_title_dga_rating',
      $this->t('DGA Rating (Main Menu)'),
      $this->t('Title for the main DGA Rating menu item in the sidebar.'),
      $config->get('menu_title_dga_rating_en') ?? 'DGA Rating',
      $config->get('menu_title_dga_rating_ar') ?? 'تقييم DGA',
      TRUE
    );

    $form['menu_items']['menu_title_dashboard'] = $this->buildBilingualField(
      'menu_title_dashboard',
      $this->t('Rating Dashboard'),
      $this->t('Title for the Rating Dashboard menu item/tab.'),
      $config->get('menu_title_dashboard_en') ?? 'Rating Dashboard',
      $config->get('menu_title_dashboard_ar') ?? 'لوحة تحكم التقييمات',
      TRUE
    );

    $form['menu_items']['menu_title_settings'] = $this->buildBilingualField(
      'menu_title_settings',
      $this->t('Settings'),
      $this->t('Title for the Settings menu item/tab.'),
      $config->get('menu_title_settings_en') ?? 'Settings',
      $config->get('menu_title_settings_ar') ?? 'الإعدادات',
      TRUE
    );

    $form['menu_items']['menu_title_translations'] = $this->buildBilingualField(
      'menu_title_translations',
      $this->t('Translations'),
      $this->t('Title for the Translations menu item/tab.'),
      $config->get('menu_title_translations_en') ?? 'Translations',
      $config->get('menu_title_translations_ar') ?? 'الترجمات',
      TRUE
    );

    return parent::buildForm($form, $form_state);
  }

  /**
   * Builds a bilingual text field with English and Arabic columns.
   *
   * @param string $field_name
   *   Base field name (without _en or _ar suffix).
   * @param string $title
   *   Field title.
   * @param string $description
   *   Field description.
   * @param string $default_en
   *   Default English value.
   * @param string $default_ar
   *   Default Arabic value.
   * @param bool $required
   *   Whether the field is required.
   *
   * @return array
   *   Form field array.
   */
  protected function buildBilingualField($field_name, $title, $description, $default_en, $default_ar, $required = FALSE) {
    $config = $this->config('dga_rating.settings');

    $field = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-wrapper']],
      '#tree' => TRUE,
    ];

    $field['label'] = [
      '#type' => 'markup',
      '#markup' => '<div class="bilingual-field-label"><strong>' . $title . '</strong></div>',
    ];

    if ($description) {
      $field['description'] = [
        '#type' => 'markup',
        '#markup' => '<div class="description">' . $description . '</div>',
      ];
    }

    $field['columns'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-columns', 'clearfix']],
      '#tree' => TRUE,
    ];

    $field['columns']['en'] = [
      '#type' => 'textfield',
      '#title' => $this->t('English'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_en') ?? $default_en,
      '#required' => $required,
      '#attributes' => ['class' => ['bilingual-field-en']],
      '#prefix' => '<div class="bilingual-column bilingual-column-en">',
      '#suffix' => '</div>',
    ];

    $field['columns']['ar'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Arabic'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_ar') ?? $default_ar,
      '#required' => $required,
      '#attributes' => ['class' => ['bilingual-field-ar']],
      '#prefix' => '<div class="bilingual-column bilingual-column-ar">',
      '#suffix' => '</div>',
    ];

    return $field;
  }

  /**
   * Builds a bilingual textarea field with English and Arabic columns.
   *
   * @param string $field_name
   *   Base field name (without _en or _ar suffix).
   * @param string $title
   *   Field title.
   * @param string $description
   *   Field description.
   * @param string $default_en
   *   Default English value.
   * @param string $default_ar
   *   Default Arabic value.
   * @param bool $required
   *   Whether the field is required.
   *
   * @return array
   *   Form field array.
   */
  protected function buildBilingualTextarea($field_name, $title, $description, $default_en, $default_ar, $required = FALSE) {
    $config = $this->config('dga_rating.settings');

    $field = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-wrapper']],
      '#tree' => TRUE,
    ];

    $field['label'] = [
      '#type' => 'markup',
      '#markup' => '<div class="bilingual-field-label"><strong>' . $title . '</strong></div>',
    ];

    if ($description) {
      $field['description'] = [
        '#type' => 'markup',
        '#markup' => '<div class="description">' . $description . '</div>',
      ];
    }

    $field['columns'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bilingual-field-columns', 'clearfix']],
      '#tree' => TRUE,
    ];

    $field['columns']['en'] = [
      '#type' => 'textarea',
      '#title' => $this->t('English'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_en') ?? $default_en,
      '#required' => $required,
      '#rows' => 3,
      '#attributes' => ['class' => ['bilingual-field-en']],
      '#prefix' => '<div class="bilingual-column bilingual-column-en">',
      '#suffix' => '</div>',
    ];

    $field['columns']['ar'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Arabic'),
      '#title_display' => 'before',
      '#default_value' => $config->get($field_name . '_ar') ?? $default_ar,
      '#required' => $required,
      '#rows' => 3,
      '#attributes' => ['class' => ['bilingual-field-ar']],
      '#prefix' => '<div class="bilingual-column bilingual-column-ar">',
      '#suffix' => '</div>',
    ];

    return $field;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dga_rating.settings');
    
    // Helper function to safely get form values
    $getValue = function($path) use ($form_state) {
      $value = $form_state->getValue($path);
      return $value !== NULL ? $value : '';
    };

    // Closed state
    $config->set('average_text_en', $getValue(['closed_state', 'average_text', 'columns', 'en']));
    $config->set('average_text_ar', $getValue(['closed_state', 'average_text', 'columns', 'ar']));
    $config->set('reviews_text_en', $getValue(['closed_state', 'reviews_text', 'columns', 'en']));
    $config->set('reviews_text_ar', $getValue(['closed_state', 'reviews_text', 'columns', 'ar']));
    $config->set('rate_button_text_en', $getValue(['closed_state', 'rate_button_text', 'columns', 'en']));
    $config->set('rate_button_text_ar', $getValue(['closed_state', 'rate_button_text', 'columns', 'ar']));

    // Form section
    $config->set('form_title_en', $getValue(['form_section', 'form_title', 'columns', 'en']));
    $config->set('form_title_ar', $getValue(['form_section', 'form_title', 'columns', 'ar']));
    $config->set('form_description_en', $getValue(['form_section', 'form_description', 'columns', 'en']));
    $config->set('form_description_ar', $getValue(['form_section', 'form_description', 'columns', 'ar']));
    $config->set('close_button_text_en', $getValue(['form_section', 'close_button_text', 'columns', 'en']));
    $config->set('close_button_text_ar', $getValue(['form_section', 'close_button_text', 'columns', 'ar']));
    $config->set('rating_question_en', $getValue(['form_section', 'rating_question', 'columns', 'en']));
    $config->set('rating_question_ar', $getValue(['form_section', 'rating_question', 'columns', 'ar']));
    $config->set('rating_instructions_en', $getValue(['form_section', 'rating_instructions', 'columns', 'en']));
    $config->set('rating_instructions_ar', $getValue(['form_section', 'rating_instructions', 'columns', 'ar']));
    $config->set('rating_aria_label_en', $getValue(['form_section', 'rating_aria_label', 'columns', 'en']));
    $config->set('rating_aria_label_ar', $getValue(['form_section', 'rating_aria_label', 'columns', 'ar']));
    $config->set('feedback_label_en', $getValue(['form_section', 'feedback_label', 'columns', 'en']));
    $config->set('feedback_label_ar', $getValue(['form_section', 'feedback_label', 'columns', 'ar']));
    $config->set('submit_button_text_en', $getValue(['form_section', 'submit_button_text', 'columns', 'en']));
    $config->set('submit_button_text_ar', $getValue(['form_section', 'submit_button_text', 'columns', 'ar']));

    // Submitted section
    $config->set('submitted_rating_text_en', $getValue(['submitted_section', 'submitted_rating_text', 'columns', 'en']));
    $config->set('submitted_rating_text_ar', $getValue(['submitted_section', 'submitted_rating_text', 'columns', 'ar']));
    $config->set('submitted_success_text_en', $getValue(['submitted_section', 'submitted_success_text', 'columns', 'en']));
    $config->set('submitted_success_text_ar', $getValue(['submitted_section', 'submitted_success_text', 'columns', 'ar']));
    $config->set('thank_you_en', $getValue(['submitted_section', 'thank_you', 'columns', 'en']));
    $config->set('thank_you_ar', $getValue(['submitted_section', 'thank_you', 'columns', 'ar']));

    // Labels
    $config->set('star_singular_en', $getValue(['labels', 'star_singular', 'columns', 'en']));
    $config->set('star_singular_ar', $getValue(['labels', 'star_singular', 'columns', 'ar']));
    $config->set('star_plural_en', $getValue(['labels', 'star_plural', 'columns', 'en']));
    $config->set('star_plural_ar', $getValue(['labels', 'star_plural', 'columns', 'ar']));

    // Errors
    $config->set('error_message_en', $getValue(['errors', 'error_message', 'columns', 'en']));
    $config->set('error_message_ar', $getValue(['errors', 'error_message', 'columns', 'ar']));

    // Validation messages
    $config->set('validation_rating_required_en', $getValue(['validation', 'validation_rating_required', 'columns', 'en']));
    $config->set('validation_rating_required_ar', $getValue(['validation', 'validation_rating_required', 'columns', 'ar']));
    $config->set('validation_feedback_required_en', $getValue(['validation', 'validation_feedback_required', 'columns', 'en']));
    $config->set('validation_feedback_required_ar', $getValue(['validation', 'validation_feedback_required', 'columns', 'ar']));
    $config->set('validation_feedback_too_long_en', $getValue(['validation', 'validation_feedback_too_long', 'columns', 'en']));
    $config->set('validation_feedback_too_long_ar', $getValue(['validation', 'validation_feedback_too_long', 'columns', 'ar']));

    // Menu Items
    $config->set('menu_title_dga_rating_en', $getValue(['menu_items', 'menu_title_dga_rating', 'columns', 'en']));
    $config->set('menu_title_dga_rating_ar', $getValue(['menu_items', 'menu_title_dga_rating', 'columns', 'ar']));
    $config->set('menu_title_dashboard_en', $getValue(['menu_items', 'menu_title_dashboard', 'columns', 'en']));
    $config->set('menu_title_dashboard_ar', $getValue(['menu_items', 'menu_title_dashboard', 'columns', 'ar']));
    $config->set('menu_title_settings_en', $getValue(['menu_items', 'menu_title_settings', 'columns', 'en']));
    $config->set('menu_title_settings_ar', $getValue(['menu_items', 'menu_title_settings', 'columns', 'ar']));
    $config->set('menu_title_translations_en', $getValue(['menu_items', 'menu_title_translations', 'columns', 'en']));
    $config->set('menu_title_translations_ar', $getValue(['menu_items', 'menu_title_translations', 'columns', 'ar']));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
