<?php

namespace Drupal\dify_search_api\Plugin\search_api\backend;

use Drupal\dify\DifyClient;
use Drupal\dify_search_api\Traits\ArrayToCsvTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Plugin\search_api\data_type\value\TextValue;
use Drupal\search_api\Query\QueryInterface;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a backend for indexing items using Dify API.
 *
 * @SearchApiBackend(
 *   id = "dify",
 *   label = @Translation("Dify Backend"),
 *   description = @Translation("Index items using Dify API.")
 * )
 */
final class SearchApiDifyBackend extends BackendPluginBase implements PluginFormInterface {

  use PluginFormTrait, ArrayToCsvTrait;

  /**
   * Constructs a SearchApiDifyBackend object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    protected FileSystemInterface $file_system,
    protected StateInterface $state
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_system'),
      $container->get('state'),
    );
  }

  /**
   * The Dify client.
   *
   * @var \Drupal\dify\DifyClient
   */
  protected DifyClient $client;

  /**
   * Get the Dify client.
   *
   * @return \Drupal\dify\DifyClient
   *   The Dify client.
   */
  public function getClient(): DifyClient {
    if (!isset($this->client)) {
      // Get credentials from state (never exported)
      $base_url = $this->state->get('dify.knowledge.base_url');
      $api_key = $this->state->get('dify.knowledge.api_key');

      if (empty($base_url) || empty($api_key)) {
        throw new \RuntimeException('Dify credentials are not properly configured. Please configure them via the module settings.');
      }

      $this->client = new DifyClient(rtrim($base_url, '/'), $api_key);
    }
    return $this->client;
  }

  /**
   * Get the dataset ID from secure state.
   *
   * @return string
   *   The dataset ID.
   */
  protected function getDatasetId(): string {
    return $this->state->get('dify.knowledge.dataset') ?: '';
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Get current values from state (never exported)
    $current_base_url = $this->state->get('dify.knowledge.base_url');
    $current_api_key = $this->state->get('dify.knowledge.api_key');
    $current_dataset = $this->state->get('dify.knowledge.dataset');

    $form['credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('Knowledge Base Credentials'),
      '#open' => TRUE,
    ];

    $form['credentials']['api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your Dify knowledge base API key. This will be stored securely.'),
      '#default_value' => $current_api_key,
      '#attributes' => [
        'autocomplete' => 'new-password',
      ],
    ];

    $form['credentials']['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#description' => $this->t('The base URL for your Dify knowledge base API.'),
      '#default_value' => $current_base_url,
      '#required' => TRUE,
    ];

    $form['credentials']['dataset'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dataset ID'),
      '#description' => $this->t('The dataset ID to use for indexing. This will be stored securely.'),
      '#default_value' => $current_dataset,
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'details',
      '#title' => $this->t('Current Status'),
      '#open' => TRUE,
    ];

    $form['status']['info'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p><strong>Base URL:</strong> @url_status<br><strong>API Key:</strong> @key_status<br><strong>Dataset ID:</strong> @dataset_status</p><p><em>All credentials and configuration are stored securely and will not be exported with your configuration.</em></p>', [
        '@url_status' => !empty($current_base_url) ? $current_base_url : $this->t('Not configured'),
        '@key_status' => !empty($current_api_key) ? $this->t('Configured') : $this->t('Not configured'),
        '@dataset_status' => !empty($current_dataset) ? $current_dataset : $this->t('Not configured'),
      ]),
    ];



    // Metadata configuration section.
    $form['metadata'] = [
      '#type' => 'details',
      '#title' => $this->t('Metadata Configuration'),
      '#description' => $this->t('Configure how metadata is added to documents in Dify.'),
      '#open' => TRUE,
    ];

    $form['metadata']['add_content_url'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add content URL metadata'),
      '#description' => $this->t('Automatically add the relative URL of the content as metadata to each document in Dify. This allows the chatbot to provide links back to the original content.'),
      '#default_value' => $this->configuration['add_content_url'] ?? TRUE,
    ];

    $form['metadata']['content_url_field_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Content URL field name'),
      '#description' => $this->t('The name of the metadata field that will store the content URL. Must use lowercase letters, numbers, and underscores only.'),
      '#default_value' => $this->configuration['content_url_field_name'] ?? 'content_url',
      '#pattern' => '^[a-z0-9_]+$',
      '#states' => [
        'visible' => [
          ':input[name="backend_config[metadata][add_content_url]"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="backend_config[metadata][add_content_url]"]' => ['checked' => TRUE],
        ],
      ],
    ];



    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $credentials = $form_state->getValue('credentials') ?? [];
    $base_url = $credentials['base_url'] ?? '';
    $api_key = $credentials['api_key'] ?? '';
    $dataset = $credentials['dataset'] ?? '';

    // Only validate if values are provided
    if (!empty($base_url) && !filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('credentials][base_url', $this->t('Please enter a valid URL.'));
    }

    if (!empty($api_key) && !preg_match('/^[a-zA-Z0-9_-]+$/', $api_key)) {
      $form_state->setErrorByName('credentials][api_key', $this->t('API key must contain only letters, numbers, hyphens, and underscores.'));
    }

    if (!empty($dataset) && !preg_match('/^[a-zA-Z0-9_-]+$/', $dataset)) {
      $form_state->setErrorByName('credentials][dataset', $this->t('Dataset ID must contain only letters, numbers, hyphens, and underscores.'));
    }
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    // Get credentials from the credentials group
    $credentials = $values['credentials'] ?? [];
    $api_key = $credentials['api_key'] ?? '';
    $base_url = $credentials['base_url'] ?? '';
    $dataset = $credentials['dataset'] ?? '';

    // Save credentials to state (never exported)
    if (!empty($api_key)) {
      $this->state->set('dify.knowledge.api_key', $api_key);
    }

    if (!empty($base_url)) {
      $this->state->set('dify.knowledge.base_url', $base_url);
    }

    if (!empty($dataset)) {
      $this->state->set('dify.knowledge.dataset', $dataset);
    }

    // Remove credentials from plugin configuration
    unset($values['credentials']);

    $this->setConfiguration($values);
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    return [
      'add_content_url' => TRUE,
      'content_url_field_name' => 'content_url',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function viewSettings(): array {
    $infos = [];
    try {
      $client = $this->getClient();
      $dataset = $client->getDocuments($this->getDatasetId(), limit: 1);

      $infos[] = [
        'label' => $this->t('Connection status'),
        'info' => $this->t('Connected to the Dify API.'),
        'status' => 'success',
      ];

      $infos[] = [
        'label' => $this->t('Documents in dataset'),
        'info' => $this->t('@count document(s)', [
          '@count' => $dataset['total'],
        ]),
      ];
    }
    catch (\Exception $e) {
      $infos[] = [
        'label' => $this->t('Connection status'),
        'info' => $this->t('An error occurred while trying to connect to the Dify API: @error', [
          '@error' => $e->getMessage(),
        ]),
        'status' => 'error',
      ];
    }

    return $infos;
  }

  /**
   * {@inheritDoc}
   */
  public function indexItems(IndexInterface $index, array $items): array {
    $objects = [];
    foreach ($items as $item) {
      $fields = $item->getFields();

      // Create an array of fields.
      $data = [];
      foreach ($fields as $field) {
        $data[$field->getFieldIdentifier()] = $this->buildFieldValues($field, $field->getType());
      }

      $response = $this->indexItemAsFile($data, $item);

      if (!empty($response)) {
        $objects[$item->getId()] = $response;
      }
    }

    return array_keys($objects);
  }



  /**
   * Index an item as a file.
   *
   * @param array $data
   *   The data to be indexed.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to be indexed.
   *
   * @return array
   *   The response from the Dify API.
   */
  protected function indexItemAsFile(array $data, ItemInterface $item): array {
    $real_file_path = $this->prepareCsvFile($item);
    $success = $this->arrayToCsv($data, $real_file_path);

    if (!$success) {
      throw new \RuntimeException("Unable to create CSV file for item: {$item->getId()}");
    }

    $item_id = $this->getItemId($item);
    $client = $this->getClient();

    try {
      $existing_document = $client->getDocumentByKeyword($this->getDatasetId(), $item_id);
      if ($existing_document && !empty($existing_document['name']) && is_string($existing_document['name']) && str_starts_with($existing_document['name'], $item_id)) {
        $response = $client->updateDocumentFromFile($this->getDatasetId(), $existing_document['id'], $real_file_path);
        $document_id = $existing_document['id'];
      }
      else {
        $response = $client->createDocumentFromFile($this->getDatasetId(), $real_file_path);
        $document_id = $response['document']['id'] ?? NULL;
      }

      // Add metadata with content URL if document was created/updated successfully.
      if (!empty($document_id)) {
        $this->addContentUrlMetadata($document_id, $item);
      }
    }
    catch (\Exception | GuzzleException $e) {
      $this->getLogger()->error('Error indexing item as file: @error', ['@error' => $e->getMessage()]);
      $response = [];
    }

    // Clean up temporary file.
    if (file_exists($real_file_path)) {
      unlink($real_file_path);
    }
    return $response;
  }

  /**
   * Prepare a CSV file for indexing.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to be indexed.
   *
   * @return bool|string
   *   The path to the CSV file. Returns FALSE if the file path could not be
   *   created.
   */
  protected function prepareCsvFile(ItemInterface $item): bool|string {
    $dir = 'private://search_api_dify';
    $this->file_system->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    return $this->file_system->realpath($this->file_system->createFilename($this->getItemId($item) . '.csv', $dir));
  }

  /**
   * {@inheritDoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    $client = $this->getClient();
    $existing_documents = $client->getAllDocuments($this->getDatasetId());

    foreach ($item_ids as $item_id) {
      $new_item_id = $this->getItemId($item_id);
      foreach ($existing_documents as $existing_document) {
        if (!empty($existing_document['name']) && is_string($existing_document['name']) && str_starts_with($existing_document['name'], $new_item_id)) {
          try {
            $client->deleteDocument($this->getDatasetId(), $existing_document['id']);
          }
          catch (\Exception | GuzzleException $e) {
            $this->getLogger()
              ->warning('An error occurred while trying to delete document: @error', [
                '@error' => $e->getMessage(),
              ]);
          }
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function deleteAllIndexItems(IndexInterface $index, $datasource_id = NULL): void {
    $client = $this->getClient();
    $documents_with_errors = $client->deleteAllDocuments($this->getDatasetId());
    if (!empty($documents_with_errors)) {
      $this->getLogger()
        ->warning('The following documents could not be deleted: @documents', [
          '@documents' => implode(', ', array_column($documents_with_errors, 'name')),
        ]);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function search(QueryInterface $query) {
    // @todo Implement search() method.
  }

  /**
   * Builds field values.
   *
   * @param \Drupal\search_api\Item\FieldInterface $field
   *   The field.
   * @param string $field_type
   *   The field type.
   *
   * @return string
   *   The fields value.
   */
  public function buildFieldValues(FieldInterface $field, string $field_type): string {
    $values = [];
    foreach ($field->getValues() as $value) {
      if ($value instanceof TextValue) {
        $values[] = $value->toText();
        continue;
      }
      $values[] = match ($field_type) {
        'string' => (string) $value,
        'boolean' => (boolean) $value,
        default => $value,
      };
    }
    return implode(', ', $values);
  }

  /**
   * Get a modified item ID.
   *
   * @param \Drupal\search_api\Item\ItemInterface|string $item
   *   A search API item or an item ID.
   *
   * @return string
   *   The modified item ID.
   */
  protected function getItemId(ItemInterface|string $item): string {
    $item_id = $item instanceof ItemInterface ? $item->getId() : $item;

    // Replace invalid characters for filenames and document names
    // Keep only alphanumeric characters, hyphens, underscores, and dots.
    $clean_id = preg_replace('/[^a-zA-Z0-9\-_.]/', '-', $item_id);

    // Remove multiple consecutive hyphens.
    $clean_id = preg_replace('/-+/', '-', $clean_id);

    // Remove leading/trailing hyphens.
    return trim($clean_id, '-');
  }

  /**
   * Generate a relative URL for the content item.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The search API item.
   *
   * @return string|null
   *   The relative URL of the content or NULL if it cannot be generated.
   */
  protected function generateContentUrl(ItemInterface $item): ?string {
    try {
      // Get the original object (entity) from the search API item.
      $entity = $item->getOriginalObject()->getValue();

      if (!$entity instanceof EntityInterface) {
        return NULL;
      }

      // Generate the URL using Drupal's URL system.
      $url = $entity->toUrl('canonical', ['absolute' => FALSE]);

      // Return the relative path.
      return $url->toString();
    }
    catch (\Exception $e) {
      // Log the error and return NULL if URL generation fails.
      $this->getLogger()->warning('Failed to generate URL for item @item: @error', [
        '@item' => $item->getId(),
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Add content URL metadata to a Dify document.
   *
   * @param string $document_id
   *   The Dify document ID.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The search API item.
   */
  protected function addContentUrlMetadata(string $document_id, ItemInterface $item): void {
    // Skip if content URL metadata is disabled.
    if (empty($this->configuration['add_content_url'])) {
      return;
    }

    $client = $this->getClient();
    $content_url = $this->generateContentUrl($item);

    // Skip if we couldn't generate a URL.
    if (empty($content_url)) {
      return;
    }

    $field_name = $this->configuration['content_url_field_name'] ?? 'content_url';

    try {
      // Ensure the content URL metadata field exists.
      $metadata_field = $client->ensureMetadataField($this->getDatasetId(), $field_name, 'string');

      if (empty($metadata_field) || empty($metadata_field['id'])) {
        $this->getLogger()->warning('Failed to create or retrieve @field_name metadata field for dataset @dataset', [
          '@field_name' => $field_name,
          '@dataset' => $this->getDatasetId(),
        ]);
        return;
      }

      // Assign the URL metadata to the document.
      $operation_data = [
        [
          'document_id' => $document_id,
          'metadata_list' => [
            [
              'id' => $metadata_field['id'],
              'value' => $content_url,
              'name' => $field_name,
            ],
          ],
        ],
      ];

      $client->assignDocumentMetadata($this->getDatasetId(), $operation_data);

      $this->getLogger()->info('Added content URL metadata @url to document @document using field @field', [
        '@url' => $content_url,
        '@document' => $document_id,
        '@field' => $field_name,
      ]);
    }
    catch (\Exception | GuzzleException $e) {
      $this->getLogger()->warning('Failed to add content URL metadata to document @document: @error', [
        '@document' => $document_id,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDiscouragedProcessors(): array {
    return [
      'ignorecase',
      'stemmer',
      'stopwords',
      'tokenizer',
      'transliteration',
      'type_boost',
      'number_field_boost',
      'highlight',
      'ignore_character',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    try {
      $this->getClient()->getKnowledgeBaseList();
      return TRUE;
    }
    catch (\Exception | GuzzleException $e) {
      return FALSE;
    }
  }

}
