<?php

namespace Drupal\dify_widget_official\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dify\Service\DifyConfigurationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Dify Widget Official configuration form.
 */
class DifyWidgetOfficialConfigForm extends FormBase {

  /**
   * The Dify configuration service.
   *
   * @var \Drupal\dify\Service\DifyConfigurationService
   */
  protected DifyConfigurationService $configurationService;

  /**
   * Constructs a DifyWidgetOfficialConfigForm object.
   *
   * @param \Drupal\dify\Service\DifyConfigurationService $configuration_service
   *   The Dify configuration service.
   */
  public function __construct(DifyConfigurationService $configuration_service) {
    $this->configurationService = $configuration_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify.configuration_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dify_widget_official_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configurationService->getConfiguration('widget_official', 'default') ?? [];

    $form['enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable iframe widget site-wide'),
      '#default_value' => !empty($config['token']) && !empty($config['base_url']),
      '#description' => $this->t('Show the Dify iframe widget on all pages for authenticated users.'),
    ];

    $form['config'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuration'),
      '#open' => TRUE,
    ];

    $form['config']['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#description' => $this->t('The base URL for your Dify service.'),
      '#default_value' => $config['base_url'] ?? '',
      '#required' => TRUE,
    ];

    $form['config']['token'] = [
      '#type' => 'password',
      '#title' => $this->t('Iframe Token'),
      '#description' => $this->t('Your Dify iframe widget token. Laissez vide pour conserver le token actuel.'),
      '#attributes' => [
        'autocomplete' => 'new-password',
        'placeholder' => !empty($config['token']) ? $this->t('Token configuré - laissez vide pour conserver') : $this->t('Entrez votre token'),
      ],
    ];

    $form['exclusions'] = [
      '#type' => 'details',
      '#title' => $this->t('Page Exclusions'),
      '#open' => FALSE,
    ];

    $form['exclusions']['exclude_chatbot'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable on /chatbot page'),
      '#default_value' => $config['exclude_chatbot'] ?? TRUE,
      '#description' => $this->t('Prevent the widget from showing on the fullscreen chatbot page.'),
    ];

    $form['exclusions']['authenticated_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Uniquement pour les utilisateurs connectés'),
      '#default_value' => $config['authenticated_only'] ?? TRUE,
      '#description' => $this->t('Afficher le widget uniquement pour les utilisateurs authentifiés. Si décoché, le widget sera visible pour tous les visiteurs.'),
    ];

    if (!empty($config)) {
      $form['status'] = [
        '#type' => 'details',
        '#title' => $this->t('Current Status'),
        '#open' => TRUE,
      ];

      $form['status']['info'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<p><strong>Token:</strong> @token_status<br><strong>Base URL:</strong> @url_status</p>', [
          '@token_status' => !empty($config['token']) ? $this->t('Configured') : $this->t('Not configured'),
          '@url_status' => $config['base_url'] ?: $this->t('Not configured'),
        ]),
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $base_url = $form_state->getValue('base_url');
    $token = $form_state->getValue('token');

    if (!empty($base_url) && !filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('base_url', $this->t('Please enter a valid URL.'));
    }

    if (!empty($token) && !preg_match('/^[a-zA-Z0-9_-]+$/', $token)) {
      $form_state->setErrorByName('token', $this->t('Token must contain only letters, numbers, hyphens, and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get existing configuration to preserve token if not provided
    $existing_config = $this->configurationService->getConfiguration('widget_official', 'default') ?? [];

    // Preserve existing token if new one is empty
    $new_token = $form_state->getValue('token');
    $token = !empty($new_token) ? $new_token : ($existing_config['token'] ?? '');

    $config = [
      'token' => $token,
      'base_url' => $form_state->getValue('base_url'),
      'exclude_chatbot' => $form_state->getValue('exclude_chatbot'),
      'authenticated_only' => $form_state->getValue('authenticated_only'),
    ];

    try {
      $this->configurationService->saveConfiguration('widget_official', 'default', $config);
      $this->messenger()->addMessage($this->t('Widget configuration has been saved.'));
    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error saving configuration: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

}
