<?php

namespace Drupal\dify_widget_vanilla\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dify\Service\DifyConfigurationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Dify Widget JS Configuration List form.
 */
class DifyWidgetVanillaConfigListForm extends FormBase {

  /**
   * The Dify configuration service.
   *
   * @var \Drupal\dify\Service\DifyConfigurationService
   */
  protected DifyConfigurationService $configurationService;

  /**
   * Constructs a DifyWidgetVanillaConfigListForm object.
   *
   * @param \Drupal\dify\Service\DifyConfigurationService $configuration_service
   *   The Dify configuration service.
   */
  public function __construct(DifyConfigurationService $configuration_service) {
    $this->configurationService = $configuration_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify.configuration_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dify_widget_vanilla_config_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['description'] = [
      '#markup' => '<p>' . $this->t('Manage Dify JS Widget configurations. Each configuration can be used by multiple widget blocks.') . '</p>',
    ];

    $form['add_new'] = [
      '#type' => 'link',
      '#title' => $this->t('Add New Configuration'),
      '#url' => Url::fromRoute('dify_widget_vanilla.config_add'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['configurations'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Name'),
        $this->t('Base URL'),
        $this->t('API Token'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No configurations found.'),
    ];

    $configuration_names = $this->configurationService->getConfigurationNames('widget_js');

    foreach ($configuration_names as $name) {
      $config = $this->configurationService->getConfiguration('widget_js', $name);

      if ($config) {
        $form['configurations'][$name] = [
          'name' => [
            '#markup' => '<strong>' . $name . '</strong>',
          ],
          'base_url' => [
            '#markup' => !empty($config['base_url']) ? $config['base_url'] : $this->t('Not configured'),
          ],
          'api_token' => [
            '#markup' => !empty($config['api_token']) ?
              '<span style="color: green;">✓ Configured</span>' :
              '<span style="color: red;">✗ Not configured</span>',
          ],
          'operations' => [
            '#type' => 'operations',
            '#links' => $this->getOperationLinks($name),
          ],
        ];
      }
    }

    return $form;
  }

  /**
   * Gets operation links for a configuration.
   *
   * @param string $name
   *   The configuration name.
   *
   * @return array
   *   Array of operation links.
   */
  protected function getOperationLinks(string $name): array {
    $links = [];

    $links['edit'] = [
      'title' => $this->t('Edit'),
      'url' => Url::fromRoute('dify_widget_vanilla.config_edit', ['name' => $name]),
    ];

    $links['delete'] = [
      'title' => $this->t('Delete'),
      'url' => Url::fromRoute('dify_widget_vanilla.config_delete', ['name' => $name]),
    ];

    return $links;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form doesn't have a submit action
  }

}
