<?php

namespace Drupal\dify_augmented_search\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\State\StateInterface;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an AI-powered search enhancement block.
 *
 * @Block(
 *   id = "dify_augmented_search_block",
 *   admin_label = @Translation("Dify Augmented Search"),
 *   category = @Translation("Dify")
 * )
 */
class AugmentedSearchBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected StateInterface $state;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, StateInterface $state) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->state = $state;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'markdown_theme' => 'dark',
      'enable_toggle' => FALSE,
      'form_selectors' => '',
      'toggle_label' => 'AI Search',
      'toggle_default_state' => TRUE,
    ] + parent::defaultConfiguration();
  }

  protected function getDefaultColors() {
    return [
      'primary_color' => '#2563eb',
      'primary_hover' => '#1d4ed8',
      'primary_light' => '#3b82f6',
      'background_main' => '#f8f9fa',
      'background_secondary' => '#e9ecef',
      'background_tertiary' => '#dee2e6',
      'text_primary' => '#212529',
      'text_secondary' => '#6c757d',
      'text_muted' => '#adb5bd',
      'border_color' => '#e9ecef',
      'border_light' => '#f8f9fa',
      'response_bg' => '#ffffff',
      'loading_color' => '#007bff',
      'error_bg' => '#f8d7da',
      'error_text' => '#dc3545',
      'success_color' => '#28a745',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $block_uuid = $this->configuration['uuid'] ?? uniqid('block_', TRUE);
    $current_base_url = $this->state->get("dify_augmented_search.block_{$block_uuid}.base_url", '');
    $current_token = $this->state->get("dify_augmented_search.block_{$block_uuid}.token", '');

    $form['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Dify Base URL'),
      '#description' => $this->t('The base URL of your Dify service.'),
      '#default_value' => $current_base_url,
      '#required' => TRUE,
    ];

    $form['token'] = [
      '#type' => 'password',
      '#title' => $this->t('API Token'),
      '#description' => $this->t('Your Dify API token for chat messages. Leave empty to keep current token.'),
      '#attributes' => [
        'autocomplete' => 'new-password',
        'placeholder' => !empty($current_token) ? $this->t('Token configured - leave empty to keep') : $this->t('Enter your token'),
      ],
    ];

    $form['markdown_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Markdown Theme'),
      '#description' => $this->t('Choose the theme for markdown rendering in AI responses.'),
      '#default_value' => $this->configuration['markdown_theme'],
      '#options' => [
        'dark' => $this->t('Dark'),
        'light' => $this->t('Light'),
      ],
    ];

    // Toggle configuration section
    $form['toggle_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Search Toggle Configuration') . '</h3><hr>',
    ];

    $form['enable_toggle'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable search toggle'),
      '#description' => $this->t('Add a toggle to selected forms to enable/disable AI search.'),
      '#default_value' => $this->configuration['enable_toggle'] ?? FALSE,
    ];

    $form['form_selectors'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Form selectors'),
      '#description' => $this->t('CSS selectors for forms where the toggle should appear (one per line). Example: #views-exposed-form-knowledge-search-page-1'),
      '#default_value' => $this->configuration['form_selectors'] ?? '',
      '#rows' => 4,
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_toggle]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['toggle_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Toggle label'),
      '#description' => $this->t('Label text for the AI search toggle.'),
      '#default_value' => $this->configuration['toggle_label'] ?? $this->t('AI Search'),
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_toggle]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['toggle_default_state'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled by default'),
      '#description' => $this->t('Whether the AI search toggle should be checked by default.'),
      '#default_value' => $this->configuration['toggle_default_state'] ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="settings[enable_toggle]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $default_colors = $this->getDefaultColors();

    $form['colors_primary_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Primary Colors') . '</h3><hr>',
    ];

    $form['primary_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Color'),
      '#description' => $this->t('Main accent color for loading spinner and highlights.'),
      '#default_value' => $this->configuration['primary_color'] ?? $default_colors['primary_color'],
    ];

    $form['colors_background_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Background Colors') . '</h3><hr>',
    ];

    $form['background_main'] = [
      '#type' => 'color',
      '#title' => $this->t('Main Background'),
      '#description' => $this->t('Main background color of the search block.'),
      '#default_value' => $this->configuration['background_main'] ?? $default_colors['background_main'],
    ];

    $form['response_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Response Background'),
      '#description' => $this->t('Background color for AI response area.'),
      '#default_value' => $this->configuration['response_bg'] ?? $default_colors['response_bg'],
    ];

    $form['colors_text_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Text Colors') . '</h3><hr>',
    ];

    $form['text_primary'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Text'),
      '#description' => $this->t('Main text color.'),
      '#default_value' => $this->configuration['text_primary'] ?? $default_colors['text_primary'],
    ];

    $form['text_secondary'] = [
      '#type' => 'color',
      '#title' => $this->t('Secondary Text'),
      '#description' => $this->t('Secondary text color for query info.'),
      '#default_value' => $this->configuration['text_secondary'] ?? $default_colors['text_secondary'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);

    $values = $form_state->getValues();
    $block_uuid = $this->configuration['uuid'] ?? uniqid('block_', TRUE);
    $this->configuration['uuid'] = $block_uuid;
    $this->configuration['markdown_theme'] = $values['markdown_theme'];

    // Save toggle configuration
    $this->configuration['enable_toggle'] = $values['enable_toggle'];
    $this->configuration['form_selectors'] = $values['form_selectors'];
    $this->configuration['toggle_label'] = $values['toggle_label'];
    $this->configuration['toggle_default_state'] = $values['toggle_default_state'];

    $color_fields = [
      'primary_color', 'background_main', 'response_bg', 'text_primary', 'text_secondary',
    ];

    foreach ($color_fields as $field) {
      if (isset($values[$field])) {
        $this->configuration[$field] = $values[$field];
      }
    }

    $new_token = $values['token'];
    $existing_token = $this->state->get("dify_augmented_search.block_{$block_uuid}.token", '');

    $this->state->set("dify_augmented_search.block_{$block_uuid}.base_url", $values['base_url']);
    $this->state->set("dify_augmented_search.block_{$block_uuid}.token", !empty($new_token) ? $new_token : $existing_token);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $block_uuid = $this->configuration['uuid'] ?? uniqid('block_', TRUE);

    $base_url = $this->state->get("dify_augmented_search.block_{$block_uuid}.base_url");
    $token = $this->state->get("dify_augmented_search.block_{$block_uuid}.token");

    if (empty($base_url) || empty($token)) {
      return [
        '#markup' => '<div class="dify-augmented-search-not-configured">Dify Augmented Search: Please configure the block with your Dify API credentials.</div>',
      ];
    }

    $config = $this->getConfiguration();
    $library = $config['markdown_theme'] === 'light' ? 'dify_augmented_search/async_search_light' : 'dify_augmented_search/async_search';

    $default_colors = $this->getDefaultColors();
    $custom_colors = [
      'primary-color' => $config['primary_color'] ?? $default_colors['primary_color'],
      'background-main' => $config['background_main'] ?? $default_colors['background_main'],
      'response-bg' => $config['response_bg'] ?? $default_colors['response_bg'],
      'text-primary' => $config['text_primary'] ?? $default_colors['text_primary'],
      'text-secondary' => $config['text_secondary'] ?? $default_colors['text_secondary'],
      'border-color' => $default_colors['border_color'],
      'border-light' => $default_colors['border_light'],
      'loading-color' => $default_colors['loading_color'],
      'error-bg' => $default_colors['error_bg'],
      'error-text' => $default_colors['error_text'],
    ];

    return [
      '#theme' => 'dify_augmented_search_block',
      '#block_uuid' => $block_uuid,
      '#custom_colors' => $custom_colors,
      '#attached' => [
        'library' => [$library],
        'drupalSettings' => [
          'difyAugmentedSearch' => [
            $block_uuid => [
              'apiUrl' => rtrim($base_url, '/'),
              'apiToken' => $token,
              'markdownUrl' => '/dify-augmented-search/markdown/render',
              'toggleConfig' => [
                'enabled' => $config['enable_toggle'] ?? FALSE,
                'selectors' => array_filter(array_map('trim', explode("\n", $config['form_selectors'] ?? ''))),
                'label' => $config['toggle_label'] ?? 'AI Search',
                'defaultState' => $config['toggle_default_state'] ?? TRUE,
              ],
            ],
          ],
        ],
      ],
      '#cache' => [
        'contexts' => ['url.query_args'],
      ],
    ];
  }

}
