<?php

namespace Drupal\dify_widget_vanilla\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\dify_widget_vanilla\Service\MarkdownService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for markdown rendering.
 */
class MarkdownController extends ControllerBase {

  /**
   * The markdown service.
   *
   * @var \Drupal\dify_widget_vanilla\Service\MarkdownService
   */
  protected $markdownService;

  /**
   * Constructs a new MarkdownController object.
   *
   * @param \Drupal\dify_widget_vanilla\Service\MarkdownService $markdown_service
   *   The markdown service.
   */
  public function __construct(MarkdownService $markdown_service) {
    $this->markdownService = $markdown_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify_widget_vanilla.markdown_service')
    );
  }

  /**
   * Renders markdown content to HTML.
   */
  public function render(Request $request): JsonResponse {
    $data = json_decode($request->getContent(), TRUE);
    $markdown = $data['markdown'] ?? '';
    $is_complete = $data['is_complete'] ?? false;

    if (empty($markdown)) {
      return new JsonResponse(['html' => '']);
    }

    try {
      $html = $this->markdownService->toHtmlStreaming($markdown, $is_complete);
      return new JsonResponse(['html' => $html]);
    }
    catch (\Exception $e) {
      $this->getLogger('dify_widget_vanilla')->error('Markdown rendering error: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse(['html' => htmlspecialchars($markdown, ENT_QUOTES | ENT_HTML5, 'UTF-8')]);
    }
  }

}
