(function (Drupal) {
  'use strict';

  // Global toggle synchronization
  window.difyToggleSync = window.difyToggleSync || {
    allToggles: [],
    syncAll: function(isActive) {
      this.allToggles.forEach(toggle => {
        toggle.checkbox.checked = isActive;
        toggle.slider.style.backgroundColor = isActive ? '#00d4aa' : '#ccc';
        toggle.sliderButton.style.left = isActive ? '22px' : '2px';
        toggle.aiIcon.style.opacity = isActive ? '1' : '0.5';
      });
    }
  };

  class DifyAugmentedSearch {
    constructor(element, settings) {
      this.element = element;
      this.settings = settings;
      this.blockUuid = element.dataset.blockUuid;
      this.isStreaming = false;

      this.loadingElement = element.querySelector('.dify-augmented-search__loading');
      this.responseElement = element.querySelector('.dify-augmented-search__response');
      this.queryElement = element.querySelector('.dify-augmented-search__query');

      this.init();
    }

    init() {
      // Initialize toggle if enabled
      this.initializeToggle();

      // Check if AI search is disabled by user preference
      const savedState = localStorage.getItem('dify-ai-search-enabled');
      const isAiEnabled = savedState !== null ? savedState === 'true' : true;

      if (!isAiEnabled) {
        this.hide();
        return;
      }

      const query = this.detectSearchQuery();
      if (query) {
        this.startSearch(query);
      } else {
        this.hide();
      }
    }

    initializeToggle() {
      const toggleConfig = this.settings.toggleConfig;

      if (!toggleConfig || !toggleConfig.enabled || !toggleConfig.selectors.length) {
        return;
      }

      // Add toggle to each specified form
      toggleConfig.selectors.forEach(selector => {
        const forms = document.querySelectorAll(selector);

        forms.forEach(form => {
          if (!form.querySelector('.dify-ai-toggle')) {
            this.addToggleToForm(form, toggleConfig);
          }
        });
      });
    }

    addToggleToForm(form, config) {
      // Get saved state from localStorage, fallback to config default
      const savedState = localStorage.getItem('dify-ai-search-enabled');
      let isActive = savedState !== null ? savedState === 'true' : config.defaultState;

      // Create toggle container
      const container = document.createElement('div');
      container.className = 'dify-ai-toggle-container';
      container.style.cssText = `
        display: inline-flex;
        align-items: center;
        margin-right: 12px;
        gap: 8px;
      `;

      // Create toggle switch
      const toggleSwitch = document.createElement('label');
      toggleSwitch.className = 'dify-toggle-switch';
      toggleSwitch.style.cssText = `
        position: relative;
        display: inline-block;
        width: 40px;
        height: 20px;
        cursor: pointer;
      `;

      // Hidden checkbox
      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.checked = isActive;
      checkbox.style.cssText = 'opacity: 0; width: 0; height: 0;';

      // Slider
      const slider = document.createElement('span');
      slider.style.cssText = `
        position: absolute;
        cursor: pointer;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: ${isActive ? '#00d4aa' : '#ccc'};
        transition: 0.3s;
        border-radius: 20px;
      `;

      // Slider button
      const sliderButton = document.createElement('span');
      sliderButton.style.cssText = `
        position: absolute;
        content: "";
        height: 16px;
        width: 16px;
        left: ${isActive ? '22px' : '2px'};
        bottom: 2px;
        background-color: white;
        transition: 0.3s;
        border-radius: 50%;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
      `;

      // AI icon
      const aiIcon = document.createElement('span');
      aiIcon.innerHTML = '✨';
      aiIcon.style.cssText = `
        font-size: 14px;
        opacity: ${isActive ? '1' : '0.5'};
        transition: opacity 0.3s;
      `;

      // Store toggle reference for global synchronization
      const toggleData = {
        checkbox: checkbox,
        slider: slider,
        sliderButton: sliderButton,
        aiIcon: aiIcon
      };

      // Add to global toggles array
      window.difyToggleSync.allToggles.push(toggleData);

      // Event listener - only on the toggle switch, not the checkbox
      toggleSwitch.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();

        // Manually toggle the checkbox
        checkbox.checked = !checkbox.checked;
        isActive = checkbox.checked;

        // Save to localStorage for next searches
        localStorage.setItem('dify-ai-search-enabled', isActive.toString());

        // Synchronize ALL toggles on ALL pages
        window.difyToggleSync.syncAll(isActive);

        // Don't affect current search results - only future ones
      });

      // Assemble toggle
      slider.appendChild(sliderButton);
      toggleSwitch.appendChild(checkbox);
      toggleSwitch.appendChild(slider);
      container.appendChild(toggleSwitch);
      container.appendChild(aiIcon);

      // Add BEFORE the submit button
      const submitButton = form.querySelector('input[type="submit"], button[type="submit"]');
      if (submitButton && submitButton.parentNode) {
        submitButton.parentNode.insertBefore(container, submitButton);
      } else {
        form.appendChild(container);
      }

      // No need to affect current blocks - only future searches
    }





    detectSearchQuery() {
      const urlParams = new URLSearchParams(window.location.search);

      // Check if AI search is explicitly disabled
      const aiSearchParam = urlParams.get('ai_search');
      if (aiSearchParam === '0') {
        return null; // Don't perform AI search
      }

      const searchParams = ['search', 'keys', 'query', 'q', 'keyword'];

      for (const param of searchParams) {
        const value = urlParams.get(param);
        if (value && value.trim()) {
          return value.trim();
        }
      }
      return null;
    }

    async startSearch(query) {
      this.show();
      this.showLoading();
      this.setQuery(query);

      try {
        const response = await fetch(this.settings.apiUrl + '/v1/chat-messages', {
          method: 'POST',
          headers: {
            'Authorization': 'Bearer ' + this.settings.apiToken,
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            inputs: {},
            query: query,
            response_mode: 'streaming',
            user: 'drupal-search-user',
          }),
        });

        if (!response.ok) {
          throw new Error('Failed to send message to Dify API');
        }

        this.startStreaming(response);

      } catch (error) {
        console.error('Search error:', error);
        this.showError('Unable to get AI response. Please try again.');
      }
    }

    async startStreaming(response) {
      this.isStreaming = true;
      this.hideLoading();
      this.showResponse();

      try {
        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let buffer = '';
        let fullContent = '';

        while (true) {
          const { done, value } = await reader.read();

          if (done) break;

          buffer += decoder.decode(value, { stream: true });

          const lines = buffer.split('\n');
          buffer = lines.pop();

          for (const line of lines) {
            if (line.startsWith('data: ')) {
              const jsonData = line.slice(6).trim();

              if (jsonData === '') continue;

              try {
                const data = JSON.parse(jsonData);

                if (data.event === 'message' && data.answer) {
                  fullContent += data.answer;
                  this.renderMarkdown(fullContent, false, this.responseElement);
                } else if (data.event === 'message_end') {
                  this.renderMarkdown(fullContent, true, this.responseElement);
                  this.completeStreaming();
                  return;
                } else if (data.event === 'error') {
                  throw new Error(data.message || 'API Error');
                }
              } catch (parseError) {
                console.warn('Failed to parse SSE data:', jsonData);
              }
            }
          }
        }

        this.completeStreaming();

      } catch (error) {
        console.error('Streaming error:', error);
        this.showError('Connection error. Please try again.');
        this.completeStreaming();
      }
    }

    async updateResponse(content, final = false) {
      if (this.responseElement) {
        await this.renderMarkdown(content, final, this.responseElement);

        if (!final) {
          this.responseElement.classList.add('streaming');
        } else {
          this.responseElement.classList.remove('streaming');
        }
      }
    }

    async renderMarkdown(content, isComplete, targetElement) {
      try {
        const response = await fetch(this.settings.markdownUrl || '/dify-augmented-search/markdown/render', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            markdown: content,
            is_complete: isComplete
          })
        });

        if (response.ok) {
          const data = await response.json();
          targetElement.innerHTML = data.html;
          targetElement.classList.add('markdown-body');
        } else {
          targetElement.innerHTML = this.formatMarkdownFallback(content);
          targetElement.classList.add('markdown-body');
        }
      } catch (error) {
        console.warn('Markdown rendering failed, using fallback:', error);
        targetElement.innerHTML = this.formatMarkdownFallback(content);
        targetElement.classList.add('markdown-body');
      }
    }

    formatMarkdownFallback(content) {
      let formatted = content
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\*(.*?)\*/g, '<em>$1</em>')
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/^# (.+)$/gm, '<h1>$1</h1>')
        .replace(/^## (.+)$/gm, '<h2>$1</h2>')
        .replace(/^### (.+)$/gm, '<h3>$1</h3>')
        .replace(/^- (.+)$/gm, '<li>$1</li>')
        .replace(/\n/g, '<br>');

      formatted = formatted.replace(/(<li>.*<\/li>)/gs, '<ul>$1</ul>');

      return formatted;
    }

    setQuery(query) {
      if (this.queryElement) {
        this.queryElement.innerHTML = `<small>${Drupal.t('Based on your search:')} "${query}"</small>`;
      }
    }

    show() {
      this.element.style.display = 'block';
    }

    hide() {
      this.element.style.display = 'none';
    }

    showLoading() {
      if (this.loadingElement) {
        this.loadingElement.style.display = 'block';
      }
    }

    hideLoading() {
      if (this.loadingElement) {
        this.loadingElement.style.display = 'none';
      }
    }

    showResponse() {
      if (this.responseElement) {
        this.responseElement.style.display = 'block';
      }
      if (this.queryElement) {
        this.queryElement.style.display = 'block';
      }
    }

    showError(message) {
      this.hideLoading();
      if (this.responseElement) {
        this.responseElement.innerHTML = `<div class="error">${message}</div>`;
        this.responseElement.style.display = 'block';
      }
    }

    completeStreaming() {
      this.isStreaming = false;
    }
  }

  Drupal.behaviors.difyAugmentedSearch = {
    attach: function (context, settings) {
      const blocks = context.querySelectorAll('.dify-augmented-search:not(.dify-processed)');

      blocks.forEach((block) => {
        const blockUuid = block.dataset.blockUuid;
        const blockSettings = settings.difyAugmentedSearch && settings.difyAugmentedSearch[blockUuid];

        if (blockSettings) {
          block.classList.add('dify-processed');
          new DifyAugmentedSearch(block, blockSettings);
        }
      });
    }
  };

})(Drupal);
