<?php

namespace Drupal\dify_widget_vanilla\Service;

use League\CommonMark\CommonMarkConverter;
use League\CommonMark\Extension\GithubFlavoredMarkdownExtension;

/**
 * Service for rendering markdown content using League CommonMark.
 */
class MarkdownService {

  /**
   * The CommonMark converter.
   *
   * @var \League\CommonMark\CommonMarkConverter
   */
  protected $converter;

  /**
   * Constructs a new MarkdownService object.
   */
  public function __construct() {
    // Configure the converter for version 1.6
    $config = [
      'html_input' => 'escape',
      'allow_unsafe_links' => false,
      'max_nesting_level' => 10,
    ];

    // For version 1.6, we use the simpler constructor
    $this->converter = new CommonMarkConverter($config);
  }

  /**
   * Converts markdown text to HTML.
   *
   * @param string $markdown
   *   The markdown text to convert.
   *
   * @return string
   *   The converted HTML.
   */
  public function toHtml(string $markdown): string {
    if (empty($markdown)) {
      return '';
    }

    try {
      $html = $this->converter->convertToHtml($markdown);
      return $this->processLinks($html);
    }
    catch (\Exception $e) {
      // Fallback to escaped text if markdown parsing fails
      return htmlspecialchars($markdown, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }
  }

  /**
   * Converts markdown text to HTML with streaming-friendly processing.
   *
   * This method handles partial markdown content that might be incomplete
   * during streaming, providing better user experience.
   *
   * @param string $markdown
   *   The markdown text to convert.
   * @param bool $is_complete
   *   Whether this is the complete content or partial streaming content.
   *
   * @return string
   *   The converted HTML.
   */
  public function toHtmlStreaming(string $markdown, bool $is_complete = false): string {
    if (empty($markdown)) {
      return '';
    }

    // For incomplete streaming content, we need to be careful with parsing
    if (!$is_complete) {
      // Check if we're in the middle of a code block or other structure
      $lines = explode("\n", $markdown);
      $last_line = end($lines);

      // If the last line looks incomplete, try to render what we have
      // but be prepared for parsing errors
      try {
        $html = $this->converter->convertToHtml($markdown);
        return $this->processLinks($html);
      }
      catch (\Exception $e) {
        // If parsing fails on incomplete content, fall back to simple formatting
        return $this->simpleMarkdownFormat($markdown);
      }
    }

    return $this->toHtml($markdown);
  }

  /**
   * Simple markdown formatting for fallback scenarios.
   *
   * @param string $text
   *   The text to format.
   *
   * @return string
   *   The formatted HTML.
   */
  protected function simpleMarkdownFormat(string $text): string {
    $formatted = htmlspecialchars($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

    // Basic markdown patterns
    $patterns = [
      '/\*\*(.*?)\*\*/s' => '<strong>$1</strong>',
      '/\*(.*?)\*/s' => '<em>$1</em>',
      '/`(.*?)`/s' => '<code>$1</code>',
      '/^# (.+)$/m' => '<h1>$1</h1>',
      '/^## (.+)$/m' => '<h2>$1</h2>',
      '/^### (.+)$/m' => '<h3>$1</h3>',
      '/^- (.+)$/m' => '<li>$1</li>',
    ];

    foreach ($patterns as $pattern => $replacement) {
      $formatted = preg_replace($pattern, $replacement, $formatted);
    }

    // Convert line breaks
    $formatted = nl2br($formatted);

    // Wrap list items in ul tags
    $formatted = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $formatted);

    return $formatted;
  }

  /**
   * Process HTML to add target="_blank" to absolute links only.
   *
   * @param string $html
   *   The HTML content to process.
   *
   * @return string
   *   The processed HTML with target="_blank" added to absolute links.
   */
  protected function processLinks(string $html): string {
    // Use regex to find all links and process only absolute ones
    $pattern = '/<a\s+([^>]*?)href\s*=\s*["\']([^"\']*)["\']([^>]*?)>/i';

    return preg_replace_callback($pattern, function($matches) {
      $beforeHref = $matches[1];
      $url = $matches[2];
      $afterHref = $matches[3];

      // Check if the URL is absolute (starts with http:// or https://)
      if (preg_match('/^https?:\/\//i', $url)) {
        // Add target="_blank" and rel="noopener noreferrer" for absolute links
        return '<a ' . $beforeHref . 'href="' . $url . '"' . $afterHref . ' target="_blank" rel="noopener noreferrer">';
      } else {
        // Keep relative links as they are (same tab)
        return $matches[0];
      }
    }, $html);
  }

}
