<?php

namespace Drupal\dify\Service;

use Drupal\Core\State\StateInterface;

/**
 * Service for managing multiple Dify configurations.
 */
class DifyConfigurationService {

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;



  /**
   * Constructs a DifyConfigurationService object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(StateInterface $state) {
    $this->state = $state;
  }

  /**
   * Gets a list of all configuration names for a namespace.
   *
   * @param string $namespace
   *   The namespace (module name).
   *
   * @return array
   *   Array of configuration names.
   */
  public function getConfigurationNames(string $namespace = 'widget_js'): array {
    $configs = $this->state->get("dify.{$namespace}.configurations", []);
    return array_keys($configs);
  }

  /**
   * Gets a configuration by name.
   *
   * @param string $namespace
   *   The namespace (module name).
   * @param string $name
   *   The configuration name.
   *
   * @return array|null
   *   The configuration array or NULL if not found.
   */
  public function getConfiguration(string $namespace = 'widget_js', string $name = ''): ?array {
    $configs = $this->state->get("dify.{$namespace}.configurations", []);

    if (empty($name)) {
      return !empty($configs) ? reset($configs) : NULL;
    }

    return $configs[$name] ?? NULL;
  }

  /**
   * Saves a configuration.
   *
   * @param string $namespace
   *   The namespace (module name).
   * @param string $name
   *   The configuration name.
   * @param array $config
   *   The configuration array.
   *
   * @throws \InvalidArgumentException
   *   If the configuration is invalid.
   */
  public function saveConfiguration(string $namespace, string $name, array $config): void {
    $this->validateConfiguration($config);
    $this->validateConfigurationName($name);

    $configs = $this->state->get("dify.{$namespace}.configurations", []);
    $configs[$name] = array_merge($config, [
      'created' => $configs[$name]['created'] ?? time(),
      'updated' => time(),
    ]);

    $this->state->set("dify.{$namespace}.configurations", $configs);
  }

  /**
   * Deletes a configuration.
   *
   * @param string $namespace
   *   The namespace (module name).
   * @param string $name
   *   The configuration name.
   *
   * @return bool
   *   TRUE if the configuration was deleted, FALSE if it didn't exist.
   */
  public function deleteConfiguration(string $namespace, string $name): bool {
    $configs = $this->state->get("dify.{$namespace}.configurations", []);
    if (isset($configs[$name])) {
      unset($configs[$name]);
      $this->state->set("dify.{$namespace}.configurations", $configs);
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Checks if a configuration exists.
   *
   * @param string $namespace
   *   The namespace (module name).
   * @param string $name
   *   The configuration name.
   *
   * @return bool
   *   TRUE if the configuration exists.
   */
  public function configurationExists(string $namespace, string $name): bool {
    $configs = $this->state->get("dify.{$namespace}.configurations", []);
    return isset($configs[$name]);
  }

  /**
   * Gets configuration options for form select elements.
   *
   * @param string $namespace
   *   The namespace (module name).
   *
   * @return array
   *   Array of configuration names keyed by name.
   */
  public function getConfigurationOptions(string $namespace = 'widget_js'): array {
    $names = $this->getConfigurationNames($namespace);
    $options = [];

    foreach ($names as $name) {
      $config = $this->getConfiguration($namespace, $name);
      $label = $name;

      if ($config && !empty($config['base_url'])) {
        $label .= ' (' . $config['base_url'] . ')';
      }

      $options[$name] = $label;
    }

    return $options;
  }

  /**
   * Validates a configuration array.
   *
   * @param array $config
   *   The configuration to validate.
   *
   * @throws \InvalidArgumentException
   *   If the configuration is invalid.
   */
  protected function validateConfiguration(array $config): void {
    // Validate base URL
    if (!empty($config['base_url']) && !filter_var($config['base_url'], FILTER_VALIDATE_URL)) {
      throw new \InvalidArgumentException('Invalid base URL format.');
    }

    // Validate token format
    if (!empty($config['token']) && !preg_match('/^[a-zA-Z0-9_-]+$/', $config['token'])) {
      throw new \InvalidArgumentException('Token must contain only letters, numbers, hyphens, and underscores.');
    }

    // Validate API token format
    if (!empty($config['api_token']) && !preg_match('/^[a-zA-Z0-9_-]+$/', $config['api_token'])) {
      throw new \InvalidArgumentException('API token must contain only letters, numbers, hyphens, and underscores.');
    }
  }

  /**
   * Validates a configuration name.
   *
   * @param string $name
   *   The configuration name to validate.
   *
   * @throws \InvalidArgumentException
   *   If the name is invalid.
   */
  protected function validateConfigurationName(string $name): void {
    if (empty($name)) {
      throw new \InvalidArgumentException('Configuration name cannot be empty.');
    }

    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $name)) {
      throw new \InvalidArgumentException('Configuration name must contain only letters, numbers, hyphens, and underscores.');
    }

    if (strlen($name) > 50) {
      throw new \InvalidArgumentException('Configuration name must be 50 characters or less.');
    }
  }



}
