<?php

namespace Drupal\dify_widget_vanilla\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\dify\Service\DifyConfigurationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Dify Widget JS Configuration form.
 */
class DifyWidgetVanillaConfigForm extends FormBase {

  /**
   * The Dify configuration service.
   *
   * @var \Drupal\dify\Service\DifyConfigurationService
   */
  protected DifyConfigurationService $configurationService;

  /**
   * Constructs a DifyWidgetVanillaConfigForm object.
   *
   * @param \Drupal\dify\Service\DifyConfigurationService $configuration_service
   *   The Dify configuration service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(DifyConfigurationService $configuration_service, RouteMatchInterface $route_match) {
    $this->configurationService = $configuration_service;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify.configuration_service'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dify_widget_vanilla_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $name = $this->routeMatch->getParameter('name');
    $config = [];
    $is_edit = !empty($name);

    if ($is_edit) {
      $config = $this->configurationService->getConfiguration('widget_js', $name) ?? [];
    }

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Configuration Name'),
      '#description' => $this->t('A unique name for this configuration.'),
      '#default_value' => $name ?? '',
      '#required' => TRUE,
      '#disabled' => $is_edit,
      '#maxlength' => 50,
    ];

    $form['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#description' => $this->t('The base URL for your Dify service.'),
      '#default_value' => $config['base_url'] ?? '',
      '#required' => TRUE,
    ];

    $form['api_token'] = [
      '#type' => 'password',
      '#title' => $this->t('API Token'),
      '#description' => $this->t('Your Dify API token for widget conversations.'),
      '#default_value' => $config['api_token'] ?? '',
      '#attributes' => [
        'autocomplete' => 'new-password',
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $is_edit ? $this->t('Update') : $this->t('Save'),
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('dify_widget_vanilla.config_list'),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getValue('name');
    $base_url = $form_state->getValue('base_url');
    $api_token = $form_state->getValue('api_token');

    if (!empty($name) && !preg_match('/^[a-zA-Z0-9_-]+$/', $name)) {
      $form_state->setErrorByName('name', $this->t('Name must contain only letters, numbers, hyphens, and underscores.'));
    }

    if (!empty($base_url) && !filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('base_url', $this->t('Please enter a valid URL.'));
    }

    if (!empty($api_token) && !preg_match('/^[a-zA-Z0-9_-]+$/', $api_token)) {
      $form_state->setErrorByName('api_token', $this->t('API token must contain only letters, numbers, hyphens, and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getValue('name');
    $config = [
      'base_url' => $form_state->getValue('base_url'),
      'api_token' => $form_state->getValue('api_token'),
    ];

    try {
      $this->configurationService->saveConfiguration('widget_js', $name, $config);
      $this->messenger()->addMessage($this->t('Configuration "@name" has been saved.', ['@name' => $name]));
      $form_state->setRedirectUrl(Url::fromRoute('dify_widget_vanilla.config_list'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error saving configuration: @error', ['@error' => $e->getMessage()]));
    }
  }

}
