<?php

namespace Drupal\dify_widget_vanilla\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\dify\Service\DifyConfigurationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Dify Chat Widget' Block.
 *
 * @Block(
 *   id = "dify_widget_vanilla_block",
 *   admin_label = @Translation("Dify Chat Widget"),
 *   category = @Translation("Dify")
 * )
 */
class DifyWidgetVanillaBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The Dify configuration service.
   *
   * @var \Drupal\dify\Service\DifyConfigurationService
   */
  protected $configurationService;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new DifyWidgetBlock instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\dify\Service\DifyConfigurationService $configuration_service
   *   The Dify configuration service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AccountInterface $current_user, DifyConfigurationService $configuration_service, StateInterface $state) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->currentUser = $current_user;
    $this->configurationService = $configuration_service;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('dify.configuration_service'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'widget_title' => $this->t('Assistant Chat'),
      'placeholder_text' => $this->t('Type your message here...'),
      'markdown_theme' => 'dark',
      'dify_configuration' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * Get default color values.
   */
  protected function getDefaultColors() {
    return [
      'primary_color' => '#2563eb',
      'primary_hover' => '#1d4ed8',
      'primary_light' => '#3b82f6',
      'background_main' => '#0f172a',
      'background_secondary' => '#1e293b',
      'background_tertiary' => '#334155',
      'background_input' => '#475569',
      'text_primary' => '#f8fafc',
      'text_secondary' => '#cbd5e1',
      'text_muted' => '#94a3b8',
      'text_placeholder' => '#64748b',
      'border_color' => '#475569',
      'border_light' => '#64748b',
      'user_message_bg' => '#1e40af',
      'bot_message_bg' => '#0f172a',
      'error_bg' => '#fecaca',
      'error_text' => '#dc2626',
      'notification_color' => '#f59e0b',
      'success_color' => '#10b981',
      'warning_color' => '#f59e0b',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $config = $this->getConfiguration();
    $default_colors = $this->getDefaultColors();

    $form['widget_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Widget Title'),
      '#description' => $this->t('Title displayed in the floating chat widget header.'),
      '#default_value' => $config['widget_title'],
      '#maxlength' => 255,
    ];

    $form['placeholder_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder Text'),
      '#description' => $this->t('Placeholder text for the message input field.'),
      '#default_value' => $config['placeholder_text'],
      '#maxlength' => 255,
    ];

    $form['markdown_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Markdown Theme'),
      '#description' => $this->t('Choose the theme for markdown rendering in chat messages.'),
      '#default_value' => $config['markdown_theme'],
      '#options' => [
        'dark' => $this->t('Dark'),
        'light' => $this->t('Light'),
      ],
    ];

    // Dify Configuration Selection.
    $configuration_options = $this->configurationService->getConfigurationOptions('widget_js');
    if (!empty($configuration_options)) {
      $default_config = $config['dify_configuration'] ?? '';
      if (empty($default_config) || !isset($configuration_options[$default_config])) {
        $default_config = array_key_first($configuration_options);
      }

      $form['dify_configuration'] = [
        '#type' => 'select',
        '#title' => $this->t('Dify Configuration'),
        '#description' => $this->t('Select which Dify configuration this widget should use.'),
        '#default_value' => $default_config,
        '#options' => $configuration_options,
        '#required' => TRUE,
      ];
    }
    else {
      $form['dify_configuration_warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' . $this->t('No Dify configurations found. Please <a href="@url">create a configuration</a> first.', [
          '@url' => '/admin/config/search/dify-widget-vanilla/add',
        ]) . '</div>',
      ];
    }

    // === PRIMARY COLORS SECTION ===
    $form['colors_primary_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Primary Colors') . '</h3><hr>',
    ];

    $form['primary_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Color'),
      '#description' => $this->t('Color used for buttons, header and accent elements.'),
      '#default_value' => $this->configuration['primary_color'] ?? $default_colors['primary_color'],
    ];

    $form['primary_hover'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Hover Color'),
      '#description' => $this->t('Color on hover for primary elements.'),
      '#default_value' => $this->configuration['primary_hover'] ?? $default_colors['primary_hover'],
    ];

    $form['primary_light'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Light Color'),
      '#description' => $this->t('Light version of the primary color.'),
      '#default_value' => $this->configuration['primary_light'] ?? $default_colors['primary_light'],
    ];

    // === BACKGROUND COLORS SECTION ===
    $form['colors_background_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Background Colors') . '</h3><hr>',
    ];

    $form['background_main'] = [
      '#type' => 'color',
      '#title' => $this->t('Main Background'),
      '#description' => $this->t('Main background color of the widget.'),
      '#default_value' => $this->configuration['background_main'] ?? $default_colors['background_main'],
    ];

    $form['background_secondary'] = [
      '#type' => 'color',
      '#title' => $this->t('Secondary Background'),
      '#description' => $this->t('Background color of the messages area.'),
      '#default_value' => $this->configuration['background_secondary'] ?? $default_colors['background_secondary'],
    ];

    $form['background_tertiary'] = [
      '#type' => 'color',
      '#title' => $this->t('Tertiary Background'),
      '#description' => $this->t('Background color for suggestions and secondary elements.'),
      '#default_value' => $this->configuration['background_tertiary'] ?? $default_colors['background_tertiary'],
    ];

    $form['background_input'] = [
      '#type' => 'color',
      '#title' => $this->t('Input Background'),
      '#description' => $this->t('Background color for input fields.'),
      '#default_value' => $this->configuration['background_input'] ?? $default_colors['background_input'],
    ];

    // === TEXT COLORS SECTION ===
    $form['colors_text_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Text Colors') . '</h3><hr>',
    ];

    $form['text_primary'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary Text'),
      '#description' => $this->t('Primary text color.'),
      '#default_value' => $this->configuration['text_primary'] ?? $default_colors['text_primary'],
    ];

    $form['text_secondary'] = [
      '#type' => 'color',
      '#title' => $this->t('Secondary Text'),
      '#description' => $this->t('Secondary text color.'),
      '#default_value' => $this->configuration['text_secondary'] ?? $default_colors['text_secondary'],
    ];

    $form['text_muted'] = [
      '#type' => 'color',
      '#title' => $this->t('Muted Text'),
      '#description' => $this->t('Muted text color for disabled elements.'),
      '#default_value' => $this->configuration['text_muted'] ?? $default_colors['text_muted'],
    ];

    $form['text_placeholder'] = [
      '#type' => 'color',
      '#title' => $this->t('Placeholder Text'),
      '#description' => $this->t('Placeholder text color in input fields.'),
      '#default_value' => $this->configuration['text_placeholder'] ?? $default_colors['text_placeholder'],
    ];

    // === BORDER COLORS SECTION ===
    $form['colors_border_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Border Colors') . '</h3><hr>',
    ];

    $form['border_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Border Color'),
      '#description' => $this->t('Main border color.'),
      '#default_value' => $this->configuration['border_color'] ?? $default_colors['border_color'],
    ];

    $form['border_light'] = [
      '#type' => 'color',
      '#title' => $this->t('Light Border'),
      '#description' => $this->t('Light border color.'),
      '#default_value' => $this->configuration['border_light'] ?? $default_colors['border_light'],
    ];

    // === MESSAGE COLORS SECTION ===
    $form['colors_message_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Message Colors') . '</h3><hr>',
    ];

    $form['user_message_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('User Message Background'),
      '#description' => $this->t('Background color for user messages.'),
      '#default_value' => $this->configuration['user_message_bg'] ?? $default_colors['user_message_bg'],
    ];

    $form['bot_message_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Bot Message Background'),
      '#description' => $this->t('Background color for bot messages.'),
      '#default_value' => $this->configuration['bot_message_bg'] ?? $default_colors['bot_message_bg'],
    ];

    $form['error_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Error Background'),
      '#description' => $this->t('Background color for error messages.'),
      '#default_value' => $this->configuration['error_bg'] ?? $default_colors['error_bg'],
    ];

    $form['error_text'] = [
      '#type' => 'color',
      '#title' => $this->t('Error Text'),
      '#description' => $this->t('Text color for error messages.'),
      '#default_value' => $this->configuration['error_text'] ?? $default_colors['error_text'],
    ];

    // === SPECIAL STATES SECTION ===
    $form['colors_states_section'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Special States') . '</h3><hr>',
    ];

    $form['notification_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Notification Color'),
      '#description' => $this->t('Color for notification badges.'),
      '#default_value' => $this->configuration['notification_color'] ?? $default_colors['notification_color'],
    ];

    $form['success_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Success Color'),
      '#description' => $this->t('Color for success states.'),
      '#default_value' => $this->configuration['success_color'] ?? $default_colors['success_color'],
    ];

    $form['warning_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Warning Color'),
      '#description' => $this->t('Color for warning states.'),
      '#default_value' => $this->configuration['warning_color'] ?? $default_colors['warning_color'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();

    $this->configuration['widget_title'] = $values['widget_title'];
    $this->configuration['placeholder_text'] = $values['placeholder_text'];
    $this->configuration['markdown_theme'] = $values['markdown_theme'];
    $this->configuration['dify_configuration'] = $values['dify_configuration'] ?? '';

    // Save all color values.
    $color_fields = [
      'primary_color', 'primary_hover', 'primary_light',
      'background_main', 'background_secondary', 'background_tertiary', 'background_input',
      'text_primary', 'text_secondary', 'text_muted', 'text_placeholder',
      'border_color', 'border_light',
      'user_message_bg', 'bot_message_bg', 'error_bg', 'error_text',
      'notification_color', 'success_color', 'warning_color',
    ];

    foreach ($color_fields as $field) {
      if (isset($values[$field])) {
        $this->configuration[$field] = $values[$field];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Only show to authenticated users.
    if (!$this->currentUser->isAuthenticated()) {
      return [];
    }

    // Get Dify configuration from selected configuration.
    $config = $this->getConfiguration();
    $configuration_name = $config['dify_configuration'] ?? '';
    $dify_config = $this->configurationService->getConfiguration('widget_js', $configuration_name);

    if (empty($dify_config) || empty($dify_config['api_token']) || empty($dify_config['base_url'])) {
      return [];
    }

    $api_token = $dify_config['api_token'];
    $base_url = $dify_config['base_url'];
    $widget_id = 'dify-widget-' . uniqid();

    // Choose the appropriate library based on markdown theme.
    $library = $config['markdown_theme'] === 'light' ? 'dify_widget_vanilla/dify-widget-light' : 'dify_widget_vanilla/dify-widget';

    // Prepare custom colors.
    $default_colors = $this->getDefaultColors();
    $custom_colors = [
      'primary-color' => $this->configuration['primary_color'] ?? $default_colors['primary_color'],
      'primary-hover' => $this->configuration['primary_hover'] ?? $default_colors['primary_hover'],
      'primary-light' => $this->configuration['primary_light'] ?? $default_colors['primary_light'],
      'background-main' => $this->configuration['background_main'] ?? $default_colors['background_main'],
      'background-secondary' => $this->configuration['background_secondary'] ?? $default_colors['background_secondary'],
      'background-tertiary' => $this->configuration['background_tertiary'] ?? $default_colors['background_tertiary'],
      'background-input' => $this->configuration['background_input'] ?? $default_colors['background_input'],
      'text-primary' => $this->configuration['text_primary'] ?? $default_colors['text_primary'],
      'text-secondary' => $this->configuration['text_secondary'] ?? $default_colors['text_secondary'],
      'text-muted' => $this->configuration['text_muted'] ?? $default_colors['text_muted'],
      'text-placeholder' => $this->configuration['text_placeholder'] ?? $default_colors['text_placeholder'],
      'border-color' => $this->configuration['border_color'] ?? $default_colors['border_color'],
      'border-light' => $this->configuration['border_light'] ?? $default_colors['border_light'],
      'user-message-bg' => $this->configuration['user_message_bg'] ?? $default_colors['user_message_bg'],
      'bot-message-bg' => $this->configuration['bot_message_bg'] ?? $default_colors['bot_message_bg'],
      'error-bg' => $this->configuration['error_bg'] ?? $default_colors['error_bg'],
      'error-text' => $this->configuration['error_text'] ?? $default_colors['error_text'],
      'notification-color' => $this->configuration['notification_color'] ?? $default_colors['notification_color'],
      'success-color' => $this->configuration['success_color'] ?? $default_colors['success_color'],
      'warning-color' => $this->configuration['warning_color'] ?? $default_colors['warning_color'],
    ];

    // Generate CSS hash and store colors for CSS controller.
    $css_hash = md5(serialize($custom_colors));
    $this->state->set("dify_widget_colors_{$css_hash}", $custom_colors);

    // Register this hash for dynamic library creation.
    $css_hashes = $this->state->get('dify_widget_css_hashes', []);
    $css_hashes[] = $css_hash;
    $this->state->set('dify_widget_css_hashes', array_unique($css_hashes));

    $build = [
      '#theme' => 'dify_widget_vanilla_widget',
      '#widget_id' => $widget_id,
      '#title' => $config['widget_title'],
      '#placeholder' => $config['placeholder_text'],
      '#attached' => [
        'library' => [
          $library,
          "dify_widget_vanilla/custom-css-{$css_hash}",
        ],
        'drupalSettings' => [
          'difyWidgetVanilla' => [
            $widget_id => [
              'apiUrl' => '/dify-widget-vanilla/proxy/' . $configuration_name,
              'currentUser' => $this->currentUser->getAccountName(),
              'markdownUrl' => '/dify-widget-vanilla/markdown/render',
              'customColors' => $custom_colors,
              'translations' => [
                'welcomeMessage' => $this->t('Hello! How can I help you today?'),
                'thumbsUp' => $this->t('Good response'),
                'thumbsDown' => $this->t('Poor response'),
              ],
            ],
          ],
        ],
      ],
      '#cache' => [
        'contexts' => ['user'],
        'tags' => ['config:dify.settings'],
      ],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function access(AccountInterface $account, $return_as_object = FALSE) {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
