<?php

namespace Drupal\dify_widget_vanilla\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\dify\Service\DifyConfigurationService;
use GuzzleHttp\Client;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Provides proxy endpoints for Dify API calls.
 */
class ProxyController extends ControllerBase {

  /**
   * The Dify configuration service.
   *
   * @var \Drupal\dify\Service\DifyConfigurationService
   */
  protected DifyConfigurationService $configurationService;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $httpClient;

  /**
   * Constructs a ProxyController object.
   *
   * @param \Drupal\dify\Service\DifyConfigurationService $configuration_service
   *   The Dify configuration service.
   * @param \GuzzleHttp\Client $http_client
   *   The HTTP client.
   */
  public function __construct(DifyConfigurationService $configuration_service, Client $http_client) {
    $this->configurationService = $configuration_service;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify.configuration_service'),
      $container->get('http_client')
    );
  }

  /**
   * Gets configuration for a given config name.
   *
   * @param string $config_name
   *   The configuration name.
   *
   * @return array|null
   *   The configuration array or NULL if invalid.
   */
  protected function getConfig(string $config_name): ?array {
    $config = $this->configurationService->getConfiguration('widget_js', $config_name);
    return (!empty($config['api_token']) && !empty($config['base_url'])) ? $config : NULL;
  }

  /**
   * Proxies chat messages to Dify API.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $config_name
   *   The configuration name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\StreamedResponse
   *   The response.
   */
  public function chatMessages(Request $request, string $config_name) {
    $config = $this->getConfig($config_name);
    if (!$config) {
      return new JsonResponse(['error' => 'Configuration not found'], 404);
    }

    $target = rtrim($config['base_url'], '/') . '/v1/chat-messages';

    try {
      $options = [
        'headers' => [
          'Authorization' => 'Bearer ' . $config['api_token'],
          'Content-Type' => $request->headers->get('Content-Type', 'application/json'),
        ],
        'body' => $request->getContent(),
        'stream' => TRUE,
        'timeout' => 300,
      ];

      // Release session lock during streaming.
      if (PHP_SESSION_ACTIVE === session_status()) {
        @session_write_close();
      }

      $response = $this->httpClient->request('POST', $target, $options);

      return new StreamedResponse(function () use ($response) {
        $body = $response->getBody();

        // Reduce buffering for smoother SSE.
        if (function_exists('apache_setenv')) {
          @apache_setenv('no-gzip', '1');
          @apache_setenv('do-not-vary', '1');
        }
        @ini_set('zlib.output_compression', '0');
        @ini_set('output_buffering', '0');
        @set_time_limit(0);
        while (ob_get_level() > 0) {
          @ob_end_flush();
        }
        @ob_implicit_flush(TRUE);

        while (!$body->eof()) {
          $chunk = $body->read(1024);
          if ($chunk === '') {
            usleep(10000);
            continue;
          }
          echo $chunk;
          flush();
        }
      }, $response->getStatusCode(), [
        'Content-Type' => $response->getHeaderLine('Content-Type') ?: 'text/event-stream',
        'Cache-Control' => 'no-cache, no-store, must-revalidate',
        'Pragma' => 'no-cache',
        'Connection' => 'keep-alive',
        'X-Accel-Buffering' => 'no',
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => 'Proxy request failed'], 502);
    }
  }

  /**
   * Proxies messages request to Dify API.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $config_name
   *   The configuration name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function messages(Request $request, string $config_name) {
    $config = $this->getConfig($config_name);
    if (!$config) {
      return new JsonResponse(['error' => 'Configuration not found'], 404);
    }

    $target = rtrim($config['base_url'], '/') . '/v1/messages';

    try {
      $options = [
        'headers' => [
          'Authorization' => 'Bearer ' . $config['api_token'],
        ],
        'query' => $request->query->all(),
        'timeout' => 60,
      ];

      $response = $this->httpClient->request('GET', $target, $options);
      $data = json_decode($response->getBody()->getContents(), TRUE);

      return new JsonResponse($data, $response->getStatusCode(), [
        'Cache-Control' => 'no-store',
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => 'Proxy request failed'], 502);
    }
  }

  /**
   * Proxies suggested messages request to Dify API.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $config_name
   *   The configuration name.
   * @param string $message_id
   *   The message ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function suggested(Request $request, string $config_name, string $message_id) {
    $config = $this->getConfig($config_name);
    if (!$config) {
      return new JsonResponse(['error' => 'Configuration not found'], 404);
    }

    $target = rtrim($config['base_url'], '/') . '/v1/messages/' . $message_id . '/suggested';

    try {
      $options = [
        'headers' => [
          'Authorization' => 'Bearer ' . $config['api_token'],
          'Content-Type' => 'application/json',
        ],
        'query' => $request->query->all(),
        'timeout' => 60,
      ];

      $response = $this->httpClient->request('GET', $target, $options);
      $data = json_decode($response->getBody()->getContents(), TRUE);

      return new JsonResponse($data, $response->getStatusCode(), [
        'Cache-Control' => 'no-store',
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => 'Proxy request failed'], 502);
    }
  }

  /**
   * Proxies feedback submission to Dify API.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $config_name
   *   The configuration name.
   * @param string $message_id
   *   The message ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function feedbacks(Request $request, string $config_name, string $message_id) {
    $config = $this->getConfig($config_name);
    if (!$config) {
      return new JsonResponse(['error' => 'Configuration not found'], 404);
    }

    $target = rtrim($config['base_url'], '/') . '/v1/messages/' . $message_id . '/feedbacks';

    try {
      $options = [
        'headers' => [
          'Authorization' => 'Bearer ' . $config['api_token'],
          'Content-Type' => $request->headers->get('Content-Type', 'application/json'),
        ],
        'body' => $request->getContent(),
        'timeout' => 60,
      ];

      $response = $this->httpClient->request('POST', $target, $options);
      $data = json_decode($response->getBody()->getContents(), TRUE);

      return new JsonResponse($data, $response->getStatusCode(), [
        'Cache-Control' => 'no-store',
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => 'Proxy request failed'], 502);
    }
  }

}
