/**
 * @file
 * Dify Widget Vanilla - Floating chatbot widget with Dify API integration.
 */

(function (Drupal) {
  "use strict";

  /**
   * Dify Widget class.
   */
  class DifyWidget {
    constructor(element, settings) {
      this.element = element;
      this.settings = settings;
      this.widgetId = element.dataset.widgetId;
      this.currentUser = settings.currentUser || "user";
      this.storageKey = `dify_conversation_${this.currentUser}`;

      this.openButton = document.getElementById(`${this.widgetId}-open-btn`);
      this.closeButton = document.getElementById(`${this.widgetId}-close-btn`);

      this.conversationId = this.loadConversation();
      this.isStreaming = false;
      this.isExpanded = false;
      this.unreadCount = 0;
      this.feedbackCache = new Map();

      this.init();
    }

    init() {
      this.messagesContainer = this.element.querySelector(
        ".dify-widget-messages",
      );
      this.form = this.element.querySelector(".dify-input-form");
      this.input = this.element.querySelector(".message-input");
      this.sendButton = this.element.querySelector(".send-button");

      this.bindEvents();
      this.setupButtons();
      this.initResize();

      this.historyLoaded = false;
    }
    async getCsrfToken() {
      if (this._csrfToken) return this._csrfToken;
      try {
        const res = await fetch('/session/token');
        if (res.ok) {
          this._csrfToken = await res.text();
          return this._csrfToken;
        }
      } catch {
        // Ignore CSRF token fetch errors
      }
      return '';
    }


    loadConversation() {
      try {
        const stored = localStorage.getItem(this.storageKey);
        return stored ? JSON.parse(stored).conversationId : "";
      } catch {
        return "";
      }
    }

    saveConversation() {
      try {
        localStorage.setItem(
          this.storageKey,
          JSON.stringify({
            conversationId: this.conversationId,
            lastActivity: Date.now(),
          }),
        );
      } catch {
        console.warn("Could not save conversation to localStorage");
      }
    }

    clearConversation() {
      this.conversationId = "";
      localStorage.removeItem(this.storageKey);
    }

    setupButtons() {
      if (this.openButton) {
        this.openButton.addEventListener("click", () => this.openWidget());
      }

      if (this.closeButton) {
        this.closeButton.addEventListener("click", () => this.closeWidget());
      }
    }

    openWidget() {
      this.isExpanded = true;
      this.resetUnreadCount();

      this.element.style.display = "flex";
      this.element.offsetHeight;

      this.element.classList.add("opening");
      this.element.classList.remove("closing");

      if (this.openButton) this.openButton.style.display = "none";
      if (this.closeButton) this.closeButton.style.display = "flex";

      setTimeout(() => {
        const input = this.element.querySelector(".message-input");
        if (input) input.focus();

        if (this.conversationId && !this.historyLoaded) {
          this.loadConversationHistory();
          this.historyLoaded = true;
        } else {
          // If no history to load, still load feedbacks for any existing
          // messages
          this.loadFeedbacks();
        }
      }, 300);
    }

    closeWidget() {
      this.isExpanded = false;

      this.element.classList.add("closing");
      this.element.classList.remove("opening");

      setTimeout(() => {
        this.element.style.display = "none";
        if (this.openButton) this.openButton.style.display = "flex";
        if (this.closeButton) this.closeButton.style.display = "none";
      }, 250);
    }

    updateUnreadCount() {
      if (!this.isExpanded) {
        this.unreadCount += 1;
        const badge = this.element.querySelector(".notification-badge");
        if (badge) {
          badge.textContent = this.unreadCount;
          badge.style.display = this.unreadCount > 0 ? "flex" : "none";
        }
      }
    }

    resetUnreadCount() {
      this.unreadCount = 0;
      const badge = this.element.querySelector(".notification-badge");
      if (badge) {
        badge.style.display = "none";
      }
    }

    bindEvents() {
      if (this.form) {
        this.form.addEventListener("submit", (e) => {
          e.preventDefault();
          this.sendMessage();
        });
      }

      if (this.input) {
        this.input.addEventListener("input", () => {
          this.autoResizeTextarea();
          this.toggleSendButton();
        });

        this.input.addEventListener("keydown", (e) => {
          if (e.key === "Enter" && !e.shiftKey) {
            e.preventDefault();
            if (!this.isStreaming && this.input.value.trim()) {
              this.sendMessage();
            }
          }
        });
      }

      const newConversationBtn = this.element.querySelector(
        ".new-conversation-button",
      );
      if (newConversationBtn) {
        newConversationBtn.addEventListener("click", () =>
          this.startNewConversation(),
        );
      }

      const minimizeBtn = this.element.querySelector(".minimize-button");
      if (minimizeBtn) {
        minimizeBtn.addEventListener("click", () => this.closeWidget());
      }

      // Feedback button events
      this.messagesContainer.addEventListener("click", (e) => {
        if (e.target.classList.contains("feedback-btn")) {
          const messageElement = e.target.closest(".message");
          const rating = e.target.dataset.rating;
          this.sendFeedback(messageElement, rating);
        }
      });
    }

    startNewConversation() {
      this.clearConversation();

      if (this.messagesContainer) {
        const welcomeMessage =
          this.settings.translations?.welcomeMessage ||
          "Hello! How can I help you today?";
        this.messagesContainer.innerHTML = `
          <div class="dify-welcome-message">
            <div class="message bot-message">
              <div class="message-content markdown-body">
                ${welcomeMessage}
              </div>
            </div>
          </div>
        `;
      }

      this.isStreaming = false;
      this.resetUnreadCount();

      if (this.input) {
        this.input.focus();
      }
    }

    async loadConversationHistory() {
      if (!this.conversationId) return;

      try {
        const response = await fetch(
          `${this.settings.apiUrl}/messages?conversation_id=${this.conversationId}&user=${this.currentUser}&limit=10`,
          {
            headers: {
              "Content-Type": "application/json",
            },
          },
        );

        if (response.ok) {
          const data = await response.json();
          await this.renderHistoryMessages(data.data.reverse());
        }
      } catch (error) {
        console.warn("Could not load conversation history:", error);
      }
    }

    async renderHistoryMessages(messages) {
      if (!this.messagesContainer || !messages.length) return;

      this.messagesContainer.innerHTML = "";

      for (const msg of messages) {
        if (msg.query) {
          this.addMessage(msg.query, "user");
        }
        if (msg.answer) {
          const messageId = this.addMessage("", "bot");
          const messageElement = document.getElementById(messageId);
          const contentElement =
            messageElement.querySelector(".message-content");

          await this.renderMarkdown(msg.answer, true, contentElement);

          if (msg.id) {
            messageElement.dataset.messageId = msg.id;
            // Don't apply feedback state here, it will be done after loading
            // feedbacks
          }
        }
      }

      // Load feedbacks after all messages are rendered
      await this.loadFeedbacks();

      setTimeout(() => {
        this.scrollToBottom();
      }, 200);
    }

    autoResizeTextarea() {
      if (this.input) {
        this.input.style.height = "auto";
        const newHeight = Math.min(this.input.scrollHeight, 120);
        this.input.style.height = newHeight + "px";
      }
    }

    toggleSendButton() {
      if (this.input && this.sendButton) {
        const hasText = this.input.value.trim().length > 0;
        this.sendButton.disabled = !hasText || this.isStreaming;
      }
    }

    async sendMessage(messageText = null) {
      const message =
        messageText || (this.input ? this.input.value.trim() : "");
      if (!message || this.isStreaming) return;

      const userMessageId = this.addMessage(message, "user");
      this.lastUserMessageId = userMessageId;

      this.hideSuggestions();

      if (!messageText && this.input) {
        this.input.value = "";
        this.autoResizeTextarea();
      }
      this.setStreaming(true);

      try {
        const response = await fetch(
          this.settings.apiUrl + "/chat-messages",
          {
            method: "POST",
            headers: {
              "Content-Type": "application/json",
              "X-CSRF-Token": await this.getCsrfToken(),
            },
            body: JSON.stringify({
              inputs: {},
              query: message,
              response_mode: "streaming",
              user: this.currentUser,
              conversation_id: this.conversationId || undefined,
            }),
          },
        );

        if (!response.ok) {
          throw new Error("Failed to send message to Dify API");
        }

        this.startStreaming(response);
      } catch (error) {
        console.error("Send message error:", error);
        this.addMessage(
          "Sorry, an error occurred. Please try again.",
          "bot",
          true,
        );
        this.setStreaming(false);
      }
    }

    async startStreaming(response) {
      const botMessageId = this.addMessage("", "bot", false, true);
      const botMessageElement = document.getElementById(botMessageId);
      const contentElement =
        botMessageElement.querySelector(".message-content");

      if (this.lastUserMessageId) {
        const userMessageElement = document.getElementById(
          this.lastUserMessageId,
        );
        if (userMessageElement) {
          setTimeout(() => {
            this.setupDynamicSpace(userMessageElement, botMessageElement);
          }, 50);
        }
      }

      try {
        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let buffer = "";
        let fullContent = "";

        while (true) {
          const { done, value } = await reader.read();

          if (done) break;

          buffer += decoder.decode(value, { stream: true });

          const lines = buffer.split("\n");
          buffer = lines.pop();

          for (const line of lines) {
            if (line.startsWith("data: ")) {
              const jsonData = line.slice(6).trim();

              if (jsonData === "") continue;

              try {
                const data = JSON.parse(jsonData);

                if (data.event === "message" && data.answer) {
                  fullContent += data.answer;
                  this.renderMarkdown(fullContent, false, contentElement);
                } else if (data.event === "message_end") {
                  this.renderMarkdown(fullContent, true, contentElement);
                  this.conversationId =
                    data.conversation_id || this.conversationId;
                  this.saveConversation();
                  this.removeStreamingIndicators(botMessageElement);

                  if (data.message_id) {
                    botMessageElement.dataset.messageId = data.message_id;
                    this.applyFeedbackState(botMessageElement, data.message_id);
                    this.fetchSuggestions(data.message_id);
                  }

                  this.completeStreaming();
                  return;
                } else if (data.event === "error") {
                  throw new Error(data.message || "API Error");
                }
              } catch {
                console.warn("Failed to parse SSE data:", jsonData);
              }
            }
          }
        }

        this.removeStreamingIndicators(botMessageElement);
        this.completeStreaming();
      } catch (error) {
        console.error("Streaming error:", error);
        contentElement.innerHTML = "Connection error. Please try again.";
        this.removeStreamingIndicators(botMessageElement);
        botMessageElement.classList.add("error");
        this.completeStreaming();
      }
    }

    completeStreaming() {
      this.setStreaming(false);
      this.removeDynamicSpace();
    }

    async renderMarkdown(content, isComplete, targetElement) {
      try {
        const response = await fetch(
          this.settings.markdownUrl || "/dify-widget-vanilla/markdown/render",
          {
            method: "POST",
            headers: {
              "Content-Type": "application/json",
            },
            body: JSON.stringify({
              markdown: content,
              is_complete: isComplete,
            }),
          },
        );

        if (response.ok) {
          const data = await response.json();
          targetElement.innerHTML = data.html;
          targetElement.classList.add("markdown-body");
        } else {
          targetElement.innerHTML = this.formatMarkdownFallback(content);
          targetElement.classList.add("markdown-body");
        }
      } catch (error) {
        console.warn("Markdown rendering failed, using fallback:", error);
        targetElement.innerHTML = this.formatMarkdownFallback(content);
        targetElement.classList.add("markdown-body");
      }
    }

    formatMarkdownFallback(content) {
      let formatted = content
        .replace(/\*\*(.*?)\*\*/g, "<strong>$1</strong>")
        .replace(/\*(.*?)\*/g, "<em>$1</em>")
        .replace(/`(.*?)`/g, "<code>$1</code>")
        .replace(/^# (.+)$/gm, "<h1>$1</h1>")
        .replace(/^## (.+)$/gm, "<h2>$1</h2>")
        .replace(/^### (.+)$/gm, "<h3>$1</h3>")
        .replace(/^- (.+)$/gm, "<li>$1</li>")
        .replace(/\n/g, "<br>");

      formatted = formatted.replace(/(<li>.*<\/li>)/gs, "<ul>$1</ul>");

      return formatted;
    }

    removeStreamingIndicators(messageElement) {
      if (messageElement) {
        messageElement.classList.remove("streaming");

        const typingIndicator =
          messageElement.querySelector(".typing-indicator");
        if (typingIndicator) {
          typingIndicator.remove();
        }
      }
    }

    setStreaming(streaming) {
      this.isStreaming = streaming;
      this.toggleSendButton();

      if (this.sendButton) {
        const sendIcon = this.sendButton.querySelector(".send-icon");
        const loadingSpinner =
          this.sendButton.querySelector(".loading-spinner");

        if (sendIcon && loadingSpinner) {
          if (streaming) {
            sendIcon.style.display = "none";
            loadingSpinner.style.display = "inline";
          } else {
            sendIcon.style.display = "inline";
            loadingSpinner.style.display = "none";
          }
        }
      }
    }

    async fetchSuggestions(messageId) {
      try {
        const response = await fetch(
          `${this.settings.apiUrl}/messages/${messageId}/suggested?user=${encodeURIComponent(this.currentUser)}`,
          {
            method: "GET",
            headers: {
              "Content-Type": "application/json",
            },
          },
        );

        if (response.ok) {
          const data = await response.json();
          if (
            data.result === "success" &&
            data.data &&
            Array.isArray(data.data) &&
            data.data.length > 0
          ) {
            this.displaySuggestions(data.data);
          } else {
            this.hideSuggestions();
          }
        } else {
          this.hideSuggestions();
        }
      } catch (error) {
        console.warn("Failed to fetch suggestions:", error);
        this.hideSuggestions();
      }
    }

    displaySuggestions(suggestions) {
      if (!this.messagesContainer || !suggestions.length) return;

      this.hideSuggestions();

      const suggestionsMessageId = this.addSuggestionsMessage(suggestions);
      this.currentSuggestionsId = suggestionsMessageId;
    }

    addSuggestionsMessage(suggestions) {
      const messageId = `suggestions-${Date.now()}-${Math.random().toString(36).substring(2, 11)}`;
      const messageElement = document.createElement("div");
      messageElement.id = messageId;
      messageElement.className = "message suggestions-message";

      const maxSuggestions = 5;
      const limitedSuggestions = suggestions.slice(0, maxSuggestions);

      const suggestionsHTML = limitedSuggestions
        .map((suggestion) => {
          const maxLength = 50;
          const displayText =
            suggestion.length > maxLength
              ? suggestion.substring(0, maxLength) + "..."
              : suggestion;

          return `<button type="button" class="suggestion-button" data-suggestion="${suggestion.replace(/"/g, "&quot;")}" title="${suggestion.replace(/"/g, "&quot;")}">${displayText}</button>`;
        })
        .join("");

      messageElement.innerHTML = `
        <div class="suggestions-content">
          ${suggestionsHTML}
        </div>
      `;

      const buttons = messageElement.querySelectorAll(".suggestion-button");
      buttons.forEach((button) => {
        button.addEventListener("click", () => {
          const suggestion = button.getAttribute("data-suggestion");
          this.selectSuggestion(suggestion);
        });
      });

      this.messagesContainer.appendChild(messageElement);
      return messageId;
    }

    hideSuggestions() {
      if (this.currentSuggestionsId) {
        const suggestionsElement = document.getElementById(
          this.currentSuggestionsId,
        );
        if (suggestionsElement) {
          suggestionsElement.remove();
        }
        this.currentSuggestionsId = null;
      }
    }

    selectSuggestion(suggestion) {
      this.hideSuggestions();
      this.sendMessage(suggestion);
    }

    addMessage(content, type, isError = false, isStreaming = false) {
      const messageId = `msg-${Date.now()}-${Math.random().toString(36).substring(2, 11)}`;
      const messageElement = document.createElement("div");
      messageElement.id = messageId;
      messageElement.className = `message ${type}-message${isError ? " error" : ""}${isStreaming ? " streaming" : ""}`;

      const feedbackButtons =
        type === "bot" && !isError
          ? `
        <div class="message-feedback">
          <button class="feedback-btn thumb-up" data-rating="like" title="${this.settings.translations?.thumbsUp || "Good response"}">👍</button>
          <button class="feedback-btn thumb-down" data-rating="dislike" title="${this.settings.translations?.thumbsDown || "Poor response"}">👎</button>
        </div>
      `
          : "";

      messageElement.innerHTML = `
        <div class="message-content">${content}</div>
        ${isStreaming ? '<div class="typing-indicator"><span></span><span></span><span></span></div>' : ""}
        ${feedbackButtons}
      `;

      this.messagesContainer.appendChild(messageElement);

      if (type === "bot" && !this.isExpanded && !isStreaming) {
        this.updateUnreadCount();
      }

      return messageId;
    }

    async sendFeedback(messageElement, rating) {
      const messageId = messageElement.dataset.messageId;
      if (!messageId) {
        console.warn("No message ID found for feedback");
        return;
      }

      const feedbackButtons = messageElement.querySelectorAll(".feedback-btn");
      if (feedbackButtons.length === 0) {
        console.warn("No feedback buttons found");
        return;
      }

      const currentRating = this.feedbackCache.get(messageId);

      // If clicking same rating, remove feedback (set to null)
      const newRating = currentRating === rating ? null : rating;

      // Disable buttons during request
      feedbackButtons.forEach((btn) => (btn.disabled = true));

      try {
        const response = await fetch(
          `${this.settings.apiUrl}/messages/${messageId}/feedbacks`,
          {
            method: "POST",
            headers: {
              "Content-Type": "application/json",
              "X-CSRF-Token": await this.getCsrfToken(),
            },
            body: JSON.stringify({
              rating: newRating,
              user: this.currentUser,
            }),
          },
        );

        if (response.ok) {
          // Update cache
          if (newRating === null) {
            this.feedbackCache.delete(messageId);
            console.log(`Feedback removed for message ${messageId}`);
          } else {
            this.feedbackCache.set(messageId, newRating);
            console.log(
              `Feedback set to ${newRating} for message ${messageId}`,
            );
          }

          // Apply the new state (including clearing when newRating is null)
          this.applyFeedbackState(messageElement, messageId);
        } else {
          console.error(
            "Feedback request failed:",
            response.status,
            response.statusText,
          );
        }
      } catch (error) {
        console.error("Feedback error:", error);
      } finally {
        feedbackButtons.forEach((btn) => (btn.disabled = false));
      }
    }

    async loadFeedbacks() {
      if (!this.conversationId) {
        console.log("No conversation ID, skipping feedback loading");
        return;
      }

      // Skipped: not proxied
      return;
    }

    applyFeedbackState(messageElement, messageId) {
      const rating = this.feedbackCache.get(messageId);
      const feedbackButtons = messageElement.querySelectorAll(".feedback-btn");

      // Always clear all active states first
      feedbackButtons.forEach((btn) => {
        btn.classList.remove("active");
      });

      // Apply active state only if there's a valid rating
      if (rating) {
        feedbackButtons.forEach((btn) => {
          if (btn.dataset.rating === rating) {
            btn.classList.add("active");
          }
        });
      }
    }

    clearFeedbackState(messageElement) {
      const feedbackButtons = messageElement.querySelectorAll(".feedback-btn");
      feedbackButtons.forEach((btn) => {
        btn.classList.remove("active");
      });
    }

    scrollToBottom() {
      if (this.messagesContainer) {
        this.messagesContainer.scrollTo({
          top: this.messagesContainer.scrollHeight,
          behavior: "smooth",
        });
      }
    }

    smoothScrollToBottom(duration = 300) {
      if (!this.messagesContainer) return;

      const start = this.messagesContainer.scrollTop;
      const target =
        this.messagesContainer.scrollHeight -
        this.messagesContainer.clientHeight;
      const distance = target - start;

      if (distance === 0) return;

      const startTime = performance.now();

      const animateScroll = (currentTime) => {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);

        const easeOut = 1 - Math.pow(1 - progress, 3);

        this.messagesContainer.scrollTop = start + distance * easeOut;

        if (progress < 1) {
          requestAnimationFrame(animateScroll);
        }
      };

      requestAnimationFrame(animateScroll);
    }

    scrollToElement(element, duration = 300) {
      if (!this.messagesContainer || !element) return;

      const elementTop = element.offsetTop;
      const targetScrollTop = elementTop;

      const start = this.messagesContainer.scrollTop;
      const distance = targetScrollTop - start;

      const startTime = performance.now();

      const animateScroll = (currentTime) => {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);

        const easeOut = 1 - Math.pow(1 - progress, 3);

        this.messagesContainer.scrollTop = start + distance * easeOut;

        if (progress < 1) {
          requestAnimationFrame(animateScroll);
        }
      };

      requestAnimationFrame(animateScroll);
    }

    setupDynamicSpace(userMessageElement, botMessageElement) {
      if (!this.messagesContainer || !userMessageElement || !botMessageElement)
        return;

      const containerHeight = this.messagesContainer.clientHeight;
      const containerPadding = 32;
      const userMessageHeight = userMessageElement.offsetHeight;
      const messageMargin = 16;

      const availableHeight = containerHeight - containerPadding;

      const neededMinHeight = Math.max(
        availableHeight - userMessageHeight - messageMargin,
        150,
      );

      botMessageElement.style.minHeight = neededMinHeight + "px";
      botMessageElement.style.transition = "min-height 0.3s ease";

      setTimeout(() => {
        this.scrollToElement(userMessageElement, 300);
      }, 150);
    }

    removeDynamicSpace() {
      const streamingMessages = this.messagesContainer.querySelectorAll(
        '.message.bot-message[style*="min-height"]',
      );
      streamingMessages.forEach((msg) => {
        msg.style.minHeight = "";
        msg.style.transition = "";
      });
    }

    initResize() {
      this.resizeHandle = this.element.querySelector(".resize-handle");
      this.isResizing = false;
      this.resizeStartX = 0;
      this.resizeStartY = 0;
      this.startWidth = 0;
      this.startHeight = 0;
      this.startRight = 0;
      this.startBottom = 0;

      this.loadDimensions();

      if (this.resizeHandle) {
        this.resizeHandle.addEventListener(
          "mousedown",
          this.startResize.bind(this),
        );
      }

      document.addEventListener("mousemove", this.doResize.bind(this));
      document.addEventListener("mouseup", this.stopResize.bind(this));
    }

    startResize(e) {
      e.preventDefault();
      this.isResizing = true;

      this.resizeStartX = e.clientX;
      this.resizeStartY = e.clientY;

      const rect = this.element.getBoundingClientRect();
      this.startWidth = rect.width;
      this.startHeight = rect.height;
      this.startRight = parseInt(getComputedStyle(this.element).right);
      this.startBottom = parseInt(getComputedStyle(this.element).bottom);

      document.body.style.userSelect = "none";
      this.element.style.transition = "none";
    }

    doResize(e) {
      if (!this.isResizing) return;

      const deltaX = this.resizeStartX - e.clientX;
      const deltaY = this.resizeStartY - e.clientY;

      let newWidth = this.startWidth + deltaX;
      let newHeight = this.startHeight + deltaY;

      newWidth = Math.max(300, newWidth);
      newHeight = Math.max(400, newHeight);

      const maxWidth = window.innerWidth * 0.8;
      const maxHeight = window.innerHeight * 0.8;
      newWidth = Math.min(maxWidth, newWidth);
      newHeight = Math.min(maxHeight, newHeight);

      this.element.style.width = newWidth + "px";
      this.element.style.height = newHeight + "px";
    }

    stopResize() {
      if (!this.isResizing) return;

      this.isResizing = false;
      document.body.style.userSelect = "";
      this.element.style.transition = "";

      this.saveDimensions();
    }

    loadDimensions() {
      try {
        const sizeKey = `dify_widget_size_${this.currentUser}`;
        const saved = localStorage.getItem(sizeKey);
        if (saved) {
          const dimensions = JSON.parse(saved);
          this.element.style.width = dimensions.width + "px";
          this.element.style.height = dimensions.height + "px";
        }
      } catch (e) {
        console.warn("Failed to load widget dimensions:", e);
      }
    }

    saveDimensions() {
      try {
        const sizeKey = `dify_widget_size_${this.currentUser}`;
        const dimensions = {
          width: parseInt(this.element.style.width),
          height: parseInt(this.element.style.height),
        };
        localStorage.setItem(sizeKey, JSON.stringify(dimensions));
      } catch (e) {
        console.warn("Failed to save widget dimensions:", e);
      }
    }
  }

  /**
   * Drupal behavior for Dify Widget Vanilla.
   */
  Drupal.behaviors.difyWidgetVanilla = {
    attach: function (context, settings) {
      const widgets = context.querySelectorAll(
        ".dify-widget-vanilla:not(.dify-widget-processed)",
      );

      widgets.forEach((widget) => {
        const widgetId = widget.dataset.widgetId;
        const widgetSettings =
          settings.difyWidgetVanilla && settings.difyWidgetVanilla[widgetId];

        if (widgetSettings) {
          widget.classList.add("dify-widget-processed");
          new DifyWidget(widget, widgetSettings);
        }
      });
    },
  };
})(Drupal);
